# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.
import math

import common
import expandconfig
from cbb.frame.base import baseUtil
from cbb.frame.base import config
from cbb.frame.cli import cliUtil

# 2U 25 SAS框盘位数，保留一位小数，便于计算结果向上取整
SAS_2U_25 = 25.0
# 单个引擎框上最大级联环路数量:两对SAS大卡，每对12个环路
MAX_EXP_LOOP_NUM = 24
# 接口卡ID对应的环路端口对
INTF_LOOP_PORTS_MAP = {
    "CTE0.R4": ["(CTE0.R4.P{0}, CTE0.L4.P{0})".format(i) for i in range(12)],
    "CTE0.R5": ["(CTE0.R5.P{0}, CTE0.L5.P{0})".format(i) for i in range(12)],
    "CTE1.R4": ["(CTE1.R4.P{0}, CTE1.L4.P{0})".format(i) for i in range(12)],
    "CTE1.R5": ["(CTE1.R5.P{0}, CTE1.L5.P{0})".format(i) for i in range(12)],
    "CTE2.R4": ["(CTE2.R4.P{0}, CTE2.L4.P{0})".format(i) for i in range(12)],
    "CTE2.R5": ["(CTE2.R5.P{0}, CTE2.L5.P{0})".format(i) for i in range(12)],
    "CTE3.R4": ["(CTE3.R4.P{0}, CTE3.L4.P{0})".format(i) for i in range(12)],
    "CTE3.R5": ["(CTE3.R5.P{0}, CTE3.L5.P{0})".format(i) for i in range(12)],
}

# noinspection PyUnresolvedReferences
LANG = common.getLang(py_java_env)
# noinspection PyUnresolvedReferences
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    """扩容级联规格检查
        1、获取系统已有的硬盘框号及数量
        2、通过输入的扩容列表中的硬盘数量+系统成员盘数，计算扩后硬盘框数量（按每框25盘位
            计算，结果向上取整）
        3、查询系统当前配置最大级联深度，计算最大硬盘框规格数（最大2U 25 SAS框数量，按每
            框最大两对SAS大卡的环路计算）。
        4、根据硬盘框号，获取每框所属环路，判断每个环路是否满配。
    :param cli:
    :return:
    """
    all_cli_ret = ""
    try:
        # 获取扩容配置列表
        # noinspection PyUnresolvedReferences
        exp_disk_list = expandconfig.ExpandConfig(
            py_java_env).getAllExpDiskList()
        # noinspection PyUnresolvedReferences
        product_model = str(py_java_env.get("devInfo").getDeviceType())
        # 计算扩容的硬盘总数
        exp_disk_num = 0
        for exp_disk_info_obj in exp_disk_list:
            exp_disk_num += exp_disk_info_obj.getDiskNum()

        # 获取系统中的成员盘
        result_flag, cli_ret, err_msg, member_disk_list = \
            cliUtil.get_member_disk_num(cli, LANG, LOGGER)
        all_cli_ret += cli_ret
        if not result_flag:
            return cliUtil.RESULT_NOCHECK, all_cli_ret, err_msg

        # 通过硬盘数计算扩后硬盘框总数，结果向上取整
        # （扩容的硬盘总数 + 系统成员盘数）/ 2U_25_SAS框盘位数
        all_disk_enc_num = int(
            math.ceil((exp_disk_num + len(member_disk_list)) / SAS_2U_25))
        LOGGER.logInfo(
            "expand disk number: {0}, all disk enclosure number: {1}".format(
                exp_disk_num, all_disk_enc_num))

        # 查询已有的硬盘框，引擎框
        result_flag, cli_ret, err_msg, origin_disk_enc, ctrl_enc_list = \
            cliUtil.query_disk_enc_info(cli, LANG, LOGGER)
        all_cli_ret += cli_ret
        if not result_flag:
            return cliUtil.RESULT_NOCHECK, all_cli_ret, err_msg

        # 查询系统配置的最大级联深度
        max_exp_depth = cliUtil.get_enclo_max_depth(cli, LANG)
        if not max_exp_depth:
            err_msg = baseUtil.getPyResource(LANG, "get.max.exp.depth.fail")
            return cliUtil.RESULT_NOCHECK, all_cli_ret, err_msg

        # 计算当前系统最大硬盘框规格
        max_disk_enc_num = MAX_EXP_LOOP_NUM * max_exp_depth * len(ctrl_enc_list)
        # 扩容后硬盘框总数，超过最大规格，检查不通过
        if all_disk_enc_num > max_disk_enc_num:
            err_msg = baseUtil.getPyResource(LANG, "disk.enclosure.num.max")
            return False, all_cli_ret, err_msg

        # 获取接口卡信息
        result_flag, cli_ret, err_msg, intf_infos = cliUtil.get_intf_info(
            cli, LANG, LOGGER)
        all_cli_ret += cli_ret
        if not result_flag:
            return cliUtil.RESULT_NOCHECK, all_cli_ret, err_msg

        # 字典形式记录每个环路上已有的硬盘框号
        loop_enc_dict = get_all_loop_ports_dict(intf_infos)
        for disk_enc_id in origin_disk_enc:
            loop_ports = get_loop_ports(disk_enc_id)
            if loop_ports not in loop_enc_dict:
                loop_enc_dict[loop_ports] = []
            loop_enc_dict[loop_ports].append(disk_enc_id)
        LOGGER.logInfo("loop_enc_dict is: %s" % loop_enc_dict)

        # 已满配的环路列表
        full_loop_ports_list = []
        # 未满配的环路列表
        not_full_loop_ports_list = []
        for loop_ports, enc_list in loop_enc_dict.items():
            if len(enc_list) < max_exp_depth:
                not_full_loop_ports_list.append(loop_ports)
            else:
                full_loop_ports_list.append(loop_ports)
        # 排序
        full_loop_ports_list.sort()
        not_full_loop_ports_list.sort()

        # F系列不支持4U框，只需提示2U框个数
        if product_model in config.V5_HIGH_END_NOT_SUP_4U_ENC:
            err_msg = baseUtil.getPyResource(LANG, "check.exp.disk.num.no.4u",
                                             (str(max_exp_depth),
                                              str(max_exp_depth),
                                              ", ".join(
                                                  not_full_loop_ports_list),
                                              ", ".join(full_loop_ports_list)))
        else:
            err_msg = baseUtil.getPyResource(LANG, "check.exp.disk.num.has.4u",
                                             (str(max_exp_depth),
                                              str(max_exp_depth),
                                              str(max_exp_depth / 2),
                                              ", ".join(
                                                  not_full_loop_ports_list),
                                              ", ".join(full_loop_ports_list)))
        # 检查结果为建议优化
        return cliUtil.RESULT_WARNING, all_cli_ret, err_msg

    except Exception as ex:
        LOGGER.logException(ex)
        err_msg = baseUtil.getPyResource(LANG, "query.result.abnormal")
        return cliUtil.RESULT_NOCHECK, "", err_msg


def get_loop_ports(disk_enc_id):
    """通过硬盘框号(DAEXXX)判断所属环路端口

    :param disk_enc_id:
    :return:
    """
    # 硬盘框号长度为6位
    if len(disk_enc_id) == 6:
        eng_id = disk_enc_id[3]  # 引擎号
        loop_num = int(disk_enc_id[4], 16)  # 对应SAS卡端口编号，16进制转10进制
        last_num = int(disk_enc_id[5])  # 硬盘框末位编号
        # 末位编号大于等于4，为4号槽位SAS接口卡上环路
        if last_num >= 4:
            loop_ports = "(CTE{0}.R4.P{1}, CTE{0}.L4.P{1})".format(eng_id,
                                                                   loop_num)
        # 小于4，为5号槽位SAS接口卡上环路
        else:
            loop_ports = "(CTE{0}.R5.P{1}, CTE{0}.L5.P{1})".format(eng_id,
                                                                   loop_num)
    else:
        loop_ports = ""
    return loop_ports


def get_all_loop_ports_dict(intf_infos):
    """根据已有的接口卡，生成环路端口字典，key:(port0, port1), value: []

    :param intf_infos: 接口卡信息列表
    :return:
    """
    # 已有SAS接口卡的环路列表
    all_loop_ports_list = []
    for intf_info in intf_infos:
        intf_id = intf_info.get("ID", "")
        if intf_id in INTF_LOOP_PORTS_MAP:
            all_loop_ports_list.extend(INTF_LOOP_PORTS_MAP.get(intf_id, []))
    if not all_loop_ports_list:
        return {}
    list_to_change = [(ports, []) for ports in all_loop_ports_list]
    loop_enc_dict = dict(list_to_change)
    return loop_enc_dict
