# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
# noinspection PyUnresolvedReferences
from java.lang import Exception as JException

import common
import expandconfig
from cbb.frame.base import baseUtil
from cbb.frame.cli import cliUtil

# noinspection PyUnresolvedReferences
LANG = common.getLang(py_java_env)
# noinspection PyUnresolvedReferences
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    """ 检查是否曾创建文件系统

    :return:
    """
    check_item = CheckFileSystem(cli, LANG, LOGGER, py_java_env)
    check_item.start()
    return check_item.get_result()


class CheckFileSystem:
    """
    检查是否曾创建文件系统
        背景：Dorado 6.1.RC2及其之前的版本不支持文件系统跨控迁移，故需要对创建过文件
            系统的存储池引擎内扩容和引擎间控容进行拦截
        1、步骤3获取的软件版本若为6.1.RC2，则进行步骤4检查，否则通过。
        2、步骤4获取到硬盘域ID，将所有硬盘域ID带入show pool_manager logic_pool
            disk_pool_id=中查询logic pool。
        3、引擎间扩容场景下，步骤5中如果硬盘域归属引擎与新扩盘对应的引擎符合，则检查
            通过，否则继续检查。引擎内扩容场景下，继续检查。
        4、步骤6中如果存在ORIGINAL_POOL_FS_8K_M_0，ORIGINAL_POOL_FS_16K_D_1，
            DCL_META_POOL_FS_M_2，ORIGINAL_POOL_FS_8K_D_3，
            ORIGINAL_POOL_FS_32K_D_4中的任何一个logic pool则不通过，否则通过。
    """

    # 文件系统列表
    LIMIT_LOGIC_POOL = {
        "ORIGINAL_POOL_FS_8K_M_0",
        "ORIGINAL_POOL_FS_16K_D_1",
        "DCL_META_POOL_FS_M_2",
        "ORIGINAL_POOL_FS_8K_D_3",
        "ORIGINAL_POOL_FS_32K_D_4"
    }

    def __init__(self, cli, lang, logger, py_java_env):
        self.cli = cli
        self.env = py_java_env
        self.lang = lang
        self.logger = logger
        self.product_model = str(self.env.get("devInfo").getDeviceType())
        self.dd_info = dict()
        self.result_flag = True
        self.result_ret_list = list()
        self.result_err_msg = ""
        self.err_type = ""

    def start(self):
        """ 开始执行检查

        :return:
        """
        try:
            # 低端设备没有文件系统，不涉及
            if baseUtil.isV5V6LowEnd(self.product_model):
                self.result_flag = cliUtil.RESULT_NOSUPPORT
                return

            # 进入developer视图
            cliUtil.enterDeveloperMode(self.cli, self.lang)
            # 获取硬盘域信息
            self.get_dd_info()

            if self.env.get("expMode") == "EXTEND_CTRL"\
                    and baseUtil.isDoradoV6HighEnd(self.product_model)\
                    and (int(self.env.get("expCtrlOri")) == 2 or
                         self.env.get("requireInnerMetro")):
                # 在高端扩容控制器场景下，如果有引擎内扩容（原引擎数为双控）
                # 或者需要内双活的情况下，需1要对所有硬盘域进行检查
                self.logger.logInfo("check disk domain")
                dd_to_check = self.get_disk_domain()
                self.err_type = "DISK_DOMAIN"
            else:
                # 对于扩硬盘、硬盘框、存储池及低端设备扩控、非内双活引擎间扩控场景
                dd_to_check = self.check_engine_changed_dd()
                self.err_type = "STORAGE_POOL"
                # 没有在新引擎上扩容原有硬盘域，则通过
                if not dd_to_check:
                    return

            # 检查创建了文件系统的待扩容硬盘域，获取对应的存储池ID
            pool_has_file_system = self.check_logic_pool(dd_to_check)
            if pool_has_file_system:
                self.generate_err_msg(pool_has_file_system, self.err_type)
            return

        except (JException, Exception) as e:
            self.logger.logException(e)
            self.result_flag, self.result_err_msg = \
                cliUtil.RESULT_NOCHECK, \
                common.getMsg(self.lang, "query.result.abnormal")
            return
        finally:
            # 退出到cli模式
            cliUtil.enterCliModeFromSomeModel(self.cli, self.lang)

    def get_dd_info(self):
        """ 获取硬盘域信息

        :return:
        """
        cmd = "show storage_pool general"
        ret = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.result_ret_list.append(ret[1])
        if ret[0] is not True:
            raise Exception("Failed to get storage pool info.")

        pool_info_list = cliUtil.getHorizontalCliRet(ret[1])
        for pool_info in pool_info_list:
            pool_id = pool_info.get("ID")
            dd_id = pool_info.get("Disk Domain ID")
            info = self.dd_info.setdefault(dd_id, dict())
            info["pool_id"] = pool_id
            info["engine"] = self.get_dd_engine(dd_id)
            info["logic_pool"] = self.get_dd_logic_pool(dd_id)

    def get_dd_engine(self, dd_id):
        """ 获取硬盘域对应的引擎号集合

        :param dd_id:
        :return:
        """
        cmd = "show disk_domain general disk_domain_id={}".format(dd_id)
        ret = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.result_ret_list.append(ret[1])
        if ret[0] is not True:
            raise Exception("Failed to query domain[{}]".format(dd_id))

        info = cliUtil.getVerticalCliRet(ret[1])
        engines = info[0].get("Controller enclosure", "").split(",")
        return set(map(lambda x: x[3], engines))

    def get_dd_logic_pool(self, dd_id):
        """ 获取硬盘域的logic pool列表

        :param dd_id:
        :return:
        """
        cmd = "show pool_manager logic_pool disk_pool_id={}".format(dd_id)
        ret = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.result_ret_list.append(ret[1])
        if ret[0] is not True:
            raise Exception("Failed to query logic pool of disk "
                            "pool {}.".format(dd_id))

        logic_pool_info = cliUtil.getHorizontalCliRet(ret[1])
        return set([info.get("Name", "") for info in logic_pool_info])

    def check_engine_changed_dd(self):
        """ 检查扩容配置中归属引擎发生变化的硬盘域ID

        :return:
        """
        inconsistent_dd = list()
        exp_disk_dict = \
            expandconfig.ExpandConfig(self.env).getExpDiskInfo()
        for dd_id, exp_disk_list in exp_disk_dict.items():
            dd_pool_map = common.getpoolAndDiskDomainMap(py_java_env)
            disk_domain_id = dd_pool_map.get(dd_id)
            for exp_disk in exp_disk_list:
                engine_id = exp_disk.getDiskEngineId()
                if engine_id not in self.dd_info.get(disk_domain_id, {}).get(
                        "engine", set()):
                    inconsistent_dd.append(disk_domain_id)
        return inconsistent_dd

    def check_logic_pool(self, check_dd):
        """ 检查存在文件系统的存储池

        :param check_dd: 待检查硬盘域
        :return:
        """
        if self.err_type == "STORAGE_POOL":
            return [self.dd_info.get(dd).get("pool_id") for dd in check_dd
                    if self._has_fuzz_intersection(
                    self.dd_info.get(dd).get("logic_pool"),
                    self.LIMIT_LOGIC_POOL)]
        else:
            return [dd_id for dd_id in check_dd
                    if self._has_fuzz_intersection
                    (self.get_dd_logic_pool(dd_id),
                     self.LIMIT_LOGIC_POOL)]

    def generate_err_msg(self, dd_has_file_system, err_type):
        self.result_flag = False
        if err_type == "STORAGE_POOL":
            self.result_err_msg = common.getMsg(self.lang,
                                            "storage.pool.has.file.system",
                                            ", ".join(dd_has_file_system))
        else:
            self.result_err_msg = common.getMsg(self.lang,
                                                "disk.domain.has.file.system",
                                                ", ".join(dd_has_file_system))

    def get_result(self):
        """ 获取检查结果

        :return:
        """
        return (self.result_flag, "\n".join(self.result_ret_list),
                self.result_err_msg)

    def get_disk_domain(self):
        """ 获取所有硬盘域

        :return:
        """
        cmd = "show disk_domain general"
        ret = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.result_ret_list.append(ret[1])
        if ret[0] is not True:
            raise Exception("Failed to get disk domain info.")

        dd_list = list()
        dd_info_list = cliUtil.getHorizontalCliRet(ret[1])
        for dd_info in dd_info_list:
            dd_list.append(dd_info.get("ID"))

        return dd_list

    @staticmethod
    def _has_fuzz_intersection(list1, list2):
        for str1 in list1:
            for str2 in list2:
                if str1 == str2 \
                        or (str1.startswith(str2) and "_del" in str1.lower()):
                    return True
        return False
