# -*- coding: UTF-8 -*-
import cliUtil
import common

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
HIGH_DISK_ENCLOSURE = "4U 75 Slot 3.5 SAS Disks Enclosure"
NEW_HIGH_DISK_ENCLOSURE = "12G 4U 75 Slot 3.5 SAS Disks Enclosure"
NEED_TO_CHECK_MODEL = ['2100 V3', '2200 V3', '2200 V3 Enhanced', '2600 V3', '2600 V3 Enhanced', '5300 V3', '5500 V3',
                       '5600 V3', '5800 V3', '6800 V3', '18500 V3', '18800 V3', '5300 V5', '5300 V5 Enhanced',
                       '5500 V5', '5600 V5', '5800 V5', '6800 V5', '5500 V5 Elite', '18500 V5', '18800 V5']


def execute(cli):
    """
    硬盘域下硬盘所在硬盘框类型检查
    步骤1 以admin用户登录设备。
    步骤2 执行命令show system general 获取产品型号。
    步骤3 执行show enclosure命令，获取硬盘框类型。
    步骤4 执行show disk_domain general命令，获取硬盘域ID。
    步骤5 执行show disk in_domain disk_domain_id=disk_domain_id命令，获取硬盘域下所有硬盘信息。
    检查标准：
    1、如果不是以下产品型号，检查通过。
    2100 V3、2200 V3、2200 V3 Enhanced、2600 V3、2600 V3 Enhanced、5300 V3、5500 V3、5600 V3、5800 V3、6800 V3、18500 V3、
    18800 V3、5300 V5、5300 V5 Enhanced、5500 V5、5600 V5、5800 V5、6800 V5、5500 V5 Elite、18500 V5、18800 V5
    2、若同一硬盘域下同时存在普通框和高密框下的硬盘，则检查结果为不通过。
    注：类型为“4U 75 Slot 3.5 SAS Disks Enclosure”或“12G 4U 75 Slot 3.5 SAS Disks Enclosure”的硬盘框为高密框；其它类型为普通框。
    :param cli:
    :return:
    """
    cli_ret_all = ""
    err_msg_all = ''
    try:
        # 判断产品型号是否在需要检查的型号列表中
        flag, product_model, cli_ret, err_msg = cliUtil.getProductModelWithCliRet(cli, LANG)
        if flag != True:
            LOGGER.logSysAbnormal()
            return flag, cli_ret, err_msg
        if product_model not in NEED_TO_CHECK_MODEL:
            return True, cli_ret, err_msg
        cli_ret_all += cli_ret
        err_msg_all += err_msg
        # 1获取硬盘框类型
        check_ret = get_enclosure_type(cli)
        cli_ret_all += check_ret[1]
        err_msg_all += check_ret[3]
        if check_ret[0] != True:
            LOGGER.logSysAbnormal()
            return check_ret[0], cli_ret_all, err_msg_all
        enclosure_id_type_dict = check_ret[2]

        # 2判断硬盘域下硬盘的硬盘框类型是否同时存在高密框和普通硬盘框
        flag, cli_ret, err_msg = check_disk_enclosure_type_with_diskdomain(cli, enclosure_id_type_dict)
        cli_ret_all += cli_ret
        err_msg_all += err_msg
        return flag, cli_ret_all, err_msg_all

    except Exception, exception:
        LOGGER.logException(exception)
        return cliUtil.RESULT_NOCHECK, cli_ret_all, common.getMsg(LANG, "query.result.abnormal")


def get_enclosure_type(cli):
    """
    @summary: 硬盘框类型
    @param cli: cli对象
    @return: (falg, ret, err_msg)
        flag:
            True: 获取成功
            False: 获取失败
        ret:
            flag为True时，硬盘框ID，类型（Type）字典
            flag为False时，cli回显
        err_msg: 错误消息
    """
    err_msg = ""
    enclosure_id_type_dict = {}
    cmd = "show enclosure"
    LOGGER.logExecCmd(cmd)
    check_ret = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    cli_ret = check_ret[1]

    if check_ret[0] != True:
        LOGGER.logSysAbnormal()
        return check_ret[0], check_ret[1], enclosure_id_type_dict, check_ret[2]

    cli_ret_lines_list = cliUtil.getHorizontalCliRet(cli_ret)
    if len(cli_ret_lines_list) == 0:
        err_msg = common.getMsg(LANG, "cannot.get.enclosure.info")
        LOGGER.logNoPass("Cannot get information about enclosure")
        return False, cli_ret, enclosure_id_type_dict, err_msg

    for line in cli_ret_lines_list:
        enclosure_id = line.get("ID", "")
        type = line.get("Type", "")
        enclosure_id_type_dict.setdefault(enclosure_id, type)
    LOGGER.logInfo("enclosure_id_type_dict = %s" % (str(enclosure_id_type_dict)))

    return True, cli_ret, enclosure_id_type_dict, err_msg


def check_disk_enclosure_type_with_diskdomain(cli, enclosure_id_type_dict):
    """
    @summary: 检查硬盘域下的硬盘所属硬盘框是否同时存在高密框和普通硬盘框
    @param cli: cli对象
    @param enclosure_id_type_dict: 硬盘框类型
    @return: (falg, ret, err_msg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，不同时存在高密框和普通硬盘框
            flag为False时，同时存在高密框和普通硬盘框
        err_msg: 错误消息
    """
    flag = True
    err_msg = ""
    cmd = "show disk_domain general"
    check_ret = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    cli_ret = check_ret[1]
    if check_ret[0] != True:
        LOGGER.logSysAbnormal()
        return check_ret

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return True, cli_ret, err_msg

    cli_ret_lines_list = cliUtil.getHorizontalCliRet(cli_ret)
    if len(cli_ret_lines_list) == 0:
        err_msg = common.getMsg(LANG, "cannot.get.disk.domain.info")
        LOGGER.logNoPass("Cannot get information about disk domain")
        return cliUtil.RESULT_NOCHECK, cli_ret, err_msg  # 修改备注：解析回显失败导致列表长度为0

    for line in cli_ret_lines_list:
        id = line.get("ID", "")
        cmd = "show disk in_domain disk_domain_id=%s" % id
        check_ret = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
        cli_ret += check_ret[1]
        if check_ret[0] != True:
            LOGGER.logSysAbnormal()
            return check_ret[0], cli_ret, check_ret[2]

        cli_ret_lines_dict_list = cliUtil.getHorizontalCliRet(check_ret[1])
        if len(cli_ret_lines_dict_list) == 0:
            err_msg += common.getMsg(LANG, "cannot.get.disk.info.in.disk.domain")
            LOGGER.logNoPass("Failed to obtain information about disks in the disk domain.")
            return cliUtil.RESULT_NOCHECK, cli_ret, err_msg

        disk_type_list = []

        for line in cli_ret_lines_dict_list:
            disk_id = line.get("ID").split(".")[0]
            disk_type = enclosure_id_type_dict.get(disk_id, "")
            if disk_type:
                disk_type_list.append(disk_type)

        # 去除重复项
        disk_type_set = set(disk_type_list)

        # 硬盘域下硬盘只存在两种类型高密框场景（有且只有两种高密框场景）
        if HIGH_DISK_ENCLOSURE in disk_type_list and NEW_HIGH_DISK_ENCLOSURE in disk_type_list and len(
                disk_type_set) == 2:
            continue

        # 存在一种高密框和普通硬盘框场景
        elif (HIGH_DISK_ENCLOSURE in disk_type_list or NEW_HIGH_DISK_ENCLOSURE in disk_type_list) and len(
                disk_type_set) > 1:
            flag = False
            err_msg += common.getMsg(LANG, "disk.domain.exist.high.enclosure", id)

    return flag, cli_ret, err_msg
