#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

"""
@version: Toolkit V200R006C00
@time: 2020/03/21
@file: check_item_hardware_port_bit_error.py
@function:
@modify:
"""

import cliUtil
import common

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)

CHECK_PORT_LIST = ("RDMA", "SAS", "ETH", "FC", "FCoE", "PCIE")


def execute(cli):
    """
    端口误码率检查
    若RDMA端口、SAS端口、ETH端口、FC端口、FCoE端口和PCIE端口误码信息中，
    除“Frame Length Errors”类型外（即不统计此类型），存在持续增加的误码，
    则检查结果为不通过；否则检查结果为通过。
    :param cli
    :return:
    """
    cli_ret = ""
    try:
        cmd = "show port bit_error"
        LOGGER.logExecCmd(cmd)
        check_ports_error_ret = check_ports_bit_error(cli, cmd)
        cli_ret = check_ports_error_ret[1]
        if check_ports_error_ret[0] is not True:
            LOGGER.logNoPass(
                "Ethernet ports, fibre channel ports, "
                "or FCoE ports has continuous bit errors:\n" + cli_ret
            )
        else:
            LOGGER.logPass()
        return check_ports_error_ret

    except Exception as e:
        LOGGER.logException(e)
        return (
            cliUtil.RESULT_NOCHECK,
            cli_ret,
            common.getMsg(LANG, "query.result.abnormal"),
        )


def check_ports_bit_error(cli, cmd):
    """
    @summary: 检查所有端口在特定时间段内是否有持续增加的误码
    """
    flag = True
    cli_ret_all = []
    common.refreshProcess(PY_JAVA_ENV, 1, LOGGER)
    (
        first_flag,
        first_cli_ret,
        err_msg,
        first_result_dict,
    ) = get_ports_bit_error_result_dict(cli, cmd)
    cli_ret_all.append(first_cli_ret)
    if first_flag is not True:
        return (first_flag, first_cli_ret, err_msg)
    common.refreshProcess(PY_JAVA_ENV, 2, LOGGER)
    retried_times = int(
        common.PORTS_BIT_ERROR_SPEC / common.PORTS_BIT_ERROR_INTERVAL
    )
    current_process = 2
    singleProcess = 98.0 / (retried_times * common.PORTS_BIT_ERROR_INTERVAL)
    for i in range(0, retried_times):
        current_process = common.refreshProcessWithSleep(
            current_process,
            common.PORTS_BIT_ERROR_INTERVAL,
            singleProcess,
            PY_JAVA_ENV,
            LOGGER,
        )
        (
            next_flag,
            next_cli_ret,
            err_msg,
            next_result_dict,
        ) = get_ports_bit_error_result_dict(cli, cmd)
        cli_ret_all.append(next_cli_ret)
        if next_flag is not True:
            return (next_flag, "\n".join(cli_ret_all), err_msg)
        flag, err_msg = judge_ports_error(
            first_result_dict, next_result_dict, flag
        )
        if not flag:
            return (False, "\n".join(cli_ret_all), err_msg)
    return (True, "\n".join(cli_ret_all), "")


def judge_ports_error(first_result_dict, next_result_dict, flag):
    err_msg = []
    for port_name in CHECK_PORT_LIST:
        first_bit_error_list = first_result_dict.get(port_name, list())
        next_bit_error_list = next_result_dict.get(port_name, list())
        if len(first_bit_error_list) != len(next_bit_error_list):
            continue
        cmp_result = cmp_dict_list(first_bit_error_list, next_bit_error_list)
        if cmp_result == -1:
            flag = False
            err_msg.append(
                common.getMsg(LANG, "port.exists.error.bits", port_name)
            )
    return (flag, "".join(err_msg))


def get_ports_bit_error_result_dict(cli, cmd):
    """
    @summary: 获取指定端口列表对应的误码列表
    @param cli: cli对象
    @param cmd: cli命令
    @param ports: 端口列表
    @param lang: 语言对象
    @return:
        flag:
            True: 获取成功
            False: 获取失败
        cli_ret: cli回显
        errMsg: 错误时的消息
        result_dict: 以字典的形式返回对应端口的误码列表。
        键对应端口，值对应该端口的误码列表
    """
    result_dict = {}
    flag, cli_ret, msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if flag is not True:
        return (flag, cli_ret, msg, result_dict)

    for port_name in CHECK_PORT_LIST:
        (flag, cli_ret_lines_list, err_msg) = get_port_bit_error_result_list(
            cli_ret, port_name
        )
        if flag is not True:
            return (
                flag,
                cli_ret,
                err_msg,
                result_dict,
            )
        result_dict.setdefault(port_name, cli_ret_lines_list)
    return (True, cli_ret, "", result_dict)


def get_port_bit_error_result_list(cli_ret, port_name):
    """
    @summary: 获取每个端口的误码列表
    @param cli_ret: cli回显
    @param port_name: 端口名
    @param lang: 语言对象
    @return:
        flag:
            True: 获取成功
            False: 获取失败
        resultList: 误码列表
        errMsg: 错误时的消息
    """
    port_info_ret = cliUtil.getSplitedCliRet(
        cli_ret, get_split_port_words(port_name)
    )
    if len(port_info_ret) == 0:
        return (True, [], "")

    cli_ret_lines_list = cliUtil.getHorizontalCliRet(port_info_ret)
    if len(cli_ret_lines_list) == 0:
        errMsg = common.getMsg(LANG, "cannot.get.port.info", port_name)
        return (False, cli_ret_lines_list, errMsg)
    for i in cli_ret_lines_list:
        if "Frame Length Errors" in i:
            i.pop("Frame Length Errors")
    return (True, cli_ret_lines_list, "")


def cmp_dict_list(first_list, second_list):
    """
    @summary: 判断误码数是否有增加
    """
    for i in range(0, len(first_list)):
        first_dict = first_list[i]
        second_dict = second_list[i]
        ret = 0
        for key in first_dict.keys():
            first_value = str(first_dict.get(key))
            second_value = str(second_dict.get(key))
            if first_value.isdigit() and second_value.isdigit():
                ret = cmp(long(first_value), long(second_value))
            if ret != 0:
                return ret
    return 0


def get_split_port_words(port_name):
    """
    @summary: 根据端口名获取对应的分割字
    @param: port_name: 端口名
    @return: 分割字
    """
    if port_name == "PCIe":
        port_name = "PCIE"
    return port_name + " port:"
