# -*- coding: UTF-8 -*-
import cliUtil
import common
from common import UnCheckException
from config import special_hyper_rule_list, general_hyper_rule_list
import common_utils

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
import traceback

v5R7C7 = 'V500R007C73'
v3R6C6 = 'V300R006C61'

SPC_INDEX = 11

systemGeneral = "show system general"
controllerGeneral = "show controller general"
hyperMetroPairGeneral = "show hyper_metro_pair general"
localDev = common.getCurDeviceInfo(py_java_env)
cliRet = ''
lineBreak = '\n'
checkResults = []


def execute(cli):
    '''
    @summary: 双活两端版本及控制器一致性检查
    步骤1 以admin用户登录设备。
    步骤2 执行命令show hyper_metro_pair general，查看当前阵列配置的所有的双活pair，查询出“ID”字段值。
    步骤3 执行命令show system general，查看当前阵列配置的版本信息及产品型号。
    步骤4 执行命令show controller general，查看当前阵列配置的控制器数量（该命令查询出的详细信息中“Location”的数量为当前阵列控制器数量）。
    步骤5 以admin用户登录远端设备。
    步骤6 执行步骤3-4，获取远端的阵列的详细版本信息，产品型号及控制器数量。

    1 如果步骤2查询结果没有双活pair，则检查通过结束。
    2 比较步骤3查询到的本端详细版本信息和产品型号与步骤6查询到的远端详细版本信息和产品型号。如果两端信息一致（V300R006C50、V500R007C30及之后的版本两端的产品型号需支持互通配置双活，6800/18500/18800 V3/V5/F产品型号间支持互通配置双；5500/5600/5800 V3/V5/F产品型号间支持互通；2600/5300 V3/V5/F产品型号间支持互通; 其他版本如果产品型号两端分别为18500 V3与6800 V3认为一致）则通过，否则不通过。（比较字段如下：“Product Version”，“Patch Version”，“Product Model”）
    3 比较步骤4查询到的本端控制器数量与步骤6查询到的远端控制器数量，如果两端信息一致则通过，否则建议优化。（比较字段“Running Status”为“Online”的数量）
    4 V300R006C50及V500R007C30之后的版本需要产品提AR需求来更新检查项
    '''
    flag = True
    errMsg = ''
    global cliRet
    try:
        localDevSN = str(localDev.getDeviceSerialNumber())
        cliRet += "ON LOCAL DEVICE(SN:%s)\n" % localDevSN
        if not existshyperMetro(cli):
            LOGGER.logInfo('***[not hyper metro]***')
            return (True, cliRet, '')
        localFlag, localCliRet = executeAndcheckRetValidity(systemGeneral)
        if not localFlag:
            return (cliUtil.RESULT_NOCHECK, cliRet, common.getMsg(LANG, "cannot.get.system.info"))
        localProductTuple = getProductModelVersion(LOGGER, localDevSN, localCliRet)
        localProductSoftwareVersion = localProductTuple[1]
        if (localProductSoftwareVersion.startswith(
                'V300R006C') and localProductSoftwareVersion[
                                 :11] > v3R6C6) or (
                localProductSoftwareVersion.startswith(
                    'V500R007C') and localProductSoftwareVersion[
                                     :11] > v5R7C7):
            return (cliUtil.RESULT_NOCHECK, cliRet,
                    common.getMsg(LANG, "version.not.fit",
                                  localProductSoftwareVersion))

        flag, addedSnList, errMsg = common.checkAddedRemoteDevSn(py_java_env, LANG)
        if flag != True:
            return common_utils.get_result_bureau(py_java_env, cliRet, errMsg)

        flag, _, errMsg, hyperRemoteSnList = common.getRemoteDevSn(py_java_env, str(localDev.getDeviceSerialNumber()),
                                                                   LOGGER, LANG)
        if flag != True:
            return common_utils.get_result_bureau(py_java_env, cliRet, errMsg)
        domain_remote_dev_dict = common_utils.get_remote_device_info(py_java_env, localDevSN, LOGGER, LANG)
        no_check_msg = ""
        for remoteDevSN in hyperRemoteSnList:
            domain_id = domain_remote_dev_dict.get(remoteDevSN)
            domain_pair_list, pair_info_list = common_utils.get_hyper_metro_pair_id_list(
                localDevSN, domain_id, py_java_env, LOGGER, LANG)
            if not domain_pair_list:
                LOGGER.logInfo("domain {} do not have pair. sn:{}".format(
                    domain_id, localDevSN))
                continue
            if remoteDevSN not in addedSnList:
                no_check_msg += common.getMsg(
                    LANG, "not.add.remote.device.again", remoteDevSN)
                continue

            cliExecuteRet = common.getObjFromFile(
                py_java_env, LOGGER, remoteDevSN, systemGeneral, LANG)
            if cliExecuteRet[0] != True:
                return common_utils.get_result_bureau(py_java_env, cliRet + cliExecuteRet[1], common.getMsg(
                    LANG, "not.add.remote.device.again", remoteDevSN))
            remoteProductTuple = getProductModelVersion(LOGGER, remoteDevSN,
                                                        cliExecuteRet[1],
                                                        False)
            remoteProductSoftwareVersion = remoteProductTuple[1]
            if (remoteProductSoftwareVersion.startswith(
                    'V300R006C') and remoteProductSoftwareVersion[
                                     :11] > v3R6C6) or (
                    remoteProductSoftwareVersion.startswith(
                        'V500R007C') and remoteProductSoftwareVersion[
                                         :11] > v5R7C7):
                no_check_msg += common.getMsg(LANG, "version.not.fit",
                                              remoteProductSoftwareVersion)

        if no_check_msg:
            return common_utils.get_result_bureau(py_java_env, cliRet, no_check_msg)

        localFlag, localCliRet = executeAndcheckRetValidity(controllerGeneral)
        if not localFlag:
            return (cliUtil.RESULT_NOCHECK, cliRet, common.getMsg(LANG, "cannot.get.controller.info"))
        localControllerNum = getControllerNum(localCliRet)
        promptMsg = {}
        LOGGER.logInfo('addedSnList = %s' % addedSnList)
        hyperRemoteSnList = list(set(hyperRemoteSnList))
        LOGGER.logInfo('hyperRemoteSnList = %s' % hyperRemoteSnList)
        for remoteDevSN in hyperRemoteSnList:
            cliRet += "\n\nON REMOTE DEVICE(SN:%s):\n" % remoteDevSN
            domain_id = domain_remote_dev_dict.get(remoteDevSN)
            domain_pair_list, pair_info_list = common_utils.get_hyper_metro_pair_id_list(
                localDevSN, domain_id, py_java_env, LOGGER, LANG)
            if not domain_pair_list:
                LOGGER.logInfo("domain {} do not have pair.".format(domain_id))
                continue
            if remoteDevSN not in addedSnList:
                flag = cliUtil.RESULT_NOCHECK
                errMsg += common.getMsg(LANG, "not.add.remote.device", remoteDevSN)
                continue

            err_msg_list = checkConsistency(localProductTuple,
                                            localControllerNum,
                                            str(remoteDevSN))
            if err_msg_list:
                flag = False
                msg = ''
                if cliUtil.isChinese(LANG):
                    msg = u'，'.join(err_msg_list)
                else:
                    msg = ','.join(err_msg_list)
                promptMsg[str(remoteDevSN)] = msg
        if flag != True:
            errMsg += getErrorMsg(promptMsg)
            LOGGER.logInfo('***[not pass, cause: %s]***' % errMsg)
            if False in checkResults:
                return (False, cliRet, errMsg)
            if cliUtil.RESULT_WARNING in checkResults:
                return (cliUtil.RESULT_WARNING, cliRet, errMsg + common.getMsg(LANG, "controller.num.warning.sugg"))
            return common_utils.get_result_bureau(py_java_env, cliRet, errMsg)
        LOGGER.logPass()
        return (flag, cliRet, errMsg)
    except UnCheckException, unCheckException:
        LOGGER.logError(str(traceback.format_exc()))
        LOGGER.logInfo("UnCheckException, errMsg: %s" % unCheckException.errorMsg)
        if not unCheckException.flag:
            return cliUtil.RESULT_NOCHECK, unCheckException.cliRet, unCheckException.errorMsg

        return unCheckException.flag, unCheckException.cliRet, unCheckException.errorMsg
    except BaseException, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, cliRet, common.getMsg(LANG, "query.result.abnormal"))


def executeAndcheckRetValidity(cmd):
    '''
    @summary: execute command and check the validity of result
    '''
    cliExecuteRet = common.getObjFromFile(py_java_env, LOGGER, str(localDev.getDeviceSerialNumber()), cmd, LANG)
    global cliRet
    cliRet += cliExecuteRet[1] + lineBreak
    if cliExecuteRet[0] != True:
        return False, ''
    return True, cliExecuteRet[1]


def getErrorMsg(promptMsg):
    '''
    @summary: get error message form prompt dictionary
    '''
    errorMsg = ''
    LOGGER.logInfo("promptMsg= %s" % str(promptMsg))
    for key in promptMsg:
        errorMsg += common.getMsg(LANG, "inconsistency.prompt",
                                  (str(localDev.getDeviceSerialNumber()), str(key), promptMsg[key]))
    if not errorMsg:
        return errorMsg
    if cliUtil.isChinese(LANG):
        errorMsg = errorMsg[:-1] + u'。'
    else:
        errorMsg = errorMsg[:-1] + "."
    return errorMsg


def existshyperMetro(cli):
    '''
    @summary: check whether or not has hyper metro
    '''
    cliExecuteRet = cliUtil.excuteCmdInCliMode(cli, hyperMetroPairGeneral, True, LANG)
    global cliRet
    cliRet += cliExecuteRet[1] + lineBreak
    if cliExecuteRet[0] != True:
        throwException('localCli execute command %s failure' % hyperMetroPairGeneral)
    if cliUtil.getHorizontalCliRet(cliExecuteRet[1]):
        return True
    return False


def check_c_version(cru_ver, ge_version_list):
    """
    检查当前版本是否在规则版本列表中，判断当前版本是否包含规则中的C版本
    :param cru_ver:当前版本
    :param ge_version_list:规则列表
    :return:True:包含, False:不包含
    """
    for ge_version in ge_version_list:
        if cru_ver.startswith(ge_version):
            return True
    return False


def checkConsistency(localProductTuple, localControllerNum, remoteDevSN):
    '''
    @summary: check the consistency of master and remote
    '''
    cliExecuteRet = common.getObjFromFile(py_java_env, LOGGER, remoteDevSN, systemGeneral, LANG)
    global cliRet
    cliRet += cliExecuteRet[1] + lineBreak
    if cliExecuteRet[0] != True:
        raise UnCheckException(cliExecuteRet[2], cliRet)
    remoteProductTuple = getProductModelVersion(LOGGER, remoteDevSN, cliExecuteRet[1])
    err_type_list = check_product_consistency(localProductTuple,
                                              remoteProductTuple)
    err_ctrl_num_msg = checkControllerConsistency(localControllerNum,
                                                  remoteDevSN)
    if err_type_list:
        checkResults.append(False)

    if err_ctrl_num_msg:
        err_type_list.append(err_ctrl_num_msg)
        checkResults.append(cliUtil.RESULT_WARNING)

    return err_type_list


def check_model_can_link(local_model, remote_model, model_rule_list):
    """检查型号是否支持互通
    :param local_model:本端型号
    :param remote_model:远端型号
    :param model_rule_list:规则列表
    :return:True:支持，False:不支持
    """
    # 型号规则遍历，必定有结果
    for model_list_rule in model_rule_list:
        if local_model not in model_list_rule \
                or remote_model not in model_list_rule:
            continue

        # 型号支持互通,需break,避免值被覆盖
        return True

    return False


def check_general_version(local_info_tuple, remote_info_tuple):
    """不同版本和型号 互通规则检查
    :param local_info_tuple:本端信息
    :param remote_info_tuple:远端信息
    :return: flag:是否互通，list:错误消息
    """
    err_msg_list = []
    local_model, local_version, local_patch, local_pro_ver = local_info_tuple
    remote_model, remote_version, remote_patch, remote_pro_ver = \
        remote_info_tuple

    # 互通版本时，SPC部分版本要一致
    local_version_spc = local_version[11:]
    remote_version_spc = remote_version[11:]

    # 1. check通用互通，大版本是否支持互通
    version_can_link = False

    # 型号是否支持互通
    model_can_link = False

    for rule in general_hyper_rule_list:
        ge_version_list = rule.get("c_version")
        ge_model_rule_list = rule.get("model_rule_list")

        # 如果版本满足规则继续检查型号是否满足规则，否则继续遍历其他规则
        if not check_c_version(local_pro_ver, ge_version_list) \
                or not check_c_version(remote_pro_ver, ge_version_list):
            # 版本不互通场景，继续遍历规则，最后都没命中的话，
            # 则所有信息必须一致
            continue

        # 大版本支持互通
        version_can_link = True

        # 型号规则遍历，必定有结果
        model_can_link = check_model_can_link(local_model, remote_model,
                                              ge_model_rule_list)
        LOGGER.logInfo("model_can_link:{}".format(model_can_link))
        # 如果型号规则遍历完，如果型号不互通，则返回不互通。
        if not model_can_link:
            version_can_link = False
            continue

        # 当两端软件版本相同时，检查补丁版本，如果补丁版本一致则通过，否则不通过
        if local_version == remote_version and local_pro_ver == remote_pro_ver \
                and local_patch != remote_patch:
            err_key = "common.prduct.version.or.patch"
            err_msg_list.append(common.getMsg(LANG, err_key))

        err_msg = check_same_c_version(
            local_version, remote_version, local_patch, remote_patch)
        if err_msg:
            err_msg_list.append(err_msg)

        # 版本互通时-全部信息比较完成。
        return True, list(set(err_msg_list))

    # 返回版本是否互通
    return version_can_link, list(set(err_msg_list))


def check_same_c_version(
        local_version, remote_version, local_patch, remote_patch):
    """
    是否相同C版本，相同时补丁和版本要一致。
    :param local_version:
    :param remote_version:
    :param local_patch:
    :param remote_patch:
    :return: 错误消息
    """
    LOGGER.logInfo("version_a:{}, version_b:{},patch_a:{},patch_b:{}".format(
        local_version, remote_version, local_patch, remote_patch))
    if local_version[:SPC_INDEX] == remote_version[:SPC_INDEX]:
        if local_version == remote_version and local_patch == remote_patch:
            LOGGER.logInfo("same version!")
            return ''

        err_key = "common.prduct.version.or.patch"
        return common.getMsg(LANG, err_key)
    LOGGER.logInfo("not c version same!")
    return ''


def check_special_version(local_info_tuple, remote_info_tuple):
    """大版本不支持互通的其他特殊互通，其他版本先检查特殊型号是否支持互通
    :param local_info_tuple:本端信息
    :param remote_info_tuple:远端信息
    :return:flag:是否互通，list:错误消息
    """
    err_msg_list = []
    local_model, local_version, local_patch, _ = local_info_tuple
    remote_model, remote_version, remote_patch, _ = remote_info_tuple

    special_model_can_link = False
    for rule in special_hyper_rule_list:
        ge_version_list = rule.get("c_version")
        ge_model_rule_list = rule.get("model_rule_list")

        # 检查型号
        special_model_can_link = check_model_can_link(local_model,
                                                      remote_model,
                                                      ge_model_rule_list)
        if not special_model_can_link:
            continue

        # 如果未配置版本，则需要版本和补丁一致。
        if not ge_version_list:
            if local_version != remote_version \
                    or local_patch != remote_patch:
                err_key = "common.prduct.version.or.patch"
                err_msg_list.append(common.getMsg(LANG, err_key))

            return True, list(set(err_msg_list))

        # 如果配置了版本，则需要判断版本是否互通
        else:
            if not check_c_version(local_version, ge_version_list) \
                    or not check_c_version(remote_version, ge_version_list):
                # 型号相同，但不在这个版本，则认为不互通，需要信息一致。
                special_model_can_link = False
                continue

            # 型号互通，比较版本要一致,补丁版本要一致
            if local_version != remote_version \
                    or local_patch != remote_patch:
                err_key = "common.prduct.version.or.patch"
                err_msg_list.append(common.getMsg(LANG, err_key))

            return True, list(set(err_msg_list))

    # 返回是否特殊型号或版本互通
    return special_model_can_link, list(set(err_msg_list))


def check_product_consistency(local_info_tuple, remote_info_tuple):
    """检查版本型号是否一致
    :param local_info_tuple:本端信息
    :param remote_info_tuple:远端信息
    :return:err_msg_list错误消息列表
    """
    err_msg_list = []
    # 如果型号、版本号、补丁都相等。
    if local_info_tuple == remote_info_tuple:
        return err_msg_list

    local_model, local_version, local_patch, local_pro_ver = local_info_tuple
    remote_model, remote_version, remote_patch, remote_patch_pro_ver = \
        remote_info_tuple

    # 互通版本和型号判断
    can_link, err_msg_tmp = check_general_version(local_info_tuple,
                                                  remote_info_tuple)
    err_msg_list.extend(err_msg_tmp)
    LOGGER.logInfo(u"general is find:%s, %s" % (can_link, err_msg_tmp))
    if can_link:
        return list(set(err_msg_list))

    # 特殊型号和版本判断。
    can_link, err_msg_tmp = check_special_version(local_info_tuple,
                                                  remote_info_tuple)
    err_msg_list.extend(err_msg_tmp)
    LOGGER.logInfo(u"special is find:%s, %s" % (can_link, err_msg_tmp))
    if can_link:
        return list(set(err_msg_list))

    LOGGER.logInfo("other sence is find.")
    # 其他不互通场景，必须所有都一致。
    if local_model != remote_model:
        err_msg_list.append(common.getMsg(LANG, "common.prduct.model"))

    if local_version != remote_version or local_patch != remote_patch:
        err_key = "common.prduct.version.or.patch"
        err_msg_list.append(common.getMsg(LANG, err_key))

    return list(set(err_msg_list))


def checkControllerConsistency(localControllerNum, remoteDevSN):
    '''
    @summary: check the consistency of the number of controller
    '''
    cliEexcuteRet = common.getObjFromFile(py_java_env, LOGGER, remoteDevSN, controllerGeneral, LANG)
    global cliRet
    cliRet += cliEexcuteRet[1] + lineBreak
    if cliEexcuteRet[0] != True:
        throwException('remoteCli execute command %s failure' % controllerGeneral)
    remoteControllerNum = getControllerNum(cliEexcuteRet[1])
    if not remoteControllerNum == localControllerNum:
        return common.getMsg(LANG, "common.prduct.ctrl.num")

    return ''

def getControllerNum(ret):
    '''
    @summary: get the total number of controllers by reading 'location' info
    '''
    locationLineList = list(
        filter(lambda line: 'Running Status' in line.strip() and line.strip().endswith('Online'), ret.splitlines()))
    LOGGER.logInfo("***[getControllerNum= %s]****" % str(locationLineList))
    return len(locationLineList)


def getProductModelVersion(LOGGER, DevSN, ret, need_ret=True):
    '''
    @summary: get product model, product version and patch version
    '''
    global cliRet
    cmd = "show upgrade package"
    vertivalCliRetList = cliUtil.getVerticalCliRet(ret)
    model = ''
    version = ''
    patch = ''
    product_version = ''
    if vertivalCliRetList:
        dictRet = vertivalCliRetList[0]
        model = dictRet.get('Product Model', '')
        product_version = dictRet.get('Product Version', '')
        cliExecuteRet = common.getObjFromFile(py_java_env, LOGGER, DevSN, cmd, LANG)
        if need_ret:
            cliRet += cliExecuteRet[1] + lineBreak
        checkResult = getVersion(cliExecuteRet[1], LANG)
        version = checkResult[3]
        patch = checkResult[4]
    return (model, version, patch, product_version)


def throwException(exceptionMsg):
    '''
    @summary: throw custom exception
    '''
    raise BaseException(exceptionMsg)


def getVersion(cliRet, LANG):
    """
    @summary: 获取阵列软件版本与热补丁版本
    """
    (flag, cliRet, errMsg), softwareVersionList, hotPatchVersionList = parse_upgradePackage(cliRet, LANG)
    if flag != True:
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg, "", "")

    softwareVersion = softwareVersionList[0].get("Current Version")
    hotPatchVersion = hotPatchVersionList[0].get("Current Version")
    return (True, cliRet, "", softwareVersion, hotPatchVersion)


def parse_upgradePackage(cliRet, lang):
    '''
    @summary: 执行show upgrade packge命令，将Software Version与HotPatch Version的
              回显存放到同一个字典列表中，并增加Version Type键来区分Version类型,并返回CLI回显
    @param cli: cli对象
    @param lang: 语言lang
    '''
    errMsg = ""
    flag = True

    softwareVersionIndex = cliRet.find("Software Version")
    hotPatchVersionIndex = cliRet.find("HotPatch Version")
    softwareVersionList = cliUtil.getHorizontalCliRet(cliRet[(softwareVersionIndex):hotPatchVersionIndex])
    hotPatchVersionList = cliUtil.getHorizontalCliRet(cliRet[(hotPatchVersionIndex):])

    if len(softwareVersionList) == 0 or len(hotPatchVersionList) == 0:
        errMsg += common.getMsg(lang, "cannot.get.upgrade.package.info")
        return ((False, cliRet, errMsg), softwareVersionList, hotPatchVersionList)

    for softwareVersion in softwareVersionList:
        if softwareVersion["Current Version"] == "--":
            flag = False
            errMsg += common.getMsg(lang, "cannot.get.contrller.version.info", softwareVersion.get("Name", ""))

    if not common.isControllerVersionSame(softwareVersionList):
        errMsg += common.getMsg(lang, "controller.version.not.consistence")
        return ((False, cliRet, errMsg), softwareVersionList, hotPatchVersionList)

    return ((flag, cliRet, errMsg), softwareVersionList, hotPatchVersionList)
