﻿# -*- coding: UTF-8 -*-
import cliUtil
import common

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    '''
    SAN双活状态检查
        1 如果双活仲裁链路状态不为Link Up，表示不正常；
        2 如果双活仲裁服务器运行状态不为Online，表示不正常；
        3 如果双活域的运行状态不为Normal表示不正常；
        4 如果双活Pair的健康状态不为Normal或者链路状态不为Link Up或者运行状态不为Normal/Synchronizing/Paused中的一种表示不正常；
        5 如果双活一致性组的健康状态不为Normal或者运行状态不为Normal/Synchronizing/Paused中的一种表示不正常；
        6 其它情况均属于正常状态。

    '''
    flag_lst = []
    cliRet = ""
    errMsg = ""

    try:
        for ret in (
        coordinatorPointServerLink(cli), coordinatorPointServer(cli), hyperClusterDomain(cli), hyperConsistGroup(cli),
        hyperClusterPair(cli), hyperVstorePair(cli)):
            flag_lst.append(ret[0])
            cliRet += ret[1]
            errMsg += ret[2]
        flag_lst = set(flag_lst)
        if False in flag_lst:
            return False, cliRet, errMsg
        elif cliUtil.RESULT_NOSUPPORT in flag_lst:
            return cliUtil.RESULT_NOSUPPORT, cliRet, errMsg
        elif cliUtil.RESULT_NOCHECK in flag_lst:
            return cliUtil.RESULT_NOCHECK, cliRet, errMsg
        else:
            return True, cliRet, errMsg
    except Exception, exception:
        LOGGER.logException(exception)
        return cliUtil.RESULT_NOCHECK, cliRet, common.getMsg(LANG, "query.result.abnormal")


def coordinatorPointServerLink(cli):
    '''
    @summary: 检查双活仲裁服务器链路状态是否正常
    @param cli: cli对象
    @return: (flag, cliRet, errMsg)
        flag:
            True: 状态健康
            False: 状态不健康
        cliRet:
            Flag为True时，返回设备上的信息
            Flag为False时，返回cli回显结果
        errMsg: 错误消息
        suggestionInfo:修改建议（15365）
    '''
    errMsg = ""
    flag = True

    cmd = "show quorum_server_link general"
    LOGGER.logExecCmd(cmd)
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if checkRet[0] != True:
        LOGGER.logSysAbnormal()
        return checkRet

    cliRet = checkRet[1]
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)

    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = common.getMsg(LANG, "cannot.get.quorum.server.link.info")
        LOGGER.logNoPass("Failed to obtain the status of active-active quorum links.")
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg)  # 修改备注：解析回显失败导致列表长度为0

    for line in cliRetLinesList:
        linkId = line.get("Link ID", "")
        linkStatus = line.get("Link Status", "")
        if linkStatus != common.STATUS_LINK_UP:
            flag = False
            errMsg += common.getMsg(LANG, "quorum.server.link.status.abnormal", (linkId, linkStatus))
            LOGGER.logNoPass(
                "The status of the links to active-active quorum server [%s] is abnormal (link status: %s)."
                % (linkId, linkStatus))

    if not flag:
        return (flag, cliRet, errMsg)

    return (flag, cliRet, errMsg)


def coordinatorPointServer(cli):
    '''
    @summary: 查看双活仲裁服务器运行状态是否正常
    @param cli: cli对象
    @return: (flag, cliRet, errMsg)
        flag:
            True: 状态健康
            False: 状态不健康
        cliRet:
            Flag为True时，返回设备上的信息
            Flag为False时，返回cli回显结果
        errMsg: 错误消息
        suggestionInfo:修改建议（15363）
    '''
    errMsg = ""
    flag = True

    cmd = "show quorum_server general|filterColumn include columnList=Server\sID,Running\sStatus"
    LOGGER.logExecCmd(cmd)
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if checkRet[0] != True:
        LOGGER.logSysAbnormal()
        return checkRet

    cliRet = checkRet[1]
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)

    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = common.getMsg(LANG, "cannot.get.quorum.server.info")
        LOGGER.logNoPass("Failed to obtain the running status of the active-active quorum server.")
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg)  # 修改备注：解析回显失败导致列表长度为0

    for line in cliRetLinesList:
        serverId = line.get("Server ID", "")
        runningStatus = line.get("Running Status", "")
        if runningStatus != common.STATUS_ONLINE:
            flag = False
            errMsg += common.getMsg(LANG, "quorum.server.running.status.abnormal", (serverId, runningStatus))
            LOGGER.logNoPass("The running status of active-active quorum server [%s] is abnormal (running status: %s)."
                             % (serverId, runningStatus))

    if not flag:
        return (flag, cliRet, errMsg)

    return (flag, cliRet, errMsg)


def hyperClusterDomain(cli):
    '''
    @summary: 检查双活域运行状态是否正常
    @param cli: cli对象
    @return: (flag, cliRet, errMsg)
        flag:
            True: 状态健康
            False: 状态不健康
        cliRet:
            Flag为True时，返回设备上的信息
            Flag为False时，返回cli回显结果
        errMsg: 错误消息
        suggestionInfo:修改建议（15362）
    '''
    errMsg = ""
    flag = True

    cmd = "show hyper_metro_domain general"
    LOGGER.logExecCmd(cmd)
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if checkRet[0] != True:
        LOGGER.logSysAbnormal()
        return checkRet

    cliRet = checkRet[1]
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)

    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = common.getMsg(LANG, "cannot.get.hyper.metro.domain.info")
        LOGGER.logNoPass("Failed to obtain the running status of the active-active domain.")
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg)  # 修改备注：解析回显失败导致列表长度为0

    for line in cliRetLinesList:
        Id = line.get("ID", "")
        runningStatus = line.get("Running Status", "")
        if runningStatus != common.STATUS_NORMAL:
            flag = False
            errMsg += common.getMsg(LANG, "hyper.metro.domain.running.status.abnormal", (Id, runningStatus))
            LOGGER.logNoPass("The Health Status of hyper cluster domain [%s] is abnormal (Running Status:%s)"
                             % (Id, runningStatus))
    if not flag:
        return (flag, cliRet, errMsg)

    return (flag, cliRet, errMsg)


def hyperClusterPair(cli):
    '''
    @summary: 检查双活Pair状态是否正常
    @param cli: cli对象
    @return: (flag, cliRet, errMsg)
        flag:
            True: 状态健康
            False: 状态不健康
        cliRet:
            Flag为True时，返回设备上的信息
            Flag为False时，返回cli回显结果
        errMsg: 错误消息
        suggestionInfo:修改建议（15361）
    '''
    errMsg = ""
    flag = True

    cmd = "show hyper_metro_pair general |filterRow column=Type " \
          "predict=equal_to value=LUN |filterColumn include columnList=ID," \
          "Health\sStatus,Link\sStatus,Running\sStatus"
    LOGGER.logExecCmd(cmd)
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if checkRet[0] != True:
        LOGGER.logSysAbnormal()
        return checkRet

    cliRet = checkRet[1]
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)

    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = common.getMsg(LANG, "cannot.get.hyper.metro.pair.info")
        LOGGER.logNoPass("Failed to obtain the status of the active-active pair.")
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg)  # 修改备注：解析回显失败导致列表长度为0

    for line in cliRetLinesList:
        Id = line.get("ID", "")
        healthStatus = line.get("Health Status", "")
        linkStatus = line.get("Link Status", "")
        runningStatus = line.get("Running Status", "")

        if healthStatus != common.STATUS_NORMAL:
            flag = False
            errMsg += common.getMsg(LANG, "hyper.metro.pair.health.status.abnormal", (Id, healthStatus))
            LOGGER.logNoPass("The health status of active-active pair [%s] is abnormal (health status: %s)."
                             % (Id, healthStatus))

        if linkStatus not in (common.STATUS_LINK_UP, "Linkup"):
            flag = False
            errMsg += common.getMsg(LANG, "hyper.metro.pair.link.status.abnormal", (Id, linkStatus))
            LOGGER.logNoPass("The link status of active-active pair [%s] is abnormal (link status: %s)."
                             % (Id, linkStatus))

        if runningStatus not in (common.STATUS_NORMAL, common.STATUS_SYNCHRONIZING, common.STATUS_PAUSE):
            flag = False
            errMsg += common.getMsg(LANG, "hyper.metro.pair.running.status.abnormal", (Id, runningStatus))
            LOGGER.logNoPass("The running status of active-active pair [%s] is abnormal (running status: %s)."
                             % (Id, runningStatus))

    if not flag:
        return (flag, cliRet, errMsg)

    return (flag, cliRet, errMsg)


def hyperConsistGroup(cli):
    '''
    @summary: 检查双活一致性组状态是否正常
    @param cli: cli对象
    @return: (flag, cliRet, errMsg)
        flag:
            True: 状态健康
            False: 状态不健康
        cliRet:
            Flag为True时，返回设备上的信息
            Flag为False时，返回cli回显结果
        errMsg: 错误消息
        suggestionInfo:修改建议（15364）
    '''
    errMsg = ""
    flag = True

    cmd = "show hyper_metro_consistency_group general"
    LOGGER.logExecCmd(cmd)
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)

    cliRet = checkRet[1]
    if not cliUtil.hasCliExecPrivilege(cliRet):
        return (True, "", errMsg)

    if checkRet[0] != True:
        LOGGER.logSysAbnormal()
        return checkRet

    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)

    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = common.getMsg(LANG, "cannot.get.hyper.consist.group.info")
        LOGGER.logNoPass("Failed to obtain the information about the status of the active-active consistency groups.")
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg)  # 修改备注：解析回显失败导致列表长度为0

    for line in cliRetLinesList:
        Id = line.get("ID", "")
        healthStatus = line.get("Health Status", "")
        runningStatus = line.get("Running Status", "")

        if healthStatus != common.STATUS_NORMAL:
            flag = False
            errMsg += common.getMsg(LANG, "hyper.consist.group.health.status.abnormal", (Id, healthStatus))
            LOGGER.logNoPass(
                "The health status of active-active consistency group [%s] is abnormal (health status: %s)."
                % (Id, healthStatus))

        if runningStatus not in (common.STATUS_NORMAL, common.STATUS_SYNCHRONIZING, common.STATUS_PAUSE):
            flag = False
            errMsg += common.getMsg(LANG, "hyper.consist.group.running.status.abnormal", (Id, runningStatus))
            LOGGER.logNoPass(
                "The running status of active-active consistency group [%s] is abnormal (running status: %s)."
                % (Id, runningStatus))

    if not flag:
        return (flag, cliRet, errMsg)

    return (flag, cliRet, errMsg)


def hyperVstorePair(cli):
    '''
    @summary: 检查双活租户Pair状态是否正常
    @param cli: cli对象
    @return: (flag, cliRet, errMsg)
        flag:
            True: 状态健康
            False: 状态不健康
        cliRet:
            Flag为True时，返回设备上的信息
            Flag为False时，返回cli回显结果
        errMsg: 错误消息
        suggestionInfo:修改建议
    '''
    errMsg = ""
    flag = True

    cmd = "show hyper_metro_vstore_pair general|filterColumn include columnList=ID,Health\sStatus,Running\sStatus,Link\sStatus,Config\sStatus"
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)

    cliRet = checkRet[1]
    if not cliUtil.hasCliExecPrivilege(cliRet):
        return (True, "", errMsg)

    if checkRet[0] != True:
        LOGGER.logSysAbnormal()
        return checkRet

    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)

    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = common.getMsg(LANG, "cannot.get.hyper.vstore.pair.info")
        LOGGER.logNoPass("Failed to obtain the information about the status of the active-active vstore pair.")
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg)  # 修改备注：解析回显失败导致列表长度为0

    for line in cliRetLinesList:
        Id = line.get("ID")
        healthStatus = line.get("Health Status")
        runningStatus = line.get("Running Status")
        linkStatus = line.get("Link Status")
        configStatus = line.get("Config Status")

        if healthStatus != common.STATUS_NORMAL:
            flag = False
            errMsg += common.getMsg(LANG, "hyper.vstore.pair.health.status.abnormal", (Id, healthStatus))
            LOGGER.logNoPass("The health status of active-active vstore pair [%s] is abnormal (health status: %s)."
                             % (Id, healthStatus))

        if runningStatus not in (common.STATUS_NORMAL, common.STATUS_UNSYNCHRONIZED):
            flag = False
            errMsg += common.getMsg(LANG, "hyper.vstore.pair.running.status.abnormal", (Id, runningStatus))
            LOGGER.logNoPass("The running status of active-active vstore pair [%s] is abnormal (running status: %s)."
                             % (Id, runningStatus))

        if linkStatus not in (common.STATUS_LINK_UP, "Linkup"):
            flag = False
            errMsg += common.getMsg(LANG, "hyper.vstore.pair.link.status.abnormal", (Id, linkStatus))
            LOGGER.logNoPass("The link status of active-active vstore pair [%s] is abnormal (link status: %s)."
                             % (Id, linkStatus))

        if runningStatus == common.STATUS_NORMAL and configStatus != common.STATUS_NORMAL:
            flag = False
            errMsg += common.getMsg(LANG, "hyper.vstore.pair.config.status.abnormal", (Id, configStatus))
            LOGGER.logNoPass("The config status of active-active vstore pair [%s] is abnormal (config status: %s)."
                             % (Id, configStatus))

    if not flag:
        return (flag, cliRet, errMsg)

    return (flag, cliRet, errMsg)
