# -*- coding:utf-8 -*-
import traceback
import re

import cliUtil
import common
from common import UnCheckException
from com.huawei.ism.tool.obase.exception import ToolException
from risk_version_config import (
    RISK_IP_CONFLICT_WITH_SCALEOUT_VERSION as system_software_version,
)

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env


def execute(cli):
    ip_check = IPCheck(cli, LANG, PY_JAVA_ENV, LOGGER)
    flag, msg = ip_check.execute_check()
    return flag, "\n".join(ip_check.all_ret_list), msg


class IPCheck:
    """
    检查业务IP是否使用了172.16网段
    """

    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_ret_list = []
        self.software_version = ""
        self.hot_patch_version = ""

    def execute_check(self):
        try:
            # 查询到的版本不在名单内检查项直接通过
            if not self.is_risk_version():
                return True, ""
            # 查询到的版本和热补丁关系正确直接通过
            if self.check_hot_patch_version_relationship():
                return True, ""
            info_content = self.query_port_ip()
            port_ip = self.filter_info(info_content)
            info_content1 = self.query_logical_port_info()
            logical_port_ip = self.filter_info(info_content1)
            if not port_ip and not logical_port_ip:
                return True, ""
            error_message_list = []
            if port_ip:
                msg = self.get_port_name_message(
                    port_ip,
                    "Port ID",
                    "check.ip.conflict.with.scaleout.err_port_id",
                )
                error_message_list.append(msg)
            if logical_port_ip:
                msg = self.get_port_name_message(
                    logical_port_ip,
                    "Logical Port Name",
                    "check.ip.conflict.with.scaleout.err_logical_port_id",
                )
                error_message_list.append(msg)
            if error_message_list:
                return False, "".join(error_message_list)
        except common.UnCheckException as e:
            self.logger.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except ToolException:
            self.logger.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )
        except Exception:
            self.logger.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )
        finally:
            # 退出到cli模式
            ret = cliUtil.enterCliModeFromSomeModel(self.cli, self.lang)
            self.logger.logInfo(
                "enter cli mode from some model ret is %s" % str(ret)
            )
            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                common.reConnectionCli(self.cli, self.logger)

    def get_port_name_message(self, port_info_list, port_name, resource):
        """
        获取ip不满足条件的端口名字信息
        :param port_info_list: 端口信息列表
        :param port_name: 要获取字段的名字
        :param resource: 相应resource文件里的键
        :return: message
        """
        temp_list = []
        for ip_info in port_info_list:
            temp_list.append(ip_info.get(port_name))
        msg = common.getMsg(self.lang, resource, (",".join(temp_list)))
        return msg

    def is_risk_version(self):
        """
        判断产品是否满足检查的版本及型号范围
        :param: self
        :return:boolean 真代表是风险版本
        """
        (
            flag,
            cli_ret,
            err_msg,
            self.software_version,
            self.hot_patch_version,
        ) = common.getVersion(self.cli, self.lang)
        self.all_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)
        if self.software_version in system_software_version:
            return True
        return False

    def check_hot_patch_version_relationship(self):
        """
        判断产品是否满足检查的版本和热补丁关系
        :return:
        """
        if self.software_version == "6.0.1":
            pattern = re.compile(r"SPH(\d+)", flags=re.IGNORECASE)
            match_hot_path = pattern.search(self.hot_patch_version)
            return match_hot_path and int(
                pattern.search(self.hot_patch_version).group(1)
            ) >= 6
        return False

    def query_port_ip(self):
        """
        获取port_ip信息
        :return:
        """
        cmd = "show port ip"
        flag, ret, msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)
        port_ip_info_list = []
        host_port_begin = ret.find(
            "---------------  Host Port:----------------"
        )
        management_port_begin = ret.find(
            "------------  Management Port:-------------"
        )
        maintenance_port_begin = ret.find(
            "------------  Maintenance Port:------------"
        )
        port_ip_info_list.extend(
            cliUtil.getHorizontalCliRet(
                ret[host_port_begin:management_port_begin]
            )
        )
        port_ip_info_list.extend(
            cliUtil.getHorizontalCliRet(
                ret[management_port_begin:maintenance_port_begin]
            )
        )
        port_ip_info_list.extend(
            cliUtil.getHorizontalCliRet(ret[maintenance_port_begin:])
        )
        return port_ip_info_list

    def query_logical_port_info(self):
        """
        查询所有的逻辑端口信息
        :param self:
        :return:
        """
        cmd = "show logical_port general"
        flag, ret, msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)
        # 按逐行字典的方式获取水平表格形式的cli回显列表
        logical_port_info_list = cliUtil.getHorizontalCliRet(ret)
        return logical_port_info_list

    @staticmethod
    def filter_info(content):
        """
        过滤掉Home Port Type非Bond和VLAN且Failover Group ID不在范围
        [1,1024]的字段
        :param content
        :return: [{}.....]
        """
        # 收集 IPv4 Address属于172.16.[0-3].[1-8]范围的信息。
        temp_object = list(
            filter(
                lambda x: re.match(
                    "172.16.[0-3].[1-8]$", x.get("IPv4 Address")
                ),
                content,
            )
        )
        return temp_object
