# -*- coding:utf-8 -*-
import traceback
import re

import cliUtil
import common
from common import UnCheckException
from com.huawei.ism.tool.obase.exception import ToolException
from risk_version_config import (
    RISK_LIBRARY_DEVICE_MAP_TO_STORAGE_VERSION as risk_software_versions,
)

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env
LIBRARY_DEVICE = re.compile(r"/dev/st\d+")


def execute(cli):
    check_st_device = CheckStDevice(cli, LANG, PY_JAVA_ENV, LOGGER)
    flag, msg = check_st_device.execute_check()
    return flag, "\n".join(check_st_device.all_ret_list), msg


class CheckStDevice:
    """
    检查存储上是否映射了磁带库设备
    """

    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.conn_cli = ""
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_ret_list = []
        self.software_version = ""
        self.hot_patch_version = ""

    def execute_check(self):
        try:
            # 查询到的版本不在名单内检查项直接通过
            if not self.is_risk_version():
                return True, ""
            # 查询到的版本和热补丁关系正确直接通过
            if not self.is_risk_software_and_patch_relationship():
                return True, ""
            create_ret = common.createDeviceCliContFor18000(
                self.cli, self.env, self.logger, self.lang
            )
            if create_ret[0] is not True:
                return cliUtil.RESULT_NOCHECK, create_ret[2]
            self.conn_cli = create_ret[1]
            node_id_and_controller_map = self.build_controller_name(
                self.conn_cli, self.env, self.logger, self.lang
            )
            # 遍历各个控制器
            controller_library_device_map = dict()
            self.make_dic_controller_name_and_library_device_info(
                self.conn_cli,
                self.env,
                self.logger,
                self.lang,
                "",
                self.all_ret_list,
                node_id_and_controller_map,
                controller_library_device_map,
            )
            if not controller_library_device_map:
                return True, ""
            wrong_msg = []
            system_msg = common.getMsg(
                self.lang,
                "check.st.device.system_message",
                (self.software_version, self.hot_patch_version),
            )
            wrong_msg.append(system_msg)
            for key in sorted(controller_library_device_map):
                msg = common.getMsg(
                    self.lang,
                    "check.st.device.controller_library_device",
                    (
                        key,
                        ",".join(controller_library_device_map.get(key)),
                    ),
                )
                wrong_msg.append(msg)
            return False, "".join(wrong_msg)
        except common.UnCheckException as e:
            self.logger.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except ToolException:
            self.logger.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )
        except Exception:
            self.logger.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )
        finally:
            if self.conn_cli != "":
                if (
                    common.is18000(PY_JAVA_ENV, self.conn_cli)
                    and self.conn_cli is not self.cli
                ):
                    common.closeConnection(self.conn_cli, PY_JAVA_ENV, LOGGER)
            # 退出到cli模式
            ret = cliUtil.enterCliModeFromSomeModel(self.cli, LANG)
            LOGGER.logInfo(
                "enter cli mode from some model ret is %s" % str(ret)
            )
            # 退出到cli模式
            ret = cliUtil.enterCliModeFromSomeModel(self.cli, LANG)
            self.logger.logInfo(
                "enter cli mode from some model ret is %s" % str(ret)
            )

            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                common.reConnectionCli(self.cli, LOGGER)

    def is_risk_version(self):
        """
        判断产品是否满足检查的版本及型号范围
        :param: self
        :return:boolean 真代表是风险版本
        """
        (
            flag,
            cli_ret,
            err_msg,
            self.software_version,
            self.hot_patch_version,
        ) = common.getVersion(self.cli, self.lang)
        self.all_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)
        if self.software_version in risk_software_versions:
            return True
        return False

    def is_risk_software_and_patch_relationship(self):
        """
        判断产品是否满足检查的版本和热补丁关系
        :return:
        """
        return (
            risk_software_versions.get(self.software_version)
            > self.hot_patch_version
        )

    @staticmethod
    def build_controller_name(cli, env, logger, lang):
        """
        控制器名称为
        构建controller名称
        :return:
        """
        present_node_id_list = CheckStDevice.get_engine_node_id_list(
            cli, env, logger
        )
        check_ret = cliUtil.getControllerIdList(cli, lang)
        if check_ret[0] is not True:
            raise common.UnCheckException(check_ret[2], check_ret[1])
        node_id_and_controller_map = dict(
            zip(present_node_id_list, check_ret[1])
        )
        return node_id_and_controller_map

    @staticmethod
    def get_engine_node_id_list(cli, env, logger):
        """
        @summary: 获取当前引擎的全部健康节点信息，当前节点信息。
        @return: current_engine_ctrl_id_list)
            current_engine_ctrl_id_list: 当前引擎的全部健康节点信息
        """
        # 获取当前节点信息，当前引擎的全部节点信息。
        cmd = "sys showcls"
        # 当前节点
        current_engine_ctrl_id_list = []
        flag, ret, msg = common.executeDebugCommand(cli, cmd, env, logger)
        if flag is not True:
            raise common.UnCheckException(msg, ret)
        engine_info_dict_list = cliUtil.getHorizontalNostandardCliRet(ret)
        for engine in engine_info_dict_list:
            ctrl_node_id = engine.get("id", "")
            if ctrl_node_id == "":
                continue
            current_engine_ctrl_id_list.append(ctrl_node_id)
        return current_engine_ctrl_id_list

    @staticmethod
    @common.checkAllEngineInClusterWarp
    def make_dic_controller_name_and_library_device_info(
        cli,
        env,
        logger,
        lang,
        heart_beat_cli_ret,
        all_ret_list,
        node_id_and_controller_map,
        controller_library_device_map,
    ):
        all_ret_list.append(heart_beat_cli_ret)
        (
            _,
            err_msg,
            cli_ret,
            current_ctrl,
            engine_ctrl_mapping_dict,
            current_engine,
            node_num,
        ) = common.getEngineCtrlNodeInfo(cli, logger, lang)
        current_controller = node_id_and_controller_map.get(current_ctrl, "--")
        cmd = "lsscsi"
        flag, ret, msg = cliUtil.excuteCmdInMinisystemModel(cli, cmd, lang)
        all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)
        library_device_list = LIBRARY_DEVICE.findall(ret)
        if library_device_list:
            controller_library_device_map[
                current_controller
            ] = library_device_list
