# -*- coding: UTF-8 -*-
"""
@desc: 元数据损坏风险检测
@date: 2021.1.26
"""


import traceback
from decimal import Decimal

from cliUtil import RESULT_NOCHECK
import common

from com.huawei.ism.tool.obase.exception import ToolException

from cbb.frame.context import sqlite_context

import common_cache
import cli_util_cache
from risk_version_config import META_DATA_HAS_PATCH_VERSION
from common_utils import (
    is_risk_version_fc_cause_risk,
    get_err_msg
)


PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)

# 文件系统常量
FILE_SYSTEM_CAPACITY_VALUE = Decimal(114024960)
# thin lun 常量
THIN_LUN_CAPACITY_VALUE = Decimal(136365211648)
# 如果有补丁，更严格的检查
MORE_STRICTNESS_THRESHOLD = Decimal(7) / Decimal(10)
DEFAULT_PERCENT = Decimal(1)
MAGIC_NUMBER_HUNDRED = Decimal(100)
LUN_RET_TEMPLATE = "---------\n LUN ID:{}\nSubscribed Capacity:{}\n--------\n"
FILE_SYSTEM_RET_TEMPLATE = "--------------------\nfile system id:{}\n" \
                           "has patch:{}\nAllocated Capacity:{}\n" \
                           "File system Capacity:{}\nStorage Pool Extent " \
                           "Size:{}\nSnapshot Reserve Percent:{}\n" \
                           "--------------------\n"


def execute(cli):
    """
    元数据损坏风险检测
    :param cli:
    :return:
    """

    all_cli_ret_list = []
    error_msg_list = []
    mem_version = str(PY_JAVA_ENV.get("devInfo").getProductVersion())

    try:
        (
            flag, p_version, p_patch, ret, msg,
        ) = common_cache.get_version_and_patch_cache(PY_JAVA_ENV, cli, LOGGER)
        all_cli_ret_list.append(ret)

        if flag is not True:
            return RESULT_NOCHECK, ret, msg

        if install_patch(mem_version, p_patch):
            return True, ret, ""

        has_patch_flag = has_patch(mem_version)
        is_risk, need_patch = is_risk_version_fc_cause_risk(
            cli, p_version, p_patch, LOGGER
        )

        if is_risk:
            err_key = "install.fc.driver.risk.patch"
            return (RESULT_NOCHECK, ret,
                    common.getMsg(LANG, err_key, need_patch))

        check_risk(has_patch_flag, cli, p_version, all_cli_ret_list,
                   error_msg_list)
        all_ret = "\n".join(all_cli_ret_list)

        if error_msg_list:
            return False, all_ret, "".join(error_msg_list)

        return True, all_ret, ""

    except (ToolException, Exception):
        LOGGER.logError(traceback.format_exc())
        return (
            RESULT_NOCHECK,
            "\n".join(all_cli_ret_list),
            common.getMsg(LANG, "query.result.abnormal")
        )


def check_risk(patch_flag, cli, p_version, all_cli_ret_list, error_msg_list):
    """
    检查是否存在风险
    :param patch_flag:
    :param cli:
    :param p_version:
    :param all_cli_ret_list:
    :param error_msg_list:
    :return:
    """
    sn = common.get_sn_from_env(PY_JAVA_ENV)
    cache_object = PY_JAVA_ENV.get("objectForPy")
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        cache_object, sn, LOGGER
    )

    if not cli_util_cache.is_downloaded_config(PY_JAVA_ENV, sn):
        flag = cli_util_cache.download_and_update_cache(
            sn, cache_object, LOGGER, cli, LANG, sqlite_conn, p_version
        )
        if not flag:
            return

    flag, error_msg = check_file_system(
        patch_flag, sn, cache_object, all_cli_ret_list
    )

    if not flag:
        error_msg_list.append(error_msg)

    flag, error_msg = check_thin_lun(
        patch_flag, sn, cache_object, all_cli_ret_list
    )

    if not flag:
        error_msg_list.append(error_msg)


def check_file_system(patch_flag, sn, cache_object, all_cli_ret_list):
    """
    检查文件系统
    :param patch_flag:
    :param sn:
    :param cache_object:
    :param all_cli_ret_list:
    :return:
    """
    file_system_info = cli_util_cache.get_file_system_info_from_config(
        cache_object, LOGGER, sn
    )

    if not file_system_info:
        return True, ''

    storage_pool_info = cli_util_cache.get_storage_pool_info_from_config(
        cache_object, LOGGER, sn
    )
    LOGGER.logInfo(
        "file_system_info:{}, storage_pool_info:{}".format(file_system_info,
                                                           storage_pool_info))
    risk_file_system_id = []

    for file_system in file_system_info:
        file_system_id = file_system.get("Fs Id")
        all_located_capacity = file_system.get("Allocated Capacity", "0")

        file_system_capacity = file_system.get("File system Capacity", "0")
        snapshot_reserve_percent = file_system.get(
            "Snapshot Reserve Percent", "0"
        )
        pool_id = file_system.get("Userpool Id", "")
        pool_extent_size = "4096"

        if not pool_id or pool_id == "--":
            continue

        for pool in storage_pool_info:
            if pool.get("Storage Pool ID") != pool_id:
                continue
            pool_extent_size = pool.get("Storage Pool Extent Size", "4096")
            break
        all_cli_ret_list.append(FILE_SYSTEM_RET_TEMPLATE.format(
            file_system_id, patch_flag, all_located_capacity,
            file_system_capacity, pool_extent_size,
            snapshot_reserve_percent)
        )

        if is_risk_file_system(
                snapshot_reserve_percent,
                deal_unit(all_located_capacity),
                deal_unit(file_system_capacity),
                deal_unit(pool_extent_size),
                patch_flag,
        ):
            risk_file_system_id.append(file_system_id)

    if risk_file_system_id and patch_flag:
        error_key = "metadata.risky.fs.has.patch.not.pass"
        return False, get_err_msg(
            LANG, error_key, (", ".join(risk_file_system_id), patch_flag))

    if risk_file_system_id:
        error_key = "metadata.risky.fs.no.patch.not.pass"
        return False, get_err_msg(
            LANG, error_key, ", ".join(risk_file_system_id))

    return True, ''


def deal_unit(capacity):
    """
    容错处理，一般是kb。
    :param capacity:
    :return:
    """
    if "KB" in capacity:
        capacity = capacity.rstrip("(KB)")
    elif "MB" in capacity:
        capacity = Decimal(capacity.rstrip("(MB)")) * 1024
    elif "GB" in capacity:
        capacity = Decimal(capacity.rstrip("(GB)")) * 1024 * 1024
    elif "TB" in capacity:
        capacity = Decimal(capacity.rstrip("(TB)")) * 1024 * 1024 * 1024

    return str(capacity)


def is_risk_file_system(
        snapshot_reserve_percent, all_located_capacity,
        file_system_capacity, pool_extent_size, has_patch_flag
):
    """
    检查文件系统
    """
    LOGGER.logInfo(
        "snapshot reserve:{}, all_located_capacity:{}, "
        "file_system_capacity:{}, pool_extent_size:{}".format(
            snapshot_reserve_percent, all_located_capacity,
            file_system_capacity, pool_extent_size)
    )
    # 获取快照百分比
    snapshot_reserve_percent = Decimal(
        snapshot_reserve_percent) / MAGIC_NUMBER_HUNDRED
    # 获取已分配容量
    object_allocated_capacity = (Decimal(all_located_capacity) - Decimal(
        file_system_capacity) * snapshot_reserve_percent)
    # 计算阈值，已打补丁和未打补丁的
    pool_extent_size = Decimal(pool_extent_size)
    threshold = Decimal(
        pool_extent_size
        * FILE_SYSTEM_CAPACITY_VALUE
        * MORE_STRICTNESS_THRESHOLD
        if has_patch_flag
        else pool_extent_size * FILE_SYSTEM_CAPACITY_VALUE
    )
    LOGGER.logInfo("object_allocated_capacity:{}, threshold:{}".format(
        object_allocated_capacity, threshold))

    return object_allocated_capacity > threshold


def check_thin_lun(has_patch_flag, sn, cache_object, all_cli_ret_list):
    """
    检查thin LUN
    :param has_patch_flag:
    :param sn:
    :param cache_object:
    :return:
    """
    lun_info = cli_util_cache.get_lun_info_from_config(
        cache_object, LOGGER, sn
    )
    err_thin_lun_list = []

    if not lun_info:
        return True, ""

    thin_lun = [lun for lun in lun_info if lun.get("Type") == "Thin"]
    threshold = Decimal(
        THIN_LUN_CAPACITY_VALUE * MORE_STRICTNESS_THRESHOLD
        if has_patch_flag
        else THIN_LUN_CAPACITY_VALUE
    )

    for lun in thin_lun:
        subscribed_capacity = lun.get("Subscribed Capacity") if lun.get(
            "Subscribed Capacity") is not None else lun.get(
            "Allocated Capacity", '0')
        lun_id = lun.get("ID") if lun.get("ID") is not None else lun.get(
            "Lun Id")
        all_cli_ret_list.append(
            LUN_RET_TEMPLATE.format(lun_id, subscribed_capacity)
        )
        subscribed_capacity = deal_unit(subscribed_capacity)

        if Decimal(subscribed_capacity) > threshold:
            LOGGER.logInfo(
                "subscribed_capacity:{}, threshold:{}".format(
                    subscribed_capacity, threshold
                )
            )

            err_thin_lun_list.append(lun_id)

    if err_thin_lun_list and has_patch_flag:
        error_key = "metadata.risky.thin.lun.has.patch.not.pass"
        return False, get_err_msg(
            LANG, error_key, (", ".join(err_thin_lun_list), has_patch_flag))

    if err_thin_lun_list:
        error_key = "metadata.risky.thin.lun.no.patch.not.pass"
        return False, get_err_msg(
            LANG, error_key, ", ".join(err_thin_lun_list))

    return True, ''


def has_patch(p_version):
    """
    是否存在补丁，影响阈值
    :param p_version:
    :return:
    """

    return META_DATA_HAS_PATCH_VERSION.get(p_version)


def install_patch(p_version, p_patch):
    """
    是否安装了补丁，如果安装了，检查通过
    :param p_version:
    :param p_patch:
    :return:
    """

    return has_patch(
        p_version) and p_patch >= META_DATA_HAS_PATCH_VERSION.get(p_version)
