# -*- coding: UTF-8 -*-
import re

import cliUtil
import common
import common_cache
import risk_version_config
import common_utils

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    """
    业务IP是否和内部IP冲突检查
    :param cli:
    :return:
    """
    flag, ret_list, err_msg = check_service_ip_conflict_inner(cli)
    return flag, "\n".join(ret_list), err_msg


def check_service_ip_conflict_inner(cli):
    '''
    业务IP是否和内部IP冲突检查业务流程
    :param cli:
    :return:
    '''
    all_cli_ret_list = []
    err_msg = []
    try:
        p_version, cli_ret = common_cache.get_product_version_from_system(
            PY_JAVA_ENV, cli, LOGGER
        )
        all_cli_ret_list.append(cli_ret)

        flag, ver, patch, ret, msg = common_cache.get_version_and_patch_cache(
            PY_JAVA_ENV, cli, LOGGER
        )
        all_cli_ret_list.append(ret)
        LOGGER.logInfo("version is:{}, ver is:{} patch is:{}".format(
            p_version, ver, patch))
        if flag is not True:
            return flag, all_cli_ret_list, msg

        if not is_risk_version(ver, patch):
            return True, all_cli_ret_list, ""
        check_eth_port_risk(all_cli_ret_list, cli, err_msg)

        check_logical_port(all_cli_ret_list, cli, err_msg)
        if err_msg:
            return False, all_cli_ret_list, "\n".join(err_msg)

        return True, all_cli_ret_list, ""
    except common.UnCheckException as e:
        return cliUtil.RESULT_NOCHECK, all_cli_ret_list, e.errorMsg
    except Exception as e:
        LOGGER.logException(e)
        err_msg = common.getMsg(LANG, "query.result.abnormal")
        return cliUtil.RESULT_NOCHECK, all_cli_ret_list, err_msg
    finally:
        flag, ret, err_msg = cliUtil.enterCliModeFromSomeModel(cli, LANG)
        LOGGER.logInfo("enter cli mode ret is : {}".format(ret))
        # 退出失败后为不影响后续检查项重新连接cli
        if not flag:
            common.reConnectionCli(cli, LOGGER)


def check_eth_port_risk(all_cli_ret_list, cli, err_msg):
    # 检查ETH端口IP是否在风险网段
    cmd = "show port general physical_type=ETH"
    risk_ip_segment = \
        check_ip_risk_with_port(
            cmd, cli,
            risk_version_config.RISKY_IP_NETWORK_SEGMENT_LIST,
            all_cli_ret_list)
    if risk_ip_segment:
        display_of_risk_ip = "['%s']" % ("', '".join(risk_ip_segment))
        err_msg.append(common_utils.get_err_msg(LANG,
                                                "etc_service_ip_conflict_"
                                                "inner_not_pass",
                                                display_of_risk_ip))


def check_logical_port(all_cli_ret_list, cli, err_msg):
    """
    检查逻辑端口IP是否存在风险网段
    :param all_cli_ret_list: cli回文集合
    :param cli: cli链接
    :param err_msg: 错误回显
    :return:
    """
    cmd = "show logical_port general"
    risk_ip_segment = \
        check_ip_risk_with_port(
            cmd, cli, risk_version_config.RISKY_IP_NETWORK_SEGMENT_LIST,
            all_cli_ret_list)
    if risk_ip_segment:
        display_of_risk_ip = "['%s']" % ("', '".join(risk_ip_segment))
        err_msg.append(common_utils.get_err_msg(LANG,
                                                "logical_service_ip_"
                                                "conflict_inner_not_pass",
                                                display_of_risk_ip))


def check_ip_risk_with_port(cmd, cli, risky_ip_network_segment_list,
                            all_cli_ret_list):
    '''
    #检查命令回文中是否存在风险网段的ip地址。
    :param cmd:
    :param cli:
    :param risky_ip_network_segment_list:
    :param all_cli_ret_list:
    :return:
    '''
    risk_ip_segment = []
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    all_cli_ret_list.append(cli_ret)
    if flag is not True:
        raise common.UnCheckException(err_msg, cli_ret)
    pattern = r"\b(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\b"
    ip_list = re.findall(pattern, cli_ret)
    LOGGER.logInfo("Get ip list is [%s]" % ip_list)
    for ip in ip_list:
        ip_re_match = re.match("(\d+\.\d+)", ip)
        if ip_re_match:
            ip_first_two_segment = ip_re_match.group(1)
            if ip_first_two_segment in risky_ip_network_segment_list:
                risk_ip_segment.append("%s.0.0" % ip_first_two_segment)
    return list(set(risk_ip_segment))


def is_risk_version(p_version, p_patch):
    """
    检查是否是风险版本
    若系统软件版本为V500R007C60SPC300且安装V500R007C60SPH302或更高版本补丁，则检查通过
    若系统软件版本为V500R007C60SPC100且安装V500R007C60SPH101或更高版本补丁，则检查通过，否则继续检查；
    :param p_version: 版本
    :param p_patch: 补丁
    :return: False: 非风险版本，True: 风险版本
    """
    patch_dict = {
        "V500R007C60SPC300": 302,
        "V500R007C60SPC100": 101,
    }
    if p_version not in patch_dict:
        return True

    reg = re.compile(r"SPH(\d+)")
    match_res = reg.search(p_patch)
    if match_res:
        p_patch_num = match_res.group(1)
        if int(p_patch_num) >= patch_dict.get(p_version):
            return False
    return True
