# -*- coding: UTF-8 -*-

import os
import traceback

import java.lang.Exception as JException
from defusedxml import ElementTree as ET

import common
import cliUtil

#硬件配置文件路径
HWConfigFile = r"cfg/hardwareConfig.xml"

#光模块：BOM编码和厂商对应关系
BOM_ModelMap = {
"34060365":("FTLF8524P3BNL-HW",
            "PLRXPLVESG4HWN",
            "LTD8542-BE+",
            "PLRXPLVESG462N"),
"3406G000":("AFBR-57D9AMZ",),
"34060521":("AFBR-57D7APZ",
            "AFBR-57D7APZ-HW1",
            "AFBR-57D9AMZ-HW1",
            "PLRXPLVCSH4HW",
            "FTLF8528P3BCV"),
"34060494":("TPP1XEDSRCZAB00",
            "LTF8502-BC+",
            "PLRXPLSCS43HW",
            "AFBR-703SDZ-HW1",
            "FTLX8571D3BCL-HW",
            "TRS2000EN-S003"),
"34060779":("FTLF8529P3BCV",),
"3406G001":("FCBG410QB1C15",),
"34060662-001":("JQPR04SRAB1HW",
                "LTA8513-PC+",
                "FTL410QE3C-HU",
                "TR-QQ85S-N00",),
"34060874":("1064155015",
            "7120-028-01")
}


def _parseElecLabelInfo(cliRet):
    '''
    @summary: 解析硬件的电子标签信息
    @param cliRet: 硬件的信息回文
    @return: elecLabelDict：硬件电子标签的字典
    '''

    elecLabelDict = {}
    if "Electronic Label" not in cliRet:
        return elecLabelDict
    
    elecLableStart = False
    for line in cliRet.splitlines():
        if "Electronic Label" in line:
            elecLableStart = True
            continue
        if elecLableStart:
            field = line.split("=")
            if len(field) == 2:
                elecLabelDict[field[0].strip()] = field[1].strip()
    
    return elecLabelDict

def _ParseconfigFile(filePath=HWConfigFile):
    '''
    @summary: 解析硬件的配置文件
    @param filePath: 硬件的配置文件路径
    @return: itemConfigDict：硬件配置信息数据（字典结构，key为硬件类型，value也为字典）
    '''
    
    itemConfigDict = {}
    scriptCurPath = os.path.split(os.path.realpath(__file__))[0]
    root = ET.parse(os.path.join(scriptCurPath, filePath))
    
    itemNodeList = root.getiterator("item")
    for itemNode in itemNodeList:
        itemType = itemNode.attrib["type"]
        itemConfig = {}
        ConfigList = itemNode.getchildren()
        for config in ConfigList:
            itemConfig[config.tag] = config.text if config.text else ""
        #保存单个硬件信息
        itemConfigDict[itemType] = itemConfig

    return itemConfigDict

def _ifConfigValid(hwConfig):
    '''
    @summary: 检查硬件配置是否有效
    @param hwConfig: 硬件的配置信息
    @return: True/False
    '''

    if not hwConfig:
        return False
    keyConfig = hwConfig.get("key")
    needParam = hwConfig.get("needParam")
    checkCmd = hwConfig.get("cmd")
    cmdRetFormat = hwConfig.get("cmdRetFormat")
    itemValue = hwConfig.get("value")
    
    #检查配置合规性
    if keyConfig not in ("SN", "BOM") \
            or needParam not in ("True", "False")\
            or cmdRetFormat not in ("Horizontal", "Vertical"):
        return False
    elif not(checkCmd and itemValue):
        return False

    return True

def _switchSN2BomCode(SerialNumber):
    '''
    @summary: 从SN号中获取BOM编码
    @param SerialNumber: 硬件的SN号
    @return: BOM编码
    '''

    BomCode = ""
    #（1）以21开头的20位数字或字母的SN中，第3-10位代表这个物料对应物料种类的BOM编码。
    #以210305G06M10DC000253为例，这种控制器的BOM编码是0305G06M；
    if SerialNumber.startswith("21") and len(SerialNumber) == 20:
        BomCode = SerialNumber[2:10]
    #（2）以02或03开头的16位数字或字母，第1-6位前面加上03凑成8位数字或字母，代表BOM编码。
    #以02G258D0BB001344为例，它代表一个级联板唯一存在，BOM编码为0302G258；
    elif (SerialNumber.startswith("02") or SerialNumber.startswith("03"))\
        and len(SerialNumber) == 16:
        BomCode = "03" + SerialNumber[0:6]

    return BomCode

def getFibreModuleBOMByModel(Model):
    '''
    @summary: 通过光模块的BOM编码信息
    @param Model：光模块型号
    @return: BOM编码
    '''

    for bom,modelList in BOM_ModelMap.items():
        if Model in modelList:
            return bom
    return ""      

def getCtrlBOMCode(ctrlInfo):
    '''
    @summary: 通过控制器的BOM编码信息
    @param ctrlInfo：控制器信息
    @return: BOM编码
    '''

    elecLabelDict = _parseElecLabelInfo(ctrlInfo)
    return elecLabelDict.get("Item", "")

def getHardwareBomCode(type, ssh, ID=""):
    '''
    @summary: 通过命令获取硬件的BOM编码信息（不包含光模块、控制器）
    @param type：硬件类型，ssh：命令接口， ID：所要查询的硬件ID（默认为空）
    @return: BOM编码
    '''

    cliRet = ""
    if isinstance(type, tuple):
        itemConfigDict = _parse_config_file_with_tuple(py_java_env=type[1], logger=type[2])
        hwConfig = itemConfigDict.get(type[0])
    else:
        itemConfigDict = _ParseconfigFile()
        hwConfig = itemConfigDict.get(type)
    if not _ifConfigValid(hwConfig):
        return (cliRet, "")
        
    #获取详细配置信息
    keyConfig = hwConfig.get("key")
    needParam = hwConfig.get("needParam")
    checkCmd = hwConfig.get("cmd")
    cmdRetFormat = hwConfig.get("cmdRetFormat")
    itemValue = hwConfig.get("value")
    itemSubValue = hwConfig.get("subValue")

    #查询详细信息
    if needParam == "True":
        checkCmd = checkCmd + ID
    cliRet = ssh.execCmd(checkCmd)
    
    #电子标签需要特殊处理
    if itemValue == "Electronic Label":
        elecLabelDict = _parseElecLabelInfo(cliRet)
        return (cliRet, elecLabelDict.get("Item", ""))
    
    #不同的回文格式调用不同的解析接口
    if cmdRetFormat == "Horizontal":
        cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    else:
        cliRetLinesList = cliUtil.getVerticalCliRet(cliRet)

    referenceValue = ""
    subValue = ''
    for cliRetLines in cliRetLinesList:
        referenceValue = cliRetLines.get(itemValue, "")
        subValue = cliRetLines.get(itemSubValue)
        if referenceValue and keyConfig == "SN" and isBomInSn(referenceValue):
            break

    if not isBomInSn(referenceValue) and subValue:
        return cliRet, subValue

    if not referenceValue:
        return (cliRet, "")
    
	#如果为SN号，需要转换
    if keyConfig == "SN" and isBomInSn(referenceValue):
        return (cliRet, _switchSN2BomCode(referenceValue))
    elif keyConfig == "SN":
        return cliRet, subValue
    else:
        return (cliRet, referenceValue)
	

def isBomInSn(serialNumber):
    """检查一个Serial Number 是否包含BOM编码。
      （1）以21开头的20位数字或字母的SN中，第3-10位代表这个物料对应物料种类的BOM编码。
    #以210305G06M10DC000253为例，这种控制器的BOM编码是0305G06M；
      （2）以02或03开头的16位数字或字母，第1-6位前面加上03凑成8位数字或字母，代表BOM编码。
    #以02G258D0BB001344为例，它代表一个级联板唯一存在，BOM编码为0302G258；

    :param serialNumber:Serial Number
    :return: True or False
    """
    def is20ByteSnWithBom(sn):
        return sn.startswith("21") and len(sn) == 20

    def is16ByteSnWithBom(sn):
        return (sn.startswith("02") or sn.startswith("03")) and len(sn) == 16

    return is20ByteSnWithBom(serialNumber) or is16ByteSnWithBom(serialNumber)


def _parse_config_file_with_tuple(py_java_env, logger):
    '''
    @summary: 解析硬件的配置文件
    @param filePath: 硬件的配置文件路径
    @param py_java_env: 上下文
    @return: itemConfigDict：硬件配置信息数据（字典结构，key为硬件类型，value也为字典）
    '''
    file_read_lock = py_java_env.get("fileReadLock")
    obj_py = py_java_env.get("objectForPy")
    file_read_lock.lock()
    try:
        if obj_py.get("hardware_config_xml_parser_res"):
            logger.logInfo("xml parse res from memory!")
            return obj_py.get("hardware_config_xml_parser_res")
        item_config_dict = _ParseconfigFile()
        logger.logInfo("save xml parse res to memory!")
        obj_py.put("hardware_config_xml_parser_res", item_config_dict)
        return item_config_dict
    except (JException, Exception):
        logger.logError("parse xml error:{}".format(traceback.format_exc()))
        raise common.UnCheckException("parse xml error", "")
    finally:
        file_read_lock.unlock()
