# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import re
import codecs
import copy
import time
import locale


class SesLogParser(object):
    def __init__(self, logger):
        self.logger = logger
        self.exp_ses_info = []

    def parse(self, ctrl_name, log_ses_files):
        """
        从message文件中解析dq信息
        :param ctrl_name: 一个控制器
        :param log_ses_files: 一个控制器下所有的ses log文件
        :return:
        """
        self.exp_ses_info = []
        self._set_locale_env()
        for file_name in log_ses_files:
            self.pick_exp_ses_info(self._get_sys_segment(file_name))
        return self.exp_ses_info

    @staticmethod
    def _get_sys_segment(file_name):
        segment_list = []
        with codecs.open(file_name, "r") as f:
            res = f.readlines()
            tmp_list = []
            seg_start = False
            for line in res:
                line = line.strip()
                if not line:
                    continue

                if "----SES (" in line:
                    seg_start = True
                    tmp_list.append(line)
                    continue

                if "===END GET" in line:
                    segment_list.append(copy.copy(tmp_list))
                    tmp_list = []
                    seg_start = False

                if seg_start:
                    tmp_list.append(line)
        return segment_list

    def pick_exp_ses_info(self, segment_list):
        rule_dict = {
            "exp_ses_1": "(?P<errorInfo>Main firmware version.*)",
            "exp_ses_2": "(?P<errorInfo>Interrupt happened: peer board system reset)",
            "exp_ses_3": "Exc type:(?P<errorInfo>.*)",
        }
        title_rule = "\(WWN:(?P<wwn>\w+), Inner ID:\d+, LED ID:(?P<ledId>\w+)\).*----(?P<time>.*)----"
        for line_list in segment_list:
            # 每个ses片段来匹配规则是否命中，可能命中多个用list表示
            self._parse_segment_data(line_list, rule_dict, title_rule)

    def _parse_segment_data(self, line_list, rule_dict, out_put_info_rule):
        """
        每个片段做解析
        :param line_list:
        :param rule_dict:
        :param out_put_info_rule:
        :return:
        """
        title_line = str(line_list[:1])
        for line in line_list[1:]:
            parse_res_list = self._parse_rule(line, rule_dict)
            if not parse_res_list:
                continue
            title_res = re.compile(out_put_info_rule, re.IGNORECASE).search(
                title_line)
            if title_res:
                show_info = title_res.groupdict()
                show_info["time"] = self.format_time(show_info.get("time", ""))
                for parse_res in parse_res_list:
                    parse_res.update(show_info)
                    match_data = parse_res.get("matched_data", {})
                    match_data.update(show_info)
                    parse_res["matched_data"] = match_data
                    parse_res["origin_info"] = "{}\n{}".format(
                        title_line, parse_res.get("origin_info", "")
                    )
                    parse_res["oridinal"] = "{}_{}".format(
                        parse_res.get("err_type"), parse_res.get("time")
                    )
                    parse_res["obj_id"] = "{}_{}".format(
                        parse_res.get("ledId"), parse_res.get("wwn")
                    )
            self.exp_ses_info.extend(parse_res_list)

    @staticmethod
    def _parse_rule(line, rule_dict):
        """
        规则匹配
        :param line:
        :param rule_dict:
        :return:
        """
        parse_res_list = []
        for rule_name, rule in rule_dict.items():
            parse_res_dict = {}
            res = re.compile(rule, re.IGNORECASE).search(line)
            if res:
                parse_res_dict["err_type"] = rule_name
                parse_res_dict["matched_data"] = res.groupdict()
                parse_res_dict["origin_info"] = line
                parse_res_list.append(parse_res_dict)

        return parse_res_list

    def _set_locale_env(self):
        try:
            # 此操作保证中文环境日志日期正常解析
            locale.setlocale(locale.LC_ALL, "en.GBK")
        except locale.Error:
            self.logger.error("not support set local")

    @staticmethod
    def format_time(time_str):
        """
        格式化时间：传入的Sun Jun   6 13:11:32 2021->2021-06-06 13:11:32
        :param time_str:
        :return:
        """
        try:
            return time.strftime(
                "%Y-%m-%d %H:%M:%S",
                time.strptime(time_str, "%a %b %d %H:%M:%S %Y"),
            )
        except Exception:
            return time_str
