# -*- coding: UTF-8 -*-
import re
import common
import cliUtil
from common_utils import get_err_msg
from cbb.frame.cli import cliUtil as cbb_cliUtil

# 版本和补丁需求表
REQUIRE_VERSION_AND_PATCH = {
    "6.0.0": "",
    "6.0.1": "6.0.1.SPH16",
    "6.1.0": "6.1.0.SPH7"
}

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    """
    后台任务风险检查
    """
    check_disk_domain = CheckDiskDomainMemory(cli, LANG, LOGGER)
    flag, err_msg = check_disk_domain.execute_check()
    return flag, "\n".join(check_disk_domain.cli_ret_list), err_msg


class CheckDiskDomainMemory:
    def __init__(self, cli, lang, logger):
        self.cli = cli
        self.lang = lang
        self.logger = logger
        self.cli_ret_list = []
        self.product_version = ""
        self.patch_version = ""

    def execute_check(self):
        try:
            # 检查是否为风险版本
            if not self.is_risk_version():
                return True, ""

            # 检查 Cache Capacity 是否足够
            if self.has_enough_cache_capacity():
                return True, ""

            # 检查硬盘域容量使用率
            return self.check_disk_domain_usage()

        except common.UnCheckException as e:
            self.logger.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception as e:
            self.logger.logError(str(e))
            err_msg = common.getMsg(self.lang, "query.result.abnormal")
            return cliUtil.RESULT_NOCHECK, err_msg

    def is_risk_version(self):
        """
        判断风险版本，6.0.0所有版本，6.0.1 SPH16之前的所有版本，6.1.0 SPH7之前所有版本
        :return:
        """
        flag, self.product_version, self.patch_version, cli_ret = \
            cbb_cliUtil.get_system_version_with_ret(self.cli, self.lang)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            err_msg = common.getMsg(self.lang,
                                    "cannot.get.product.version.info")
            raise common.UnCheckException(err_msg, "")

        if not self.patch_version:
            self.patch_version = "--"
            return True
        patch_num = self.get_patch_num(self.patch_version)
        if not patch_num:
            return True

        require_patch = REQUIRE_VERSION_AND_PATCH.get(self.product_version)
        if not require_patch:
            return True
        require_patch_num = self.get_patch_num(require_patch)
        return patch_num < require_patch_num

    @staticmethod
    def get_patch_num(patch_version):
        """
        获取补丁版本的数字
        :return:
        """
        patch = re.search(r"SPH(\d+)", patch_version)
        if not patch:
            return ""
        return int(patch.group(1))

    def has_enough_cache_capacity(self):
        """
        判断 Cache Capacity 是否大于256.00GB
        :return:
        """

        flag, cache_capacity, cli_ret, err_msg = \
            common.get_device_cache_capacity_with_ret(
                self.cli, self.lang, self.logger)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            raise common.UnCheckException(err_msg, cli_ret)

        return cliUtil.transfer_capacity(cache_capacity, "GB") > 256

    def check_disk_domain_usage(self):
        """
        检查硬盘域占用率是否达到70%
        :return:
        """
        cmd = "show disk_domain general |filterColumn include columnList=ID," \
              "Total\sCapacity,Free\sCapacity,Hot\sSpare\sCapacity"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            raise common.UnCheckException(err_msg, cli_ret)
        self.logger.logInfo("cli_ret:{}".format(cli_ret))
        info_dict_list = cliUtil.getHorizontalCliRet(cli_ret)
        self.logger.logInfo("info_dict_list:{}".format(info_dict_list))

        for info_dict in info_dict_list:
            total_capacity = info_dict.get("Total Capacity", "0.00GB")
            free_capacity = info_dict.get("Free Capacity", "0.00GB")
            hot_spare_capacity = info_dict.get("Hot Spare Capacity", "0.00GB")
            if not self.is_risk_domain_capacity_usage(
                    total_capacity, free_capacity, hot_spare_capacity):
                continue
            return False, self.get_err_info()
        return True, ""

    @staticmethod
    def is_risk_domain_capacity_usage(
            total_capacity, free_capacity, hot_spare_capacity):
        """
        判断硬盘域使用率是否小于70%。
        已用容量 usedCap = Total Capacity - Free Capacity - Hot Spare Capacity。
        usedCap >= (Total Capacity - Hot Spare Capacity) * 70%，如果大于等于70%，
        则检查不通过，否则检查通过。
        :return:
        """
        # 将单位统一换算为GB
        total_capacity = cliUtil.transfer_capacity(total_capacity, "GB")
        free_capacity = cliUtil.transfer_capacity(free_capacity, "GB")
        hot_spare_capacity = cliUtil.transfer_capacity(hot_spare_capacity, "GB")

        used_capacity = total_capacity - free_capacity - hot_spare_capacity

        return used_capacity >= ((total_capacity - hot_spare_capacity) * 0.7)

    def get_err_info(self):
        """
        获取错误信息
        :return:
        """
        require_patch = REQUIRE_VERSION_AND_PATCH.get(self.product_version)
        if not require_patch:
            return get_err_msg(self.lang, "capacity.usage.error.for.6.0.0")

        return get_err_msg(
            self.lang, "software.io.delay.risk.patch.not.pass",
            (self.product_version, self.patch_version, require_patch))
