# -*- coding: UTF-8 -*-
import re

import common
import cliUtil
import expandconfig
from common_utils import get_err_msg
from cbb.frame.cli import cliUtil as cbb_cliUtil
from risk_version_config import HIGH_IDLE_DISK_NUM_VERSION


PY_JAVA_ENV = py_java_env
LOGGER = PY_LOGGER
MIN_HIGH_IDLE_DISK_NUM = 25
THRESHOLD_NUM = 5.0
NEED_UPGRADE_VERSION = "6.0.0"
UPGRADE_TARGET_VERSION = "6.1.2"

def execute(cli):
    """
    高空闲率盘数检查
    :param cli: cli连接
    :return:
    """
    check_item = CheckHighIdleDiskNumExpantion(cli)
    flag = check_item.check()
    return flag, "\n".join(check_item.cli_ret_list), check_item.err_msg


class CheckHighIdleDiskNumExpantion:
    def __init__(self, cli):
        self.cli = cli
        self.lang = common.getLang(PY_JAVA_ENV)
        self.logger = common.getLogger(LOGGER, __file__)
        self.software_version = ""
        self.hot_patch_version = ""
        self.cli_ret_list = []
        self.err_msg = ""
        # 获取扩容配置数据
        self.exp_domain_disk_info = self._init_exp_domain_disk()

    @staticmethod
    def _init_exp_domain_disk():
        domain_disk_info = {}
        exp_disk_list = expandconfig.ExpandConfig(
            PY_JAVA_ENV
        ).getAllExpDiskList()
        for disk_info in exp_disk_list:
            disk_type = disk_info.getDiskType()
            disk_num = disk_info.getDiskNum()
            disk_domain_id = disk_info.getDiskDomainId()
            disk_number_dict = domain_disk_info.get(disk_domain_id, {})
            disk_number_dict[disk_type] = (
                    disk_number_dict.get(disk_type, 0) + disk_num
            )
            domain_disk_info[disk_domain_id] = disk_number_dict
        return domain_disk_info

    def check(self):
        try:
            # 检查是否为风险版本和补丁
            if not self.is_risk_version_and_patch():
                return True

            # 检查BOM编码
            return self.check_high_idle_disk()

        except common.UnCheckException as e:
            self.logger.logError(str(e))
            self.err_msg = e.errorMsg
            return cliUtil.RESULT_NOCHECK
        except Exception as e:
            self.logger.logError(str(e))
            self.err_msg = common.getMsg(self.lang, "query.result.abnormal")
            return cliUtil.RESULT_NOCHECK

    def is_risk_version_and_patch(self):
        """
        判断是否为风险版本
        :return:
        """
        (
            flag,
            self.software_version,
            self.hot_patch_version,
            cli_ret,
        ) = cbb_cliUtil.get_system_version_with_ret(self.cli, self.lang)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            err_msg = common.getMsg(
                self.lang, "cannot.get.product.version.info"
            )
            raise common.UnCheckException(err_msg, "")

        self.logger.logInfo(
            "version:{}, patch:{}".format(
                self.software_version, self.hot_patch_version
            )
        )
        # 不是风险版本，检查通过
        if self.software_version not in HIGH_IDLE_DISK_NUM_VERSION.keys():
            return False
        # 风险版本，不存在补丁,需要继续检查
        if not self.hot_patch_version:
            self.hot_patch_version = "--"
            return True

        patch_num = self.get_patch_num(self.hot_patch_version)
        if not patch_num:
            return True

        require_patch = HIGH_IDLE_DISK_NUM_VERSION.get(self.software_version)
        if not require_patch:
            return True
        require_patch_num = self.get_patch_num(require_patch)
        return patch_num < require_patch_num

    @staticmethod
    def get_patch_num(patch_version):
        """
        获取补丁版本的数字
        :return:
        """
        patch = re.search(r"SPH(\d+)", patch_version)
        if not patch:
            return 0
        return int(patch.group(1))

    def check_high_idle_disk(self):
        """
        检查高空闲率的盘数
        :return:
        """
        domain_ids = self.get_domain_id()
        self.logger.logInfo("domain_ids:{}".format(domain_ids))
        if not domain_ids:
            return True
        for domain_id in domain_ids:
            current_domain_exp_disk_num_dict = self.exp_domain_disk_info.get(
                domain_id, {}
            )
            if self.has_too_many_high_idle_disk(
                domain_id, current_domain_exp_disk_num_dict
            ):
                self.set_srr_msg()
                return False

        return True

    def _get_disk_location_info(self):
        flag, ret, disk_list, err_msg = common.getDiskList(self.cli, self.lang)
        if flag is not True:
            raise common.UnCheckException(err_msg, ret)
        self.cli_ret_list.append(ret)
        disk_type_location_dict = {}
        for disk_info in disk_list:
            disk_type = disk_info.get("Type")
            disk_id_list = disk_type_location_dict.get(disk_type, [])
            disk_id_list.append(disk_info.get("ID"))
            disk_type_location_dict[disk_type] = disk_id_list
        return disk_type_location_dict

    def _get_disk_type_mgr_id_info(self):
        disk_type_mgr_id_dict = {}

        # 获取硬盘location和类型的映射
        disk_type_location_dict = self._get_disk_location_info()

        # 获取mgr disk id 和 disk location映射关系
        mgr_disk_info_dict = self._get_mgr_disk_info()
        self.logger.logInfo(
            "disk_type_location_dict:{}, mgr_disk_info_dict:{}".format(
                disk_type_location_dict, mgr_disk_info_dict
            )
        )
        # 通过location转换为数字的disk id
        for user_disk_id, mgr_disk_id in mgr_disk_info_dict.items():
            for disk_type, disk_id_list in disk_type_location_dict.items():
                if user_disk_id not in disk_id_list:
                    continue
                mgr_id_list = disk_type_mgr_id_dict.get(disk_type, [])
                mgr_id_list.append(mgr_disk_id)
                disk_type_mgr_id_dict[disk_type] = mgr_id_list
        return disk_type_mgr_id_dict

    def _get_mgr_disk_info(self):
        mgr_disk_info_dict = {}
        cmd = "show disk_mgr disk_list"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInDeveloperMode(
            self.cli, cmd, True, self.lang
        )
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            raise common.UnCheckException(err_msg, cli_ret)
        if cliUtil.queryResultWithNoRecord(cli_ret):
            return mgr_disk_info_dict

        disk_info_list = cliUtil.getHorizontalCliRet(cli_ret)
        for disk_info in disk_info_list:
            user_disk_id = disk_info.get("User Disk ID")
            disk_id = disk_info.get("Disk ID")
            mgr_disk_info_dict[user_disk_id] = disk_id
        return mgr_disk_info_dict

    def get_domain_id(self):
        """
        获取硬盘域ID
        :return:
        """
        disk_domain_id_list = []
        cmd = "show disk_domain general"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            err_msg = common.getMsg(self.lang, "cannot.get.disk.domain.info")
            raise common.UnCheckException(err_msg, "")
        if cliUtil.queryResultWithNoRecord(cli_ret):
            return disk_domain_id_list
        disk_domain_info_list = cliUtil.getHorizontalCliRet(cli_ret)

        for disk_domain_info in disk_domain_info_list:
            disk_domain_id = disk_domain_info.get("ID")
            disk_domain_id_list.append(disk_domain_id)
        return disk_domain_id_list

    def _get_block_mgr_disk_info(self, domain_id):
        ctrl_id_list = self.get_ctrl_id_list()
        self.logger.logInfo("ctrl_id_list:{}".format(ctrl_id_list))
        for ctrl_id in ctrl_id_list:
            cmd = (
                "show block_manager bitmap disk_pool_id={} space_type=data "
                "controller={}".format(domain_id, ctrl_id)
            )
            flag, cli_ret, err_msg = cliUtil.excuteCmdInDeveloperMode(
                self.cli, cmd, True, self.lang
            )
            self.cli_ret_list.append(cli_ret)
            if flag is not True:
                self.logger.logInfo(
                    "The current controller does not "
                    "return correct echo:{}".format(ctrl_id)
                )
                continue
            return cliUtil.getHorizontalCliRet(cli_ret)

        # 所有控制器都不支持该命令
        err_msg = common.getMsg(self.lang, "cannot.get.diskindomain.info")
        raise common.UnCheckException(err_msg, "")

    def has_too_many_high_idle_disk(
        self, domain_id, current_domain_exp_disk_num_dict
    ):
        """
        判断高空闲率的盘数量是否过多
        :return:
        """
        disk_type_mgr_id_dict = self._get_disk_type_mgr_id_info()
        dict_list = self._get_block_mgr_disk_info(domain_id)
        self.logger.logInfo(
            "dict_list:{}, disk_type_mgr_id_dict:{}, domain_id:{}, exp_disk_num_dict:{}".format(
                dict_list,
                disk_type_mgr_id_dict,
                domain_id,
                current_domain_exp_disk_num_dict,
            )
        )
        for disk_type, disk_mgr_id_list in disk_type_mgr_id_dict.items():
            current_domain_exp_disk_num = current_domain_exp_disk_num_dict.get(
                disk_type, 0
            )
            if current_domain_exp_disk_num > MIN_HIGH_IDLE_DISK_NUM:
                return True
            high_idle_disk_num = self.get_high_idle_disk_num(
                dict_list, disk_mgr_id_list
            )
            if (
                high_idle_disk_num + current_domain_exp_disk_num
                >= MIN_HIGH_IDLE_DISK_NUM
            ):
                self.logger.logInfo(
                    "disk type:{}, the disk num is > 25, high_idle_disk_num:{}, "
                    "current_domain_exp_disk_num:{}".format(
                        disk_type,
                        high_idle_disk_num,
                        current_domain_exp_disk_num,
                    )
                )
                return True
        return False

    def get_high_idle_disk_num(self, dict_list, disk_mgr_id_list):
        """
        获取高空闲率盘数量
        :return:
        """
        # 获取所有的空闲率
        free_percent_list = []

        for dict_info in dict_list:
            if dict_info.get("Disk ID") not in disk_mgr_id_list:
                continue
            free_percent = dict_info.get("Free Percent(%)", "").strip()

            if not free_percent or free_percent == "--":
                err_msg = common.getMsg(
                    self.lang, "cannot.get.diskindomain.info"
                )
                raise common.UnCheckException(err_msg, "")

            free_percent_list.append(float(free_percent))
        self.logger.logInfo("free_percent_list:{}".format(free_percent_list))
        # 最小空闲率
        min_free_percent = min(free_percent_list)
        # 获取高空闲率盘数
        high_idle_disk_num = 0
        for free_percent in free_percent_list:
            if free_percent - min_free_percent >= THRESHOLD_NUM:
                high_idle_disk_num += 1
        self.logger.logInfo("high_idle_disk_num:{}".format(high_idle_disk_num))

        return high_idle_disk_num

    def get_ctrl_id_list(self):
        """
        获取控制器ID列表
        :return:
        """
        (
            flag,
            controller_id_list,
            err_msg,
            cli_ret,
        ) = cliUtil.getControllerIdListWithRet(self.cli, self.lang)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            err_msg = common.getMsg(self.lang, "cannot.get.controller.info")
            raise common.UnCheckException(err_msg, "")

        return controller_id_list

    def set_srr_msg(self):
        """
        设置错误信息
        """
        if self.software_version == NEED_UPGRADE_VERSION:
            self.err_msg = get_err_msg(
                self.lang,
                "storagepool.param.check.not.pass.upgrade.version",
                (self.software_version, UPGRADE_TARGET_VERSION),
            )
        else:
            self.err_msg = get_err_msg(
                self.lang,
                "software.io.delay.risk.patch.not.pass",
                (
                    self.software_version,
                    self.hot_patch_version,
                    HIGH_IDLE_DISK_NUM_VERSION.get(self.software_version),
                ),
            )
