# -*- coding:utf-8 -*-
import os
import cliUtil
import common
import traceback
from common import UnCheckException

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ALL_CLI_RET = ''

# 条件小于
SMALLER_THAND = "smaller than"

# 条件等于
EQUAL = "equal"

# 硬件配置文件路径
RISK_SN_FILE = os.path.join('cfg', 'risk_sn.txt')

FW_NOT_PASS_MODE = {
    # 外购盘
    'LT1800RO': 'H507', 'LT0900RO': 'H507', 'MG06ACA10TE': '080D', 'MG06ACA10TA': '080D',
    'MG06ACA800A': '080D', 'MG06ACA800E': '080D', 'HSSD-D7N23AL1T9N': '3308',
    #自研盘
    'HSSD-D7N23AL960N': '3308', 'HSSD-D7T23AL3T8N': '3308', 'HSSD-D7N23AL3T8N': '3308', 'HSSD-D7Q23AV3T2N': '3308',
    'HSSD-D7P23AL960N': '3308', 'HSSD-D7P23AL480N': '3308', 'HSSD-D7P23AL1T9N': '3308', 'HSSD-D7N23AL3T8V': '3308',
    'HSSD-D7223AL960N': '8308', 'HSSD-D7223AL1T9N': '8308', 'HSSD-D7223AL3T8N': '8308', 'HSSD-D7223AL7T6N': '8308',
    'HSSD-D7223AL15TN': '8308', 'HSSD-D7523AL960N': '8308', 'HSSD-D7523AL1T9N': '8308', 'HSSD-D7523AL3T8N': '8308',
    'HSSD-D7523AL7T6N': '8308', 'HSSD-D7223AL3T8E': '8308', 'HSSD-D7223AL7T6E': '8308', 'HSSD-D7523AV3T2N': '8308',
    'HSSD-D7G23AN7T6N': '8308', 'HSSD-D7B23AL3T8N': '8308', 'HSSD-D7923AN7T6N': '8308'
}

FW_NOT_PASS_AND_MATCH_RISK_SN_MODE = {
    'ST600MM0009': 'N0F5',
    'ST1200MM0129': 'C0G4',
    'ST1800MM0129': 'C0G4',
    'ST2400MM0129': 'C0G4'
}

FW_NOT_PASS_VERSION = 'H506'

FW_MODEL_AAG4_LIST = [
    'HUC101818CS4200', 'HUC101812CS4200'
]

FW_MODEL_A8G0_LIST = [
    'HUS156060VLS600'
]


# 检查特定硬盘 FW_MODEL_XX_LIST的硬盘的固件版本，fw 就是指定的固件版本。needCheckNas: 是否需要检查nas。
# condition:判断条件，如果需要新增。可在方法getConditionValue 中增加操作符。
DISK_MODEL_AND_FW_MAP = {
    tuple(FW_MODEL_AAG4_LIST): {"fw": 'AAG4', "needCheckNas": False,
                                "condition": SMALLER_THAND},
    tuple(FW_MODEL_A8G0_LIST): {"fw": 'A8G0', "needCheckNas": False,
                                "condition": SMALLER_THAND}
}

ALL_LOWER_MODEL_LIST = []
ALL_LOWER_MODEL_LIST.extend(FW_MODEL_AAG4_LIST)
ALL_LOWER_MODEL_LIST.extend(FW_MODEL_A8G0_LIST)


def execute(cli):
    """
    检查方法
        步骤1 以admin用户登录CLI；
        步骤2 执行show disk general |filterColumn include columnList=ID,Manufacturer,Model,
        Firmware\sVersion,Serial\sNumber命令，查询硬盘信息；若Model值满足以HSSD开头则继续执行第3步；
        步骤3 执行show file_system general命令，查询是否存在NAS文件系统。

    检查标准
        1.如果硬盘的检查标准满足以下条件之一，则检查结果为不通过。
        1）步骤2中Model字段值为LT1800RO或LT0900RO，且硬盘Firmware Version值小于H507。
        2）步骤2中Model字段值为MG06ACA10TE，MG06ACA10TA，MG06ACA800A，MG06ACA800E，且Firmware Version字段值小于080D。
        3）步骤2中Model字段值为ST600MM0009，Firmware Version字段值小于N0F5且匹配涉及范围。
        4）步骤2中Model字段值为ST1200MM0129，ST1800MM0129，ST2400MM0129，Firmware Version字段值小于低于C0G4且匹配涉及范围。
        2.如果硬盘的检查标准满足以下条件之一，则检查结果为建议优化；否则检查结果为通过。
        1) 步骤2中Model字段值为HUC101818CS4200或者HUC101812CS4200，且Firmware Version字段值小于AAG4。
        2) 步骤2中Model字段值为HUS156060VLS600，且Firmware Version字段值小于A8G0。
        3) 步骤2中Model字段值为HSSD-D7223AL960N，HSSD-D7223AL1T9N，HSSD-D7223AL3T8N，HSSD-D7223AL7T6N，HSSD-D7223AL15TN，
        HSSD-D7523AL960N，HSSD-D7523AL1T9N，HSSD-D7523AL3T8N，HSSD-D7523AL7T6N，HSSD-D7223AL3T8E，HSSD-D7223AL7T6E，
        HSSD-D7523AV3T2N，HSSD-D7G23AN7T6N，HSSD-D7B23AL3T8N，HSSD-D7923AN7T6N，且Firmware Version字段值小于8308。
        4) 步骤2中Model字段值为HSSD-D7N23AL1T9N，HSSD-D7N23AL960N，HSSD-D7T23AL3T8N，HSSD-D7N23AL3T8N，HSSD-D7Q23AV3T2N，
        HSSD-D7P23AL960N，HSSD-D7P23AL480N，HSSD-D7P23AL1T9N，HSSD-D7N23AL3T8V，且Firmware Version字段值小于3308。


    修复建议
        如果该检查结果为建议优化，则说明硬盘固件版本过低，为了提升硬盘健壮性，请联系技术支持工程师协助将硬盘固件版本升级至合适版本。
    :param cli: cli
    :return: 检查结果
    """
    global ALL_CLI_RET

    EXIST_NAS_FLAG_DICT = {}
    err_msg = ''
    try:
        disk_info_dcts = get_disk_info(cli)
        LOGGER.logInfo("the disk_info_dcts is {}".format(disk_info_dcts))
        if len(disk_info_dcts) == 0:
            return True, ALL_CLI_RET, err_msg

        needUpgradeDiskList = []
        not_pass_disk_list = list()
        not_pass_err_meg = ''
        suggest_err_meg = ''
        risk_sns = get_risk_sn()
        for disk_info_dct in disk_info_dcts:
            disk_id = disk_info_dct.get('ID', '')
            model = disk_info_dct.get('Model', '')
            fwVersion = disk_info_dct.get('Firmware Version', '')
            if not fwVersion or fwVersion == "--":
                continue
            not_pass_err_meg += set_not_pass_info_not_with_sn(disk_info_dct, not_pass_disk_list)
            not_pass_err_meg += set_not_pass_info_with_sn(disk_info_dct, not_pass_disk_list, risk_sns)
            # 如果不在白名单中则继续其他检查。
            if model not in ALL_LOWER_MODEL_LIST:
                LOGGER.logInfo("model not in risk model list:%s" % model)
                continue

            for modelList in DISK_MODEL_AND_FW_MAP:
                if model in modelList:
                    standFW = DISK_MODEL_AND_FW_MAP[modelList].get("fw")
                    needCheckNas = DISK_MODEL_AND_FW_MAP[modelList].get(
                        "needCheckNas")
                    condition = DISK_MODEL_AND_FW_MAP[modelList].get(
                        "condition")
                    LOGGER.logInfo(
                        "standFW:%s, needCheckNas:%s, condition:%s" % (
                            standFW, needCheckNas, condition))

                    # 判断fw是否符合不通过的条件
                    if getConditionValue(fwVersion, condition, standFW):
                        LOGGER.logInfo("risk product and risk fw version.")
                        # 需要检查是否存在NAS文件系统
                        if needCheckNas:
                            exis_nas_flag = EXIST_NAS_FLAG_DICT.get(
                                "exis_nas_flag")
                            if exis_nas_flag is None:
                                exis_nas_flag = get_file_system_info(cli)
                                EXIST_NAS_FLAG_DICT[
                                    "exis_nas_flag"] = exis_nas_flag

                            if not exis_nas_flag:
                                continue

                            LOGGER.logInfo(
                                "need check nas and is risk version.")
                            needUpgradeDiskList.append(disk_id)
                            # 硬盘[ID:%s]的Model字段值为%s，且Firmware Version
                            # 字段值小于%s，且存在NAS文件系统
                            suggest_err_meg += common.getMsg(LANG,
                                                             'disk.model.'
                                                             'firmware.nas.'
                                                             'risk',
                                                             (disk_id, model,
                                                              standFW))
                        else:
                            errKey = "disk.model.firmware.risk"
                            if condition == EQUAL:
                                errKey = "disk.model.firmware.risk.new"
                            suggest_err_meg += common.getMsg(LANG, errKey, (
                                disk_id, model, standFW))
                            needUpgradeDiskList.append(disk_id)

                    # 如果找到model就break
                    break

        if needUpgradeDiskList or not_pass_disk_list:
            if not_pass_disk_list:
                err_msg += not_pass_err_meg
                err_msg = build_err_meg(not_pass_disk_list, err_msg, True)
            if needUpgradeDiskList:
                err_msg += "\n"
                err_msg += suggest_err_meg
                err_msg = build_err_meg(needUpgradeDiskList, err_msg, False)
            if not_pass_disk_list:
                return False, ALL_CLI_RET, err_msg
            return cliUtil.RESULT_WARNING, ALL_CLI_RET, err_msg

        return True, ALL_CLI_RET, err_msg

    except UnCheckException as unCheckException:
        LOGGER.logError(str(traceback.format_exc()))
        LOGGER.logInfo(
            "UnCheckException, errMsg: %s" % unCheckException.errorMsg)
        if not unCheckException.flag:
            return cliUtil.RESULT_NOCHECK, unCheckException.cliRet, unCheckException.errorMsg
        else:
            return unCheckException.flag, unCheckException.cliRet, unCheckException.errorMsg
    except Exception as exception:
        LOGGER.logException(exception)
        return cliUtil.RESULT_NOCHECK, ALL_CLI_RET, common.getMsg(LANG,
                                                                  "query.result.abnormal")


def set_not_pass_info_not_with_sn(disk_info_dct, not_pass_disk_list):
    disk_id = disk_info_dct.get('ID', '')
    model = disk_info_dct.get('Model', '')
    fw_version = disk_info_dct.get('Firmware Version', '')
    if model in FW_NOT_PASS_MODE and fw_version < FW_NOT_PASS_MODE.get(model):
        not_pass_disk_list.append(disk_id)
        LOGGER.logInfo(
            "not pass disk,ID:{},model:{},fw:{}.".format(disk_id,
                                                         model,
                                                         fw_version))
        return common.getMsg(LANG,
                             'disk.model.firmware.risk',
                             (disk_id, model, FW_NOT_PASS_MODE.get(model)))
    return ""


def set_not_pass_info_with_sn(disk_info_dct, not_pass_disk_list, risk_sns):
    disk_id = disk_info_dct.get('ID', '')
    model = disk_info_dct.get('Model', '')
    fw_version = disk_info_dct.get('Firmware Version', '')
    serial_number = disk_info_dct.get('Serial Number', '')
    if model in FW_NOT_PASS_AND_MATCH_RISK_SN_MODE and \
            fw_version < FW_NOT_PASS_AND_MATCH_RISK_SN_MODE.get(model) and is_match_risk_sn(risk_sns, serial_number):
        not_pass_disk_list.append(disk_id)
        LOGGER.logInfo(
            "not pass disk,ID:{},model:{},fw:{}.".format(disk_id,
                                                         model,
                                                         fw_version))
        return common.getMsg(LANG,
                             'disk.model.firmware.risk',
                             (disk_id, model, FW_NOT_PASS_AND_MATCH_RISK_SN_MODE.get(model)))
    return ""


def is_match_risk_sn(risk_sns, serial_number):
    return serial_number[:8] in risk_sns if len(serial_number) >= 8 else False


def get_risk_sn(file_path=RISK_SN_FILE):
    script_cur_path = os.path.split(os.path.realpath(__file__))[0]
    risk_sn_file = os.path.join(script_cur_path, file_path)
    with open(risk_sn_file, "r") as f:
        lines = f.readlines()
    return [line.strip() for line in lines]


def getConditionValue(valueA, condition, valueB):
    # 等于
    if condition == EQUAL:
        return valueA == valueB

    # 小于
    if condition == SMALLER_THAND:
        return valueA < valueB


def get_file_system_info(cli):
    """
    执行show file_system general命令，查询是否存在NAS文件系统
    :param cli:
    :return:
    """
    global ALL_CLI_RET
    # 执行show file_system general命令，查询是否存在NAS文件系统
    cmd = 'show file_system general'
    LOGGER.logExecCmd(cmd)
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cli_ret)
    # 若设备不支持查询NAS文件系统的命令，则直接返回Fasle
    if not cliUtil.hasCliExecPrivilege(cli_ret):
        return False

    # 如果存在NAS文件系统，则回显中有详细信息，否则命令执行成功，但是回显包含Command executed successfully，
    # 即不存在NAS文件系统
    if cliUtil.queryResultWithNoRecord(cli_ret):
        return False
    return True


def get_disk_info(cli):
    """
    执行show disk general
    |filterColumn include columnList=ID,Manufacturer,Model,Firmware\sVersion,Serial\sNumber命令，查询硬盘信息
    :param cli:
    :return:
    """
    global ALL_CLI_RET
    # 执行命令
    cmd = 'show disk general |filterColumn include columnList=ID,Manufacturer,Model,Firmware\\sVersion,Serial\\sNumber'
    LOGGER.logExecCmd(cmd)
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cli_ret)
    if flag != True:
        LOGGER.logInfo("Failed to get disk info, err_msg:%s" % err_msg)
        raise UnCheckException(common.getMsg(LANG, "cannot.get.info", "disk"),
                               ALL_CLI_RET, flag)

    # 回显包含Command executed successfully
    if cliUtil.queryResultWithNoRecord(cli_ret):
        disk_info_dcts = []
        return disk_info_dcts

    # 调用getHorizontalCliRet函数生成字典，记录硬盘信息
    disk_info_dcts = cliUtil.getHorizontalNostandardCliRet(cli_ret)
    return disk_info_dcts


def get_product_info(cli):
    """
    获取设备的版本和型号
    :param cli:
    :return:
    """
    global ALL_CLI_RET
    _, product_model, product_version, cli_ret = cliUtil.getProductModelAndVersion(
        cli, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cli_ret)
    return product_model, product_version


def build_err_meg(disk_list, err_msg, is_not_pass_flag=False):
    """
    构建错误信息
    :param disk_list:
    :param err_msg:
    :param is_not_pass_flag:
    :return: 返回错误信息
    """
    disk_num = len(set(disk_list))
    need_upgrade_time = disk_num * 2
    need_upgrade_time_hour, need_upgrade_time_min = divmod(need_upgrade_time,
                                                           60)
    title = common.getMsg(LANG, "suggest.operation.key")
    if is_not_pass_flag is True:
        meg = "disk.firmware.not.pass.need.upg.time"
    else:
        meg = "disk.firmware.need.upg.time"
    err_msg = err_msg + title
    if need_upgrade_time_hour != 0:
        err_msg += common.getMsg(LANG, meg,
                                 (disk_num,
                                  "%sH%sMin" % (need_upgrade_time_hour,
                                                need_upgrade_time_min)))
    else:
        err_msg += common.getMsg(LANG, meg,
                                 (disk_num,
                                  "%sMin" % need_upgrade_time_min))
    return err_msg
