# -*- coding: UTF-8 -*-
import cliUtil
import traceback
import common
from common import UnCheckException
import re

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env
allCliRet = ""

def execute(cli):
    """
    @summary: “开局巡检”场景中增加对SAS环路硬盘框级联深度的检查，检查用户的配置是否在推荐的范围内。
    """
    try:
        global allCliRet
        # 获取产品型号和版本
        ret = cliUtil.getProductModelAndVersion(cli, LANG)
        if ret[0] != True: 
            allCliRet = common.joinLines(allCliRet, ret[1])
            return (cliUtil.RESULT_NOCHECK, allCliRet, ret[2])
        
        allCliRet = common.joinLines(allCliRet, ret[3])
        productModel = ret[1]
        productVersion = ret[2]
        
        # 获取硬盘信息
        disksInfo = getDisksInfo(cli)
        if not disksInfo:
            return (True, allCliRet, "")
        
        # 获取所有引擎的sas环路
        enginesSasLoops = getEnginesSasLoops(disksInfo, productModel)
        LOGGER.logInfo("All engines under the loop are %s" % str(enginesSasLoops))
        if not enginesSasLoops:
            # 没有sas环路(盘框一体设备)，无需检查
            LOGGER.logInfo("The system has not sas loop, check pass.")
            return (True, allCliRet, "")
        
        # 获取sas环路的级联深度
        sasLoopsExpansionDepth = getSasLoopsExpansionDepth(disksInfo, productModel)
        LOGGER.logInfo("Depth of all loops are %s" % str(sasLoopsExpansionDepth))
        if not sasLoopsExpansionDepth:
            errMsg = common.getMsg(LANG, "cannot.get.cascading.depth")
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
        
        # 获取sas环路下的硬盘类型
        sasLoopsDiskTypes = getSasLoopsDiskTypes(disksInfo, productModel)
        LOGGER.logInfo("Disk type of all loops are %s" % str(sasLoopsDiskTypes))
        if not sasLoopsDiskTypes:
            errMsg = common.getMsg(LANG, "cannot.get.disk.enclosure")
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
        
        # 获取sas环路下的硬盘框
        sasLoopsEnclosures = getSasLoopsEnclosures(disksInfo, productModel)
        LOGGER.logInfo("Enclosures of all loops are %s" % str(sasLoopsEnclosures))
        if not sasLoopsEnclosures:
            errMsg = common.getMsg(LANG, "cannot.get.disk.enclosure")
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
        
        # 获取所有硬盘框的类型
        enclosuresType, enclosuresInfo = getEnclosuresType(cli)
        LOGGER.logInfo("All enclosures type are %s" % str(enclosuresType))
        if not enclosuresType:
            errMsg = common.getMsg(LANG, "cannot.get.disk.enclosure")
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
        
        # 获取所有硬盘框的高度
        enclosuresHeight = getEnclosuresHeight(enclosuresInfo)
        LOGGER.logInfo("All enclosures height are %s" % str(enclosuresHeight))
        if not enclosuresHeight:
            errMsg = common.getMsg(LANG, "cannot.get.disk.enclosure")
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
        
        # 检查sas环路的级联深度
        errMsg = checkSasLoopExpansionDepth(sasLoopsExpansionDepth, sasLoopsDiskTypes, sasLoopsEnclosures, enclosuresType, enclosuresHeight, productModel, productVersion)
        if errMsg:
            return (cliUtil.RESULT_WARNING, allCliRet, errMsg)
        
        # 获取所有引擎下的SAS端口数量
        engIds = enginesSasLoops.keys()
        engineSasPortNumber = getEngineSasPortNumber(cli, productModel, engIds)
        LOGGER.logInfo("Sas ports of engines are %s." % str(engineSasPortNumber))
        if not engineSasPortNumber:
            return (True, allCliRet, "")
        
        # 检查引擎下级联框是否均衡
        flag, errMsg = checkSasLoopBalance(enginesSasLoops, sasLoopsExpansionDepth, sasLoopsDiskTypes, sasLoopsEnclosures, enclosuresType, engineSasPortNumber, enclosuresInfo)
        return (flag, allCliRet, errMsg)

    except UnCheckException, unCheckException:
        LOGGER.logException(unCheckException)
        return (cliUtil.RESULT_NOCHECK, unCheckException.cliRet, unCheckException.errorMsg)

    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, allCliRet, common.getMsg(LANG, "query.result.abnormal"))


def getDisksInfo(cli):
    """
    @summary: 获取设备上所有硬盘的信息
    @return: 硬盘信息字典组成的列表
    """
    global allCliRet

    cmd = "show disk general"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    allCliRet = common.joinLines(allCliRet, cliRet)
    
    if flag != True:
        errMsg = common.getMsg(LANG, "cannot.get.diskinfo")
        LOGGER.logNoPass("Get disk failed!")
        raise UnCheckException(errMsg, allCliRet)
    
    if cliUtil.queryResultWithNoRecord(cliRet):
        return []
    
    disksInfo = cliUtil.getHorizontalCliRet(cliRet)
    if len(disksInfo) == 0:
        errMsg = common.getMsg(LANG, "cannot.get.diskinfo")
        LOGGER.logNoPass("Parse disk failed!")
        raise UnCheckException(errMsg, allCliRet)
    
    return disksInfo


def getSasLoopByDisk(diskID, productModel):
    """
    @summary:根据硬盘ID的计算该硬盘所在硬盘框所在的环路
    """    
    if re.match("DAE([0-9]|[a-f]|[A-F]){3}", diskID) == None:
        return (None, None)
    
    engineNumber =  int(diskID[3], 16)
    secondNumber = int(diskID[4], 16)
    thirdNumber = int(diskID[5], 16)
    sasLoopNumber = secondNumber
    
    if productModel in ["18500 V3", "18800 V3", "18500F V3", "18800F V3", "18500 V5", "18800 V5", "18500F V5", "18800F V5", "6800 V5", "6800F V5"]:
        sasLoopNumber = (thirdNumber / 4) * 12 + secondNumber
    
    if productModel in ["6800 V3", "6900 V3", "6800F V3"]:
        if thirdNumber >= 8:
            sasLoopNumber = secondNumber + 16
     
    return (engineNumber, sasLoopNumber)


def getEnginesSasLoops(disksInfo, productModel):
    """
    @summary: 获取所有引擎下的SAS环路
    @return: 以引擎id为key，该引擎下的所有sas环路好列表为value的字典
    """
    enginesSasLoops = {}
      
    for diskInfo in disksInfo:
        diskID = diskInfo.get("ID", "")
        engineNumber, sasLoopNumber = getSasLoopByDisk(diskID, productModel)
        if engineNumber is None or sasLoopNumber is None:
            continue
        
        sasLoops = enginesSasLoops.get(engineNumber, [])
        if sasLoopNumber not in sasLoops:
            sasLoops.append(sasLoopNumber)
            enginesSasLoops[engineNumber] = sasLoops
    
    return enginesSasLoops


def getExpansionDepthByDisk(diskID, productModel):
    """
    @summary:根据硬盘ID的计算该硬盘所在硬盘框的级联深度
    """    
    if re.match("DAE([0-9]|[a-f]|[A-F]){3}", diskID) == None:
        return None
        
    thirdNumber = int(diskID[5], 16)
    
    if productModel in ["18500 V3", "18800 V3", "18500F V3", "18800F V3", "18500 V5", "18800 V5", "18500F V5", "18800F V5", "6800 V5", "6800F V5"]:
        return thirdNumber % 4 + 1
    
    if productModel in ["6800 V3", "6900 V3", "6800F V3"]:
        if thirdNumber < 8:
            return thirdNumber + 1
        else:
            return thirdNumber - 7
        
    return thirdNumber + 1


def getSasLoopsExpansionDepth(disksInfo, productModel):
    """
    @summary: 获取sas环路的级联深度
    @return: 以sas环路(引擎id, sas环路号)为key，该sas环路的级联深度
    """
    sasLoopsExpansionDepth = {}
      
    for diskInfo in disksInfo:
        diskID = diskInfo.get("ID", "")
        engineNumber, sasLoopNumber = getSasLoopByDisk(diskID, productModel)
        if engineNumber is None or sasLoopNumber is None:
            continue
        
        # 获取硬盘所在SAS环路的级联深度
        expansionDepth = getExpansionDepthByDisk(diskID, productModel)
        if expansionDepth == None:
            continue
        
        sasLoop = (engineNumber, sasLoopNumber)
        
        # 存储sas环路级联深度
        sasLoopExpansionDepth = sasLoopsExpansionDepth.get(sasLoop, 0)
        if expansionDepth > sasLoopExpansionDepth:
            sasLoopsExpansionDepth[sasLoop] = expansionDepth
    
    return sasLoopsExpansionDepth


def getSasLoopsDiskTypes(disksInfo, productModel):
    """
    @summary: 获取sas环路的硬盘类型
    @return: 以sas环路(引擎id, sas环路号)为key，该sas环路包含的所有硬盘类型列表为value
    """
    sasLoopsDiskTypes = {}
      
    for diskInfo in disksInfo:
        diskID = diskInfo.get("ID", "")
        engineNumber, sasLoopNumber = getSasLoopByDisk(diskID, productModel)
        if engineNumber is None or sasLoopNumber is None:
            continue
        
        diskType = diskInfo.get("Type", "")
        if not diskType:
            continue
        
        sasLoop = (engineNumber, sasLoopNumber)
        
        # 存储sas环路级联深度
        diskTypes = sasLoopsDiskTypes.get(sasLoop, [])
        if diskType not in diskTypes:
            diskTypes.append(diskType)
            sasLoopsDiskTypes[sasLoop] = diskTypes
    
    return sasLoopsDiskTypes


def getExpansionEnclosureByDisk(diskID):
    """
    @summary:根据硬盘ID的计算该硬盘所在硬盘框号
    """
    if re.match("DAE([0-9]|[a-f]|[A-F]){3}.", diskID) == None:
        return None
    
    return diskID.split(".")[0]


def getSasLoopsEnclosures(disksInfo, productModel):
    """
    @summary: 获取sas环路的硬盘框
    @return: 以sas环路(引擎id, sas环路号)为key，该sas环路包含的所有硬盘框
    """
    sasLoopsEnclosures = {}
      
    for diskInfo in disksInfo:
        diskID = diskInfo.get("ID", "")
        engineNumber, sasLoopNumber = getSasLoopByDisk(diskID, productModel)
        if engineNumber is None or sasLoopNumber is None:
            continue
        
        expansionEnclosure = getExpansionEnclosureByDisk(diskID)
        if not expansionEnclosure:
            continue
        
        sasLoop = (engineNumber, sasLoopNumber)
        
        # 存储sas环路级联深度
        expansionEnclosures = sasLoopsEnclosures.get(sasLoop, [])
        if expansionEnclosure not in expansionEnclosures:
            expansionEnclosures.append(expansionEnclosure)
            sasLoopsEnclosures[sasLoop] = expansionEnclosures
    
    return sasLoopsEnclosures
        
def getEnclosuresType(cli):
    """
    @summary: 获取所有硬盘框的类型
    @return: 以硬盘框ID为key，该硬盘框的类型为value
    """
    global allCliRet

    cmd = "show enclosure|filterColumn include columnList=ID,Height(U),Type"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    allCliRet = common.joinLines(allCliRet, cliRet)
    
    if flag != True:
        errMsg = common.getMsg(LANG, "cannot.get.enclosure.info")
        LOGGER.logNoPass("Get enclosure failed!")
        raise UnCheckException(errMsg, allCliRet)
    
    enclosuresInfo = cliUtil.getHorizontalCliRet(cliRet)
    if len(enclosuresInfo) == 0:
        errMsg = common.getMsg(LANG, "cannot.get.enclosure.info")
        LOGGER.logNoPass("Get enclosure failed!")
        raise UnCheckException(errMsg, allCliRet)
    
    enclosuresType = {}
    
    for enclosureInfo in enclosuresInfo:
        id = enclosureInfo.get("ID", "")
        type = enclosureInfo.get("Type", "")
        if not id or not type:
            errMsg = common.getMsg(LANG, "cannot.get.enclosure.info")
            LOGGER.logNoPass("Get enclosure failed!")
            raise UnCheckException(errMsg, allCliRet)
        
        enclosuresType[id] = type
    
    return (enclosuresType, enclosuresInfo)


def getEnclosuresHeight(enclosuresInfo):
    """
    @summary: 获取所有硬盘框的高度
    @return: 以硬盘框ID为key，该硬盘框的高度为value
    """
    enclosuresHeight = {}
    
    for enclosureInfo in enclosuresInfo:
        id = enclosureInfo.get("ID", "")
        height = enclosureInfo.get("Height(U)", "")
        if not id or not height:
            errMsg = common.getMsg(LANG, "cannot.get.enclosure.info")
            LOGGER.logNoPass("Get enclosure failed!")
            raise UnCheckException(errMsg, allCliRet)
        
        enclosuresHeight[id] = height
    
    return enclosuresHeight


def getSasLoopEnclosuresType(sasLoopEnclosures, enclosuresType):
    """
    @summary: 获取sas环路中的级联框的类型
    """
    sasLoopEnclosuresType = []
    
    for sasLoopEnclosure in sasLoopEnclosures:
        enclosureType = enclosuresType.get(sasLoopEnclosure)
        sasLoopEnclosuresType.append(enclosureType)
    
    return list(set(sasLoopEnclosuresType))


def getSasLoopEnclosuresHeight(sasLoopEnclosures, enclosuresHeight):
    """
    @summary: 获取sas环路中的级联框的高度
    """
    sasLoopEnclosuresHeight = []
    
    for sasLoopEnclosure in sasLoopEnclosures:
        enclosureHeight = enclosuresHeight.get(sasLoopEnclosure)
        sasLoopEnclosuresHeight.append(enclosureHeight)
    
    return list(set(sasLoopEnclosuresHeight))


def checkSasLoopEnclosureType(sasLoopEnclosuresType):
    """
    @summary: 检查sas环路中的硬盘框是否存在普通框与高密框混插的情况。
    """
    typeNum = len(sasLoopEnclosuresType)
    
    #只有新高密框和老高密框
    if typeNum == 2 and common.OLD_HIGH_ENC_TYPE in sasLoopEnclosuresType and common.NEW_HIGH_ENC_TYPE in sasLoopEnclosuresType:
        return True
    
    #存在高密框和普通硬盘框场景    
    if typeNum > 1 and (common.OLD_HIGH_ENC_TYPE in sasLoopEnclosuresType or common.NEW_HIGH_ENC_TYPE in sasLoopEnclosuresType):
        return False
        
    return True


def isHighEnclosure(enclosureType):
    """
    @summary: 检查硬盘框是否为高密框
    """
    if enclosureType == common.OLD_HIGH_ENC_TYPE or  enclosureType == common.NEW_HIGH_ENC_TYPE:
        return True
    
    return False


def getOptimalSasLoopExpansionDepth(productModel, productVersion, sasLoopDiskTypes, sasLoopEnclosureType, sasLoopEnclosureHeight):
    """
    @summary: 获取最佳的级联深度配置
    """
    if productVersion < "V300R005C00":
        if "SSD" in sasLoopDiskTypes:
            return 2
        elif productModel in ["HVS85T", "HVS88T", "18500", "18800", "18800F", "18500 V3", "18800 V3"]:
            return 4
        else:
            return 5
        
    if productVersion in ["V300R005C00", "V300R005C01"]:
        if productModel in ["2200 V3", "2600 V3", "2600 V3 for Video"]:
            return 5
        
    if productVersion in ["V300R006C00", "V300R006C01"]:
        if productModel in ["2600F V3", "5500F V3", "5600F V3", "5800F V3", "6800F V3", "18500F V3", "18800F V3"] and not isHighEnclosure(sasLoopEnclosureType):
            return 1
        if productModel in ["2200 V3", "2600 V3", "5300 V3", "5500 V3", "5600 V3", "5800 V3", "6800 V3", "18500 V3", "18800 V3"] and not isHighEnclosure(sasLoopEnclosureType):
            return 2
        if productModel in ["2600 V3 for Video", "2800 V3"] and not isHighEnclosure(sasLoopEnclosureType):
            return 3
    
    if productVersion in ["V300R006C10", "V300R006C20", "V300R006C30", "V300R006C30", "V300R006C50"]:
        if productModel in ["2600F V3", "2600F V3 Enhanced", "5500F V3", "5600F V3", "5800F V3", "6800F V3", "18500F V3", "18800F V3"] and not isHighEnclosure(sasLoopEnclosureType):
            return 1
        if productModel in ["2100 V3", "2200 V3", "2200 V3 Enhanced", "2600 V3", "2600 V3 Enhanced", "5300 V3", "5500 V3", "5600 V3", "5800 V3", "6800 V3"] and not isHighEnclosure(sasLoopEnclosureType):
            return 2
        if productModel in ["2600 V3 for Video", "2800 V3"]:
            if isHighEnclosure(sasLoopEnclosureType):
                return 1
            else:
                return 3
        if productModel in ["18500 V3", "18800 V3"]:
            if sasLoopEnclosureHeight == "2":
                return 2
            if sasLoopEnclosureHeight == "4":
                return 1
          
    if productVersion in ["V500R007C00", "V500R007C10", "V500R007C20", "V500R007C30"]:
         if productModel in ["5300F V5", "5500F V5", "5600F V5", "5800F V5", "6800F V5", "18500F V5", "18800F V5"] and not isHighEnclosure(sasLoopEnclosureType):
             return 1
         if productModel in ["5300 V5", "5300 V5 Enhanced", "5500 V5 Elite", "5500 V5", "5600 V5", "5800 V5"]:
             return 2
         if productModel in ["2800 V5"] and not isHighEnclosure(sasLoopEnclosureType):
             return 3
         if productModel in ["6800 V5"] and isHighEnclosure(sasLoopEnclosureType):
             return 2
         if productModel in ["18500 V5", "18800 V5"]:
            if sasLoopEnclosureHeight == "2":
                return 2
            if sasLoopEnclosureHeight == "4":
                return 1
    
    # 未发现最佳级联深度配置
    return 0
        

def checkEnclosuresHeight(sasLoopEnclosuresHeight, productModel):
    """
    @summary: 检查sas环路中硬盘框高度是否一致
    """
    if productModel in ["HVS85T", "HVS88T", "18500", "18800", "18800F", "18500 V3", "18800 V3", "18500 V5", "18800 V5", "18500F V5", "18800F V5"]:
        if len(sasLoopEnclosuresHeight) != 1:
            return False
        
    return True


def isEnclosureAbnormal(sasLoopEnclosures, expansionDepth):
    """
    @summary: 检查sas环路中硬盘框ID是否异常
    """
    # 框号异常（DAE999）或者
    if "DAE999" in sasLoopEnclosures:
        return True
    
    # 级联深度与硬盘框数量不相等（硬盘框移位后框号不会更新）时检查结果不准确
    if len(sasLoopEnclosures) != expansionDepth:
        return True
    
    return False


def checkSasLoopExpansionDepth(sasLoopsExpansionDepth, sasLoopsDiskTypes, sasLoopsEnclosures, enclosuresType, enclosuresHeight, productModel, productVersion):
    """
    @summary: 检查sas环路下的级联深度是否在推荐的范围内
    @return: errMsg
    """
    global allCliRet
    allErrMsg = ""
    
    for sasLoop in sasLoopsExpansionDepth:
        expansionDepth = sasLoopsExpansionDepth.get(sasLoop)
        sasLoopEnclosures = sasLoopsEnclosures.get(sasLoop)
        
        # 检查级联框ID是否异常
        if isEnclosureAbnormal(sasLoopEnclosures, expansionDepth):
            errMsg = common.getMsg(LANG, "expansion.enclosure.id.abnormal")
            LOGGER.logNoPass("sas loop (%s) expansion enclosure id is abnormal!" % ",".join(sasLoopEnclosures))
            raise UnCheckException(errMsg, allCliRet)
        
        sasLoopEnclosuresType = getSasLoopEnclosuresType(sasLoopEnclosures, enclosuresType)
        
        # 检查是否存在级联框混插
        isPass = checkSasLoopEnclosureType(sasLoopEnclosuresType)
        if not isPass:
            errMsg = common.getMsg(LANG, "commom.high-density.enclosures.mixed", ",".join(sasLoopEnclosures))
            allErrMsg += errMsg
            continue
        
        sasLoopEnclosuresHeight = getSasLoopEnclosuresHeight(sasLoopEnclosures, enclosuresHeight)
        
        # 检查级联框高度是否一致
        isPass = checkEnclosuresHeight(sasLoopEnclosuresHeight, productModel)
        if not isPass:
            errMsg = common.getMsg(LANG, "expansionEnclosure.height.not.same", ",".join(sasLoopEnclosures))
            allErrMsg += errMsg
            continue
        
        sasLoopDiskTypes = sasLoopsDiskTypes.get(sasLoop)
        sasLoopEnclosureType = sasLoopEnclosuresType[0]
        sasLoopEnclosureHeight = sasLoopEnclosuresHeight[0]
        
        optimalExpansionDepth = getOptimalSasLoopExpansionDepth(productModel, productVersion, sasLoopDiskTypes, sasLoopEnclosureType, sasLoopEnclosureHeight)
        LOGGER.logInfo("SAS loop (%s) expansion depth is %i, optimal expansion depth is %i." % (",".join(sasLoopEnclosures), expansionDepth, optimalExpansionDepth))
        if optimalExpansionDepth == 0:
            continue
        
        # 检查级联深度是否在最佳配置范围内
        if expansionDepth > optimalExpansionDepth:
            errMsg = common.getMsg(LANG, "cascading.depth.exceeding", (",".join(sasLoopEnclosures), expansionDepth, optimalExpansionDepth))
            allErrMsg += errMsg
        
    return allErrMsg

 
def getEngineSasPortNumber(cli, productModel, engIds):
    """
    @summary: 获取设备同一引擎下的可用SAS端口数
    """
    global allCliRet    
    engineSasPortNumber = {}
    
    cmd = "show port general physical_type=SAS"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    allCliRet = common.joinLines(allCliRet, cliRet)
    if flag != True: 
        raise UnCheckException(errMsg, allCliRet)
    
    for engId in engIds:
        portNumber = None
        
        if productModel in ["18500 V3", "18800 V3", "18500F V3", "18800F V3", "18500 V5", "18800 V5", "18500F V5", "18800F V5", "6800 V5", "6800F V5"]:
            lPortRe = re.compile(r"CTE%s\.L\d\.P\d" % engId)
            rPortRe = re.compile(r"CTE%s\.R\d\.P\d" % engId)
            lPortNum = len(lPortRe.findall(cliRet))
            rPortNum = len(rPortRe.findall(cliRet))
            portNumber = min(lPortNum,rPortNum)
            
        if productModel in ["6800 V3", "6900 V3", "6800F V3"]: 
            lPortReIOM0 = re.compile(r"CTE%s\.L\d\.IOM0\.P\d" % engId)
            rPortReIOM0 = re.compile(r"CTE%s\.R\d\.IOM0\.P\d" % engId)
            lPortReIOM1 = re.compile(r"CTE%s\.L\d\.IOM1\.P\d" % engId)
            rPortReIOM1 = re.compile(r"CTE%s\.R\d\.IOM1\.P\d" % engId)
            lPortNumIOM0 = len(lPortReIOM0.findall(cliRet))
            rPortNumIOM0 = len(rPortReIOM0.findall(cliRet))
            lPortNumIOM1 = len(lPortReIOM1.findall(cliRet))
            rPortNumIOM1 = len(rPortReIOM1.findall(cliRet))
            portNumber = min(lPortNumIOM0,rPortNumIOM0) + min(lPortNumIOM1,rPortNumIOM1)
            
        if productModel in ["5600 V3","5800 V3", "5600F V3", "5800F V3", "5600 V5", "5800 V5", "5600F V5", "5800F V5"]: 
            aPortRe = re.compile(r"CTE%s\.A\d\.P\d" % engId)
            bPortRe = re.compile(r"CTE%s\.B\d\.P\d" % engId)
            aPortNum = len(aPortRe.findall(cliRet))  
            bPortNum = len(bPortRe.findall(cliRet))
            portNumber = min(aPortNum,bPortNum)  
            
        if productModel in ["2100 V3", "2200 V3", "2200 V3 Enhanced", "2600 V3", "2600 V3 Enhanced","2600 V3 for Video", "2600F V3", "2600F V3 Enhanced", "2800 V3", "5300 V3", "5500 V3", "5500F V3", "2800 V5", "5300 V5", "5300 V5 Enhanced", "5500 V5 Elite", "5500 V5"]: 
            aEXPPortRe = re.compile(r"CTE%s\.A\.EXP\d" % engId) 
            bEXPlPortRe = re.compile(r"CTE%s\.B\.EXP\d" % engId) 
            aIOMPortRe = re.compile(r"CTE%s\.A\.IOM\d\.P\d" % engId) 
            bIOMPortRe = re.compile(r"CTE%s\.B\.IOM\d\.P\d" % engId) 
            aPortNumEXP = len(aEXPPortRe.findall(cliRet))
            bPortNumEXP = len(bEXPlPortRe.findall(cliRet))
            aPortNumIOM = len(aIOMPortRe.findall(cliRet))
            bPortNumIOM = len(bIOMPortRe.findall(cliRet))
            portNumber = min(aPortNumEXP, bPortNumEXP) + min(aPortNumIOM, bPortNumIOM)
        
        if portNumber:
            engineSasPortNumber[engId] = portNumber
       
    return engineSasPortNumber 
 
 
def isAllHighEnclosure(expansionEnclosuresTypes):
    """
    @summary: 检查引擎下的级联框是否全是高密框
    """
    typeNumber = len(list(set(expansionEnclosuresTypes)))
    
    if typeNumber == 1:
        if common.OLD_HIGH_ENC_TYPE in expansionEnclosuresTypes or common.NEW_HIGH_ENC_TYPE in expansionEnclosuresTypes:
            return True
        
    if typeNumber == 2:
        if common.OLD_HIGH_ENC_TYPE in expansionEnclosuresTypes and common.NEW_HIGH_ENC_TYPE in expansionEnclosuresTypes:
            return True
    
    return False
        

def checkSasLoopBalance(enginesSasLoops, sasLoopsExpansionDepth, sasLoopsDiskTypes, sasLoopsEnclosures, enclosuresType, enginesSasPortNumber, enclosuresInfo):
    """
    @summary: 检查引擎下的sas环路是否均衡
              1，若同一引擎下的SAS环路之间的硬盘框级联深度之差大于1（包含SSD盘的SAS、高密框与普通框混插、老高密框与新高密框混插的环路除外），检查结果为建议优化。
              2，若同一引擎下SAS环路号个数小于SAS端口数且存在硬盘框级联深度大于等于2的SAS环路，检查结果为建议优化 。
    """
    allErrMsg = ""
    
    for engine in enginesSasLoops:
        engineSasLoops = enginesSasLoops.get(engine)
        if not engineSasLoops:
            continue
        # 引擎下的所有级联框类型
        engineExpansionEnclosuresType = []
        
        # 引擎下是否存在级联深度大于1的环路
        isDepthGreat1 = False
        
        checkedSasLoops = []
        
        for engineSasLoop in engineSasLoops:
            sasLoop = (engine, engineSasLoop)
            
            expansionDepth = sasLoopsExpansionDepth.get(sasLoop)
            if expansionDepth >= 2:
                isDepthGreat1 = True 
            
            sasLoopDiskTypes = sasLoopsDiskTypes.get(sasLoop)
            
            # 包含SSD盘的SAS环路不检查均衡
            if "SSD" in sasLoopDiskTypes:
                continue
            
            sasLoopEnclosures = sasLoopsEnclosures.get(sasLoop)
            sasLoopEnclosuresType = getSasLoopEnclosuresType(sasLoopEnclosures, enclosuresType)
            engineExpansionEnclosuresType.extend(sasLoopEnclosuresType)
            
            # 高密框和普通硬盘框混插，新高密框和老高密框混插的环路不检查均衡    
            if (common.OLD_HIGH_ENC_TYPE in sasLoopEnclosuresType or common.NEW_HIGH_ENC_TYPE in sasLoopEnclosuresType) and len(sasLoopEnclosuresType) > 1:
                continue
            
            for checkedSasLoop in checkedSasLoops:
                checkedSasLoopDepth = sasLoopsExpansionDepth.get(checkedSasLoop)
                differenceValue = abs(expansionDepth - checkedSasLoopDepth)
                if differenceValue > 1:
                    errMsg = common.getMsg(LANG, "expansion.depth.not.Balance", (engine, ",".join(sasLoopsEnclosures.get(checkedSasLoop)), ",".join(sasLoopsEnclosures.get(sasLoop)), differenceValue))
                    allErrMsg += errMsg
               
            checkedSasLoops.append(sasLoop)
        
        # 引擎下的sas端口数量
        engineSasPortNumber = enginesSasPortNumber.get(engine)
        if engineSasPortNumber is None:
            continue
        
        if engineSasPortNumber % 2 != 0:
            errMsg = common.getMsg(LANG, "sas.port.number.abnormal",(engine, engineSasPortNumber))
            LOGGER.logInfo("Engine %s sas port number is %i." % (engine, engineSasPortNumber))
            return (False, errMsg)
        
        engineSasLoopNumber = len(engineSasLoops)
        if engineSasLoopNumber == 0:
            continue

        if not engineExpansionEnclosuresType:
            continue

        if isAllHighEnclosure(engineExpansionEnclosuresType):
            engineSasPortNumber = engineSasPortNumber / 2
            LOGGER.logInfo("Engine %s is expansioned high enclosures." % engine)
        
        LOGGER.logInfo("Engine %s sas port number is %i, sas loop number is %i, expansion depth is great 1:%s." % (engine, engineSasPortNumber, engineSasLoopNumber, str(isDepthGreat1)))
          
        if engineSasLoopNumber < engineSasPortNumber and isDepthGreat1:
            errMsg = common.getMsg(LANG, "sasPortNum.greater.than.loopNum", (engine, len(enclosuresInfo) - 1, engineSasPortNumber))
            allErrMsg += errMsg
         
    if allErrMsg:
        return (cliUtil.RESULT_WARNING, allErrMsg)
    
    return (True, "") 
                
            
    
    

    
    