# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import os
import re


class ImuLogParser(object):
    """
    imu日志解析，获取DQ信息
    涉及2个文件：
    /OSM/log/cur_debug/pangea/BMC/tmp/BMCLOG/local/imu_debug_log.tar.gz
    /OSM/log/cur_debug/pangea/BMC/tmp/BMCLOG/local/imu_debug_log_his.tar.gz
    """
    MAX_RECORD_NUM_PER_CTRL = 300

    def __init__(self, logger):
        self.logger = logger

    def parse(self, all_logs):
        """

        :param all_logs: 所有imu日志文件
        :return: dq_info：每个控制器是否存在多DQ 示例：{“0A”: True, "0B": False}
                 all_result：匹配到的原始信息
        """
        all_result = []
        dq_info = {}
        for ctrl, logs in all_logs.items():
            imu_logs = logs.get("imu", [])
            all_result.append("controller [{}] imu log records:".format(ctrl))
            one_ctrl_result = self._parse_one_ctrl(imu_logs)
            for record in one_ctrl_result:
                for key in ["dq", "node", "addr"]:
                    all_result.append(record.get(key))
                if record.get("is_multi_dq"):
                    dq_info[ctrl] = True
        return dq_info, os.linesep.join(all_result[-self.MAX_RECORD_NUM_PER_CTRL:])

    def _parse_one_ctrl(self, file_paths):
        """
        :param file_paths: 待分析的文件列表，文件是有序的，时间近的在文件排在前面
        :return:
        """
        one_ctrl_result = list()
        for file_path in file_paths:
            temp_result = self._parse_one_file(file_path)
            # 后解析的文件的数据时间比较老，数据放在前面，保证数据是从老到新的顺序排列
            one_ctrl_result = temp_result + one_ctrl_result
        return one_ctrl_result

    def _parse_one_file(self, file_path):
        self.logger.info("begin to parse file={}".format(file_path))
        all_result = []
        with open(file_path, 'r') as _log:
            for number, line in enumerate(_log):
                self._parse_err_info(line, all_result)
        self.logger.info("success parse: {}".format(file_path))
        return all_result

    @classmethod
    def _parse_err_info(cls, line, all_result):
        if not line:
            return

        dq_match = re.match("(\[RecoredDqInfo\]|\[GetDqInfo\]).*byte=(\w+).*byte=(\w+)", line)
        if dq_match:
            all_result.append(dict(dq=line, node="", addr="",
                                   is_multi_dq=cls._is_multi_dq(dq_match.group(2), dq_match.group(3))))
            return
        if not all_result:
            return
        current_record = all_result[-1]
        node_match = re.match("RASC Node.*", line)
        if node_match:
            current_record["node"] = line

        addr_match = re.match("SysAddr.*", line)
        if addr_match:
            current_record["addr"] = line

    @staticmethod
    def _is_multi_dq(dq_high_byte, dq_low_byte):
        """
        根据dq信息，判断是否是多DQ
        每一个byte称为一个device，对应4个bit位, 4个bit位中为bit为1的数量>1，则认为是多DQ

        :param dq_high_byte: dq高位信息，如：oxf
        :param dq_low_byte: dq低位信息，如：0x000001f
        :return: True-存在多DQ， False-不存在多DQ
        """
        dq_bytes = dq_high_byte[2:] + dq_low_byte[2:]
        for one_device_byte in dq_bytes:
            one_device_bits = bin(int(one_device_byte, 16))[2:]
            valid_bit_num = str(one_device_bits).count('1')
            if valid_bit_num > 1:
                return True
        return False


