# -*- coding: UTF-8 -*-

from memory_inspect.adapter.java_adapter import get_msg
from memory_inspect.rule.memory_rule import RuleBase
from memory_inspect.utils.rule_util import get_timestamp_second
from memory_inspect.utils import constants


SEVEN_DAYS = 7*24*60*60
RULE_TITLE = "\n-----------Rule {}---------\n"


class CtrlSysDiskRule(RuleBase):
    def __init__(self, rule_context):
        super(CtrlSysDiskRule, self).__init__(rule_context)
        self.new_err_obj_ids = {}

    def execute(self):
        error_ctrl = []
        self._check_ctrl_sys_disk_alarm(error_ctrl)
        self._check_ctrl_sys_disk_double_restart(error_ctrl)
        self._check_ctrl_sys_disk_hot_plug_interrupt(error_ctrl)
        self._check_ctrl_sys_disk_io_error(error_ctrl)
        self._check_ctrl_sys_disk_read_only(error_ctrl)
        self._check_ctrl_sys_disk_hard_reset_link(error_ctrl)
        if error_ctrl:
            self.err_msg.append(
                get_msg(
                    self.lang,
                    "controller.check.sys.disk.error",
                    ", ".join(list(set(error_ctrl))),
                )
            )
        self.logger.info("error ctrl:{}".format(error_ctrl))
        return bool(error_ctrl)

    def _check_ctrl_sys_disk_alarm(self, error_ctrl):
        """
        检测一周内某个控制器下的系统盘发生过1次上下电修复，
        并上报告警的情况（如下日志需要在30分钟内命中）
        正则依次为:
        bsp_power_off_rule .*Call bsp to power off port"
        bsp_power_on_rule .*Call bsp to power on port"
        ctrl_bord_boot_disk .*0x4000cf05"
        :return:
        """
        self.origin_info += RULE_TITLE.format("1")
        time_period_day = 7
        records_power_on = self._get_group_obj_records(
            constants.BSP_POWER_ON_RULE, time_period_day
        )
        records_power_off = self._get_group_obj_records(
            constants.BSP_POWER_OFF_RULE, time_period_day
        )
        records_boot_disk = self._get_group_obj_records(
            constants.CTRL_BORD_BOOT_DISK, time_period_day
        )
        for obj_id, one_obj_records in records_power_off.items():
            if obj_id not in self.new_err_obj_ids.get(
                    constants.BSP_POWER_OFF_RULE, []):
                continue
            # Power off与power on相差5分钟内
            filter_records = self.filter_record_by_time(
                one_obj_records, 300, records_power_on.get(obj_id, [])
            )
            if not filter_records:
                continue
            # 告警必须在poweron之后30分钟内
            filter_records = self.filter_record_by_time(
                filter_records, 30 * 60, records_boot_disk.get(obj_id, []),
            )
            if filter_records:
                self.logger.info(
                    "ctrl:{} has alarm error.".format(obj_id)
                )
                error_ctrl.append(obj_id)

        return error_ctrl

    def _check_ctrl_sys_disk_io_error(self, error_ctrl):
        """
        检测到一周内某个控制器下系统盘发生了IO错误
        并进行上下电，但文件系统变成只读
        正则依次为:
        mask_act_err_rule .*exception Emask.*SAct.*SErr.*action"
        bsp_power_off_rule .*Call bsp to power off port"
        bsp_power_on_rule .*Call bsp to power on port"
        ext_fs_rule .*EXT4-fs error"
        read_only_fs_rule .*Read-only file system"
        :return:
        """
        self.origin_info += RULE_TITLE.format("4")
        time_period_day = 7
        records_mask_act = self._get_group_obj_records(
            constants.MASK_ACT_ERR_RULE, time_period_day
        )
        records_power_off = self._get_group_obj_records(
            constants.BSP_POWER_OFF_RULE, time_period_day
        )
        records_power_on = self._get_group_obj_records(
            constants.BSP_POWER_ON_RULE, time_period_day
        )
        records_ext_fs = self._get_group_obj_records(
            constants.EXT_FS_RULE, time_period_day
        )
        records_fs = self._get_group_obj_records(
            constants.READ_ONLY_FS_RULE, time_period_day
        )
        self.logger.info("self.new_err_obj_ids:{}".format(
            self.new_err_obj_ids))

        for obj_id, one_obj_records in records_mask_act.items():
            if obj_id not in self.new_err_obj_ids.get(
                    constants.MASK_ACT_ERR_RULE, []):
                continue
            filter_records = self._get_all_match_records(
                one_obj_records, records_power_off, records_power_on,
                records_ext_fs, records_fs, obj_id
            )
            if not filter_records:
                continue
            self.logger.info(
                "ctrl:{} has io error and restart and read only.".format(
                    obj_id
                )
            )
            error_ctrl.append(obj_id)
        return error_ctrl

    def _get_all_match_records(self, one_obj_records, records_1, records_2,
                               records_6, records_7, obj_id):
        # 7天内下电
        filter_records = self.filter_record_by_time(
            one_obj_records, SEVEN_DAYS, records_1.get(obj_id, [])
        )
        self.logger.info("filter_records 1:{}".format(filter_records))
        if not filter_records:
            return filter_records
        # 上电5分钟内
        filter_records = self.filter_record_by_time(
            filter_records, 300,
            records_2.get(obj_id, [])
        )
        self.logger.info("filter_records 1:{}".format(filter_records))
        if not filter_records:
            return filter_records
        # 7天内顺序出现 .*EXT4-fs error
        filter_records = self.filter_record_by_time(
            filter_records, SEVEN_DAYS,
            records_6.get(obj_id, [])
        )
        self.logger.info("filter_records 1:{}".format(filter_records))
        if not filter_records:
            return filter_records

        # 7天内顺序出现.*Read-only file system"
        filter_records = self.filter_record_by_time(
            filter_records, SEVEN_DAYS,
            records_7.get(obj_id, [])
        )
        self.logger.info("filter_records 1:{}".format(filter_records))
        return filter_records

    def _check_ctrl_sys_disk_double_restart(self, error_ctrl):
        self.origin_info += RULE_TITLE.format("2")
        time_period_day = 7
        records_power_off = self._get_group_obj_records(
            constants.BSP_POWER_OFF_RULE, time_period_day
        )
        records_power_on = self._get_group_obj_records(
            constants.BSP_POWER_ON_RULE, time_period_day
        )
        self.logger.info("records_1:{}, records_2:{}, new_obj_ids:{}".format(
            records_power_off.items(),
            records_power_on.items(),
            self.new_err_obj_ids)
        )
        for obj_id, one_obj_records in records_power_off.items():
            if obj_id not in self.new_err_obj_ids.get(
                    constants.BSP_POWER_OFF_RULE, []):
                continue
            # Power off与power on相差5分钟内
            filter_records = self.filter_record_by_time(
                one_obj_records, 300, records_power_on.get(obj_id, [])
            )
            if len(filter_records) >= 2:
                self.logger.info(
                    "ctrl:{} has double restart sys disk.".format(obj_id)
                )
                error_ctrl.append(obj_id)
        return error_ctrl

    def _check_ctrl_sys_disk_hot_plug_interrupt(self, error_ctrl):
        # 检测到一周内某个控制器下的系统盘发生过五次或者以上物理链路突然闪断情况。
        # 正则为：.*Hotplug interrupt from SATA module is coming
        self.origin_info += RULE_TITLE.format("3")
        tmp_error_ctrl = self._check_ctrl_sys_disk_common_rule(
            constants.HOT_PLUG_INTERRUPT, 4)
        self.logger.info(
            "ctrl:{} has hot plug interrupt sys disk.".format(tmp_error_ctrl)
        )
        error_ctrl.extend(tmp_error_ctrl)
        return error_ctrl

    def _check_ctrl_sys_disk_read_only(self, error_ctrl):
        # 检测到一周内某个控制器下持续打印文件系统只读信息（大于50条）
        # 正则为：.*Read-only file system
        self.origin_info += RULE_TITLE.format("5")
        tmp_error_ctrl = self._check_ctrl_sys_disk_common_rule(
            constants.READ_ONLY_FS_RULE, 50)
        self.logger.info(
            "ctrl:{} has sys_disk_read_only.".format(tmp_error_ctrl)
        )
        error_ctrl.extend(tmp_error_ctrl)
        return error_ctrl

    def _check_ctrl_sys_disk_hard_reset_link(self, error_ctrl):
        # 检测到一周内某个控制器下持续出现系统盘物理链路闪断的情况（1周内大于50次）
        # 正则为：.*hard resetting link
        self.origin_info += RULE_TITLE.format("6")
        tmp_error_ctrl = self._check_ctrl_sys_disk_common_rule(
            constants.HARD_RESETTING_LINK, 50)
        self.logger.info("hard_reset_link risk ctrl:{}".format(tmp_error_ctrl))
        error_ctrl.extend(tmp_error_ctrl)
        return error_ctrl

    def _check_ctrl_sys_disk_common_rule(self, error_type, error_count):
        """
        基于代码提取的公共规则检查方法
        :param error_type:
        :param error_count:
        :return:
        """
        error_ctrl = []
        time_period_day = 7
        obj_grouped_records_1 = self._get_group_obj_records(
            error_type, time_period_day
        )
        self.logger.info("obj_grouped_records_1:{}".format(obj_grouped_records_1))
        for obj_id, one_obj_records in obj_grouped_records_1.items():
            self.logger.info("obj_id:{}, one_obj_records:{}".format(obj_id, one_obj_records))
            if obj_id not in self.new_err_obj_ids.get(error_type, []):
                continue

            if len(one_obj_records) > error_count:
                error_ctrl.append(obj_id)
        return error_ctrl

    def _get_group_obj_records(self, err_type, time_period_day):
        records_rule = self._query_err_obj(err_type, time_period_day)
        return self.group_by(records_rule, "obj_id")

    @staticmethod
    def filter_record_by_time(one_obj_records, time_seg, records):
        all_records = []
        one_obj_records = sorted(one_obj_records, key=lambda x: x.get("time"))
        for one_obj in one_obj_records:
            start_time = one_obj.get("time")
            for record in records:
                if (
                    0
                    <= get_timestamp_second(record.get("time"))
                    - get_timestamp_second(start_time)
                    < time_seg
                ):
                    # 找到5分钟之内的配对记录，退出。
                    all_records.append(record)
                    break
        return all_records

    def _query_err_obj(self, err_type, time_period_day=7):
        self._query_new_err_obj_ids(err_type)
        start_time = self.get_time_by_delta_process(
            self.rule_context.current_time, days=-1 * time_period_day
        )
        sql_str = (
            "SELECT time, obj_id, err_type FROM tbl_log_err_info "
            "WHERE time >= '{}' and sn = '{}' and err_type='{}'".format(
                start_time, self.rule_context.sn, err_type
            )
        )
        records = self.db_service.log_err_tbl.query(sql_str)
        records = self._sort_records_by_ctrl_and_time(records)
        self.add_db_record_to_origin_info(
            records[-100:], "{}(total:{}):".format(err_type, len(records))
        )
        return records

    def _query_new_err_obj_ids(self, err_type):
        self.logger.info("self.rule_context.last_err_time:{}".format(self.rule_context.last_err_time))
        sql_str = (
            "SELECT time, obj_id, err_type FROM tbl_log_err_info "
            "WHERE time > '{}' and sn = '{}' and err_type='{}' GROUP BY sn, err_type, obj_id".format(
                self.rule_context.last_err_time, self.rule_context.sn, err_type
            )
        )
        new_records = self.db_service.log_err_tbl.query(sql_str)
        self.new_err_obj_ids[err_type] = [record.get("obj_id") for record in new_records]

    def _sort_records_by_ctrl_and_time(self, data_list):
        """
        按控制器、和obj 排序
        :param data_list:
        :return:
        """
        if not data_list:
            return data_list
        result_list = []

        group_data = self.group_by(data_list, 'obj_id')
        for key, tmp_data_list in group_data.items():
            sorted_records = sorted(
                tmp_data_list, key=lambda x: x.get("time"))
            group_data[key] = sorted_records
        data_keys = list(group_data.keys())
        data_keys.sort()
        [result_list.extend(group_data.get(key)) for key in data_keys]
        return result_list
