# -*- coding: UTF-8 -*-
import os
import time
import traceback
import codecs

from utils import Products
from com.huawei.ism.tool.obase.exception import ToolException

import cliUtil
import cli_util_cache as cache_util
import common
import common_cache as cache_common
import config
from frameone.util import common as frame_common
from create_hyper_metro_cmd import CreateHyperMetroCmd
from save_hyper_metro_info_to_db import ExecuteCmdCacheTask

from cbb.frame.cli.cli_with_cache import execute_cmd_in_cli_mode


CLI_HOR_DIRECTION = "Horizontal"
CLI_HOR_DIRECTION_NO_STAND = "HorizontalNoStand"
CLI_VER_DIRECTION = "Vertical"
SSH_LOCK_STATU_OCCUPIED = 1
SSH_LOCK_STATU_FREE = 0
WAIT_SSH_LOCK_TIMEOUT = 1200
HYPERMETRO_NODE_NAME = {
    "zh": u"双活一致性",
    "en": "HyperMetro information consistency",
}
SOFTWARE_MODE_NAME = {
    "zh": u"软件状态",
    "en": "Software status",
}
FS_CAPACITY_SUDDEN_EXHAUSTION = {
    "zh": u"文件系统容量风险检查",
    "en": "Checking the File System Capacity Risk"
}


class QueryHyperMetroInfo:
    def __init__(self, env, logger, cli, cur_progress, hyper_total_progress):
        self.env = env
        self.logger = logger
        self.cli = cli
        self.cur_progress = cur_progress
        self.hyper_total_progress = hyper_total_progress
        self.lang = common.getLang(env)

    def collect_hyper_metro_info(self):
        """
        收集双活回文
        :return:
        """
        dev_node = self.env.get("devInfo")
        cur_dev_sn = dev_node.getDeviceSerialNumber()
        self.saveObjToFile(cur_dev_sn, {})
        cmd_dict = {}
        cur_progress = 0
        try:
            flag = cache_util.download_ana_config(self.env, self.logger, self.cli)
            self.logger.logInfo("is downloaded config:{0}".format(flag))

            has_hyper_item = self.hasHyperMetroCheckItems()
            self.logger.logInfo(" hasHyperMetroFlag is :{}".format(has_hyper_item))

            # 检查ssh锁，非当前批次查询，需要锁一下ssh连接
            self._check_ssh_lock()
            if has_hyper_item:
                dev_node_key = common.HYPER_DEV_STATUS.format(cur_dev_sn)
                py_obj = self.env.get("objectForPy")
                py_obj.put(dev_node_key, common.HYPER_DEV_STATUS_COLLECTING)
                cur_progress = self.pre_query_command(
                    self.cli, cmd_dict, cur_dev_sn, self.cur_progress,
                    self.hyper_total_progress
                )
                self.logger.logInfo("cur_progress {}".format(cur_progress))
                self.savePreCheckCmdEchos(self.cli, cur_dev_sn, dev_node, cmd_dict)
                self.assureAllPairsPreCheckOutputs()
            else:
                self.default_collect_hyper_info(self.cli, cur_dev_sn)
                self.assureAllPairsPreCheckOutputs(True)
                has_alarm_check = self.has_alarm_check_items()
                if has_alarm_check:
                    cur_progress = self.pre_query_developer_command(
                        self.cli, cmd_dict, cur_dev_sn, self.hyper_total_progress
                    )
                    self.logger.logInfo("cur_progress {}".format(cur_progress))
                else:
                    cur_progress += self.hyper_total_progress
                    common.refreshProcess(self.env, self.cur_progress, self.logger)
        except (ToolException, Exception) as e:
            self.logger.logException(e)
            dev_node_key = common.HYPER_DEV_STATUS.format(cur_dev_sn)
            py_obj = self.env.get("objectForPy")
            py_obj.put(dev_node_key, common.HYPER_DEV_STATUS_INTERRUPTED)
        self.saveObjToFile(cur_dev_sn, cmd_dict)
        return cur_progress

    def _check_ssh_lock(self):
        """
        非当前批次查询，需要锁一下ssh连接。当前设备使用
        背景：巡检一次巡检10台设备，可能存在当前双活对端未在同一批次。
        需要设置一个锁来保证ssh连接只有一个线程在使用。
        :return:
        """
        for times in range(WAIT_SSH_LOCK_TIMEOUT):
            if self.get_ssh_res_lock(self.env.get("devInfo")) != 0:
                time.sleep(1)
                continue
            else:
                break
        self.logger.logInfo("Force lock current SSH connection status as occupied.")
        self.set_ssh_res_lock(
            self.env.get("devInfo"), SSH_LOCK_STATU_OCCUPIED)

    def pre_query_command(
        self, cli, cmd_echo_dict, sn, cur_progress, pre_query_progress
    ):
        """
        启动多线程提前查询cli回文
        :param cli:
        :param cmd_echo_dict:
        :param sn:
        :param cur_progress:
        :param pre_query_progress:
        :return:
        """
        cmd_obj = CreateHyperMetroCmd(
            self.env, self.logger, self.cli, cmd_echo_dict, sn
        )
        cmd_list = cmd_obj.create_command()
        cmd_list_developer = []
        has_alarm_check = self.has_alarm_check_items()
        if has_alarm_check:
            cmd_list_developer = cmd_obj.create_command_developer()
        self.logger.logInfo("multi thread cmd list:{}; dev_cmd_list {}".format(cmd_list, cmd_list_developer))
        execute_cmd_cache_task = ExecuteCmdCacheTask(
            cmd_list, cli, self.env, self.logger,
            cur_progress, pre_query_progress, sn, cmd_list_developer
        )
        return execute_cmd_cache_task.execute_task()

    def pre_query_developer_command(
        self, cli, cmd_echo_dict, sn, pre_query_progress
    ):
        """
        查询developer命令回文
        :param cli:
        :param cmd_echo_dict:
        :param sn:
        :param pre_query_progress:
        :return:
        """
        cur_progress = 0
        cmd_obj = CreateHyperMetroCmd(
            self.env, self.logger, self.cli, cmd_echo_dict, sn
        )
        cmd_list = []
        cmd_list_developer = cmd_obj.create_command_developer()
        self.logger.logInfo("multi developer thread cmd list:{}".format(cmd_list_developer))
        execute_cmd_cache_task = ExecuteCmdCacheTask(
            cmd_list, cli, self.env, self.logger,
            cur_progress, pre_query_progress, sn, cmd_list_developer
        )
        return execute_cmd_cache_task.execute_task()

    def hasHyperMetroCheckItems(self):
        """
        @summary: check whether hyperMetro consistency check items has been selected.
        """
        selectedCheckItems = self.env.get("checkItems")
        hyperMetroNodeId = HYPERMETRO_NODE_NAME.get(self.lang)
        if not selectedCheckItems.get(hyperMetroNodeId):
            return False
        for check_item in selectedCheckItems.get(hyperMetroNodeId):
            self.logger.logInfo("checkItem:" + check_item.getName())
            if check_item.isEnable():
                self.logger.logInfo(
                    "checkItem %s on hyperMetro node is enabled! "
                    % check_item.getName()
                )
                return True
        return False

    def has_alarm_check_items(self):
        """
        @summary: 检测是否存在软件检查项.
        """
        selected_check_items = self.env.get("checkItems")
        alarm_node_id = SOFTWARE_MODE_NAME.get(self.lang)
        if not selected_check_items.get(alarm_node_id):
            return False
        for check_item in selected_check_items.get(alarm_node_id):
            self.logger.logInfo("checkItem:" + check_item.getName())
        # 判断是否包含文件系统容量跳变检查
            if check_item.getName() == FS_CAPACITY_SUDDEN_EXHAUSTION.get(self.lang) and check_item.isEnable():
                return True
        return False

    def assureAllPairsPreCheckOutputs(self, is_default=False):
        """
        @summary: assure all pairs of current device has started to collect pre-check items
        """
        self.logger.logInfo("now start to fetch remote devices' pre-check information.")
        isSucc, curRemoteSNs, errMsg = common.checkAddedRemoteDevSn(
            self.env, self.lang
        )
        if not isSucc or len(curRemoteSNs) < 1:
            self.logger.logInfo("isSucc:%s" % str(isSucc))
            return

        basePath = self.getPreCheckFileLocation()
        for cur_remote_sn in curRemoteSNs:
            devNode = self.getRemoteDeviceBySn(cur_remote_sn)
            filePath = os.path.join(basePath, cur_remote_sn)
            self.logger.logInfo("file localtion is %s" % filePath)
            if os.path.exists(filePath):
                self.logger.logInfo(
                    "device(SN: %s) collected hyper info." % str(
                        cur_remote_sn)
                )
                continue
            if self.get_ssh_res_lock(devNode) != 0:
                continue
            self.logger.logInfo(
                "start get remote (SN %s) hy info, ip:%s "
                % (str(cur_remote_sn), str(devNode.getIp()))
            )
            self.set_ssh_res_lock(devNode, SSH_LOCK_STATU_OCCUPIED)

            self._get_remote_dev_hyper_info(cur_remote_sn, devNode, is_default)

    def _get_remote_dev_hyper_info(self, cur_remote_sn, dev_node, is_default):
        """
        获取远端设备双活信息
        :param cur_remote_sn: 远端SN
        :param dev_node: devnode
        :param is_default: 是否是默认搜集的少部分命令
        :return:
        """
        ssh, err_id = common.getSSHbyDevNode(dev_node)
        if ssh:
            if is_default:
                self.default_collect_hyper_info(ssh, cur_remote_sn)
            else:
                cmd_echo_dict = dict()
                self.save_remote_dev_cmd(
                    cur_remote_sn, ssh, cmd_echo_dict)
                self.savePreCheckCmdEchos(
                    ssh, cur_remote_sn, dev_node, cmd_echo_dict)
                self.saveObjToFile(cur_remote_sn, {})
        else:
            # 获取连接异常需要设置标识位，防止取信息时会等待8小时超时。
            self.logger.logError(
                "{0} collected hyper info " "failed.".format(
                    cur_remote_sn)
            )
            dev_node_key = common.HYPER_DEV_STATUS.format(cur_remote_sn)
            py_obj = self.env.get("objectForPy")
            py_obj.put(dev_node_key, common.HYPER_DEV_STATUS_INTERRUPTED)
        self.set_ssh_res_lock(dev_node, SSH_LOCK_STATU_FREE)
        self.logger.logInfo(
            "finished fetching pre-check info on "
            "remote device(SN %s)." % str(cur_remote_sn)
        )

    def save_remote_dev_cmd(self, cur_remote_sn, ssh, cmd_echo_dict):
        """
        非当前批次保存双活回文
        :param cur_remote_sn:
        :param ssh:
        :param cmd_echo_dict:
        :return:
        """
        dev_node_key = common.HYPER_DEV_STATUS.format(cur_remote_sn)
        py_obj = self.env.get("objectForPy")
        py_obj.put(dev_node_key,
                   common.HYPER_DEV_STATUS_COLLECTING)

        cmd_obj = CreateHyperMetroCmd(
            self.env, self.logger, ssh, cmd_echo_dict, cur_remote_sn
        )
        cmd_list = cmd_obj.create_command()
        cmd_list_developer = []
        has_alarm_check = self.has_alarm_check_items()
        if has_alarm_check:
            cmd_list_developer = cmd_obj.create_command_developer()
        self.logger.logInfo("test cmd {} and dev_cmd {}")
        execute_cmd_cache_task = ExecuteCmdCacheTask(
            cmd_list, ssh, self.env, self.logger,
            99.0, 1, cur_remote_sn, cmd_list_developer
        )
        return execute_cmd_cache_task.execute_task()

    def getRemoteDeviceBySn(self, deviceSn):
        """
        @summary: get remotedevice's devNode by giving device SN
        """
        selectDevs = self.env.get("selectDevs")
        for device_node in selectDevs:
            if device_node.getDeviceSerialNumber() == deviceSn:
                return device_node

    def get_ssh_res_lock(self, key):
        """
        @summary: get ssh resource lock from backwards
        """
        allSSHLockMap = self.env.get("hyperMetroLockMap")
        if allSSHLockMap:
            lockStatus = allSSHLockMap.get(key.getDeviceSerialNumber())
            if lockStatus is None:
                allSSHLockMap.put(key.getDeviceSerialNumber(), 0)
                lockStatus = 0
            self.logger.logInfo("get connection lock status: %s" % str(lockStatus))
            return lockStatus
        self.logger.logInfo("failed to parse ssh lock status.")
        return -1

    def set_ssh_res_lock(self, devNode, status):
        """
        @summary: set devnode's ssh connection status
        """
        allSSHLockMap = self.env.get("hyperMetroLockMap")
        if allSSHLockMap:
            allSSHLockMap.put(devNode.getDeviceSerialNumber(), status)

    def default_collect_hyper_info(self, cli, devSn):
        # 用于判断对端是否还活着
        dev_node_key = common.HYPER_DEV_STATUS.format(devSn)
        py_obj = self.env.get("objectForPy")
        py_obj.put(dev_node_key, common.HYPER_DEV_STATUS_COLLECTING)
        cmdEchoDict = {}
        try:
            cmd = "show system general"
            self.update_cmd_cache(cli, cmd, cmdEchoDict, devSn)

            cmd = "show upgrade package"
            self.update_cmd_cache(cli, cmd, cmdEchoDict, devSn)

            self.update_hyper_switch(cli, cmdEchoDict, devSn)
            py_obj.put(dev_node_key, common.HYPER_DEV_STATUS_COLLECT_END)
        except (ToolException, Exception) as e:
            self.logger.logError(str(e))
            self.logger.logError(traceback.format_exc())
            py_obj.put(dev_node_key, common.HYPER_DEV_STATUS_INTERRUPTED)

    def savePreCheckCmdEchos(self, cli, devSn, devNode, cmd_dict):
        # 用于判断对端是否还活着
        dev_node_key = common.HYPER_DEV_STATUS.format(devSn)
        py_obj = self.env.get("objectForPy")
        # 后台查询时，下载config.
        cache_util.download_ana_config_back_end(
            py_obj, self.logger, cli, devNode, self.lang)
        try:
            self.update_hyper_switch(cli, cmd_dict, devSn)
            self.update_hypermetro_pool_info(cli, cmd_dict, devSn)
            self.update_quorum_server_info(cli, cmd_dict, devSn)
            # 仲裁链路端口信息
            self.update_arb_net_work_hyper_metro_config(cli, cmd_dict, devSn)
            py_obj.put(dev_node_key, common.HYPER_DEV_STATUS_COLLECT_END)
        except (ToolException, Exception) as e:
            self.logger.logError(str(e))
            self.logger.logError(traceback.format_exc())
            py_obj.put(dev_node_key, common.HYPER_DEV_STATUS_INTERRUPTED)

    def get_pair_local_id_list(self, cli, cmd, cmd_echo_dict, sn):
        """
        获取本地 local id
        :param cli:
        :param cmd:
        :param cmd_echo_dict:
        :param sn:
        :return:
        """
        if cmd in cmd_echo_dict:
            cli_ret = cmd_echo_dict.get(cmd, {}).get(common.CLI_RET_KEY)
        else:
            flag, cli_ret, err_msg = self.update_cmd_cache(cli, cmd, cmd_echo_dict, sn)
            cmd_echo_dict[cmd] = {
                common.FLAG_KEY: flag,
                common.CLI_RET_KEY: cli_ret,
                common.ERROR_MSG_KEY: err_msg,
            }
        pair_info_list = cliUtil.getHorizontalCliRet(cli_ret)
        return [pair_info.get("Local ID") for pair_info in pair_info_list]

    def get_san_pair_local_id_list(self, cli, cmd_echo_dict, sn):
        """
        获取SAN pair 本端id，有缓存，不会重复执行命令
        :param cli:
        :param cmd_echo_dict:
        :param sn:
        :return:
        """
        return self.get_pair_local_id_list(
            cli, cache_common.CMD_SAN_PAIR, cmd_echo_dict, sn
        )

    def get_nas_pair_local_id_list(self, cli, cmd_echo_dict, sn):
        """
        获取NAS pair本端id，有缓存，不会重复执行命令
        :param cli:
        :param cmd_echo_dict:
        :param sn:
        :return:
        """
        return self.get_pair_local_id_list(
            cli, cache_common.CMD_NAS_PAIR, cmd_echo_dict, sn
        )

    def update_hyper_switch(self, cli, cmd_dict, sn):
        """
        执行命令show hyper_metro general，查看当前双活的功能开关状态，
        获取“Syncronized LUN SN Switch”字段
        :param cli: ssh链接
        :param cmd_dict: 容器
        :param sn: sn
        :return:
        """
        context = self.env.get("objectForPy")
        cmd = "show hyper_metro general"
        cliUtil.enterDeveloperMode(cli, self.lang)
        flag, cli_ret, err_msg = execute_cmd_in_cli_mode(
            context, cli, cmd, self.lang, sn, self.logger
        )
        cmd_dict[cmd] = {
            common.FLAG_KEY: flag,
            common.CLI_RET_KEY: cli_ret,
            common.ERROR_MSG_KEY: err_msg,
        }
        cliUtil.enterCliModeFromSomeModel(cli, self.lang)

    def get_pair_local_and_remote_id_list(
        self, cli, cmd_echo_dict, sn, cmd=cache_common.CMD_SAN_PAIR
    ):
        flag, ret, msg = self.update_cmd_cache(cli, cmd, cmd_echo_dict, sn)
        ret_list = cliUtil.getHorizontalCliRet(ret)
        local_id_list = []
        remote_id_list = []
        for pair_info in ret_list:
            local_id_list.append(pair_info.get("Local ID"))
            remote_id_list.append(pair_info.get("Remote ID"))
        return local_id_list, remote_id_list

    def update_hypermetro_pool_info(self, cli, cmd_echo_dict, sn):
        """
        collect the information of pool and disk domain
        :param cli:
        :param cmdEchoDict:
        :param sn:
        :return:
        """

        local_id_list, remote_id_list = self.get_pair_local_and_remote_id_list(
            cli, cmd_echo_dict, sn
        )
        cmd = "show lun general lun_id=%s"
        pool_id_list = self.executeCmd(
            sn, cli, cmd, local_id_list, CLI_VER_DIRECTION,
            "Pool ID", cmd_echo_dict,
        )
        pool_id_list = list(set(pool_id_list))
        self.logger.logInfo("poolIdList is {}".format(pool_id_list))
        if "Dorado" in str(common.getProductModeFromContext(self.env)):
            cmd = "show storage_pool general pool_id=%s"
        else:
            cmd = "show storage_pool tier pool_id=%s"
        self.executeCmd(
            sn, cli, cmd, pool_id_list, CLI_VER_DIRECTION, "ID", cmd_echo_dict
        )
        cmd = "show lun general lun_id=%s"
        remote_lun_wwn_list = self.executeCmd(
            sn, cli, cmd, local_id_list,
            CLI_VER_DIRECTION, "Remote LUN WWN", cmd_echo_dict,
            None, None, ("Disk Location", "External"),
        )
        cmd = "show remote_lun general array_type=heterogeneity remote_lun_wwn=%s"
        self.executeCmd(
            sn, cli, cmd, remote_lun_wwn_list, CLI_VER_DIRECTION, "ID",
            cmd_echo_dict,
        )

    def update_quorum_server_info(self, cli, cmd_dict, sn):
        """
        查询 san domain和 nas domain 以及各自domain 下的pair信息
        :param cli:
        :param cmdEchoDict:
        :param sn:
        :return:
        """
        domain_cmd = "show hyper_metro_domain general"
        domain_detail_cmd = "show hyper_metro_domain general domain_id=%s"
        self.query_domain_detail(cli, cmd_dict, sn, domain_cmd, domain_detail_cmd)
        pair_cmd = (
            "show hyper_metro_pair general |filterRow column=Domain\sID "
            "predict=equal_to value=%s|filterColumn include "
            "columnList=Local\sID,Remote\sID,ID,Domain\sID"
        )
        self.query_domain_detail(cli, cmd_dict, sn, domain_cmd, pair_cmd)

        domain_cmd = "show fs_hyper_metro_domain general"
        domain_detail_cmd = "show fs_hyper_metro_domain general domain_id=%s"
        self.query_domain_detail(cli, cmd_dict, sn, domain_cmd, domain_detail_cmd)

        pair_cmd = (
            "show hyper_metro_pair general |filterRow column=Domain\sID "
            "predict=equal_to value=%s|filterColumn include "
            "columnList=Local\sID,Remote\sID,ID,Domain\sID"
        )
        self.query_domain_detail(cli, cmd_dict, sn, domain_cmd, pair_cmd)

        p_version = str(self.env.get("devInfo").getProductVersion())
        if all(
            [
                p_version not in config.RISK_QUORUM_PRODUCT_VERSION,
                not (Products.isDigitalVer(p_version) or p_version.startswith("V700")),
            ]
        ):
            return

        cmd = "arb showall"
        if not common.is18000(self.env, cli):
            cliUtil.enterDebugModeFromCliMode(cli, self.lang)
            self.update_cmd_cache(cli, cmd, cmd_dict, sn)
            cliUtil.enterCliModeFromSomeModel(cli, self.lang)

    def query_domain_detail(self, cli, cmdEchoDict, sn, domain_cmd,
                            domain_detail_cmd):
        """
        查询domain及详情信息，传入SAN或NAS的双活命令
        :param cli:
        :param cmdEchoDict:
        :param sn:
        :param domain_cmd:
        :param domain_detail_cmd:
        :return:
        """
        flag, cli_ret, err_msg = self.update_cmd_cache(cli, domain_cmd, cmdEchoDict, sn)
        cmdEchoDict[domain_cmd] = {
            common.FLAG_KEY: flag,
            common.CLI_RET_KEY: cli_ret,
            common.ERROR_MSG_KEY: err_msg,
        }
        hyperMetroDomainInfoList = cliUtil.getHorizontalCliRet(cli_ret)
        hyperMetroDomainIdList = []
        for domain_info in hyperMetroDomainInfoList:
            domain_id = domain_info.get("ID", "")
            if not domain_id:
                continue
            hyperMetroDomainIdList.append(domain_id)
        self.logger.logInfo("hyperMetroDomainIdList is %s" % hyperMetroDomainIdList)
        self.executeCmd(
            sn,
            cli,
            domain_detail_cmd,
            hyperMetroDomainIdList,
            CLI_VER_DIRECTION,
            "Quorum Server Name",
            cmdEchoDict,
        )

    def _get_hyper_metro_domain(self, cli, cmd, sn, echo_dict, res_list):
        """
        工具方法，获取domainID 信息
        :param cli:
        :param cmd:
        :param sn:
        :param echo_dict:
        :param res_list:
        :return:
        """
        flag, cli_ret, err_msg = self.update_cmd_cache(cli, cmd, echo_dict, sn)
        hyperMetroDomainInfoList = cliUtil.getHorizontalCliRet(cli_ret)
        for domain_info in hyperMetroDomainInfoList:
            domainId = domain_info.get("ID", "")
            if not domainId:
                continue
            res_list.append(domainId)

    def update_arb_net_work_hyper_metro_config(self, cli, echo_dict, sn):
        """
        查询双活仲裁信息。
        :param cli:
        :param echo_dict:
        :param sn:
        :return:
        """
        cmd = "show hyper_metro_domain general"
        cmd_nas = "show fs_hyper_metro_domain general"
        domain_id_list = []
        self._get_hyper_metro_domain(cli, cmd, sn, echo_dict, domain_id_list)
        self._get_hyper_metro_domain(cli, cmd_nas, sn, echo_dict, domain_id_list)
        self.update_cmd_cache(cli, cmd, echo_dict, sn)
        self.logger.logInfo("hyperMetroDomainIdList is %s" % domain_id_list)

        quorum_server_list = []
        cmd = "show hyper_metro_domain general domain_id=%s"
        quorum_server_list.extend(
            self._get_quorum_server(sn, cli, cmd, domain_id_list, echo_dict,
                                    "Quorum Server ID"))

        cmd = "show fs_hyper_metro_domain general domain_id=%s"
        quorum_server_list.extend(
            self._get_quorum_server(sn, cli, cmd, domain_id_list, echo_dict))

        quorum_server_list = set(quorum_server_list)
        cmd = "show quorum_server_link general server_id=%s"
        port_list = self.executeCmd(
            sn, cli, cmd, quorum_server_list, CLI_HOR_DIRECTION,
            "Local Port", echo_dict,
        )

        cmd = "show port general port_id=%s"
        port_list = list(set(port_list))
        self.logger.logInfo("port_list:{}".format(port_list))
        self.executeCmd(
            sn, cli, cmd, port_list, CLI_VER_DIRECTION,
            "ID", echo_dict,
        )

    def _get_quorum_server(self, sn, cli, cmd, domain_id_list, echo_dict,
                           server_key="Quorum ID"):
        """
        获取主仲裁ID和备仲裁ID
        :param sn:
        :param cli:
        :param cmd:
        :param domain_id_list:
        :param echo_dict:
        :return:
        """
        server_list = self.executeCmd(
            sn, cli, cmd, domain_id_list, CLI_VER_DIRECTION,
            server_key, echo_dict,
        )

        standby_server_list = self.executeCmd(
            sn, cli, cmd, domain_id_list, CLI_VER_DIRECTION,
            "Standby Quorum ID", echo_dict,
        )
        server_list.extend(standby_server_list)
        return server_list

    def executeCmd(self, sn, cli, cmd, paramList, cliDirection, outPutKey,
                   cmdEchoDict, currentProgress=None, progress=None,
                   condition=None):
        """
        @summary: 适用于依赖上一次查询的每条记录中的某一个字段作为本次cmd执行的入参。
        """
        resList = []
        stepProgress = 1
        if currentProgress and len(paramList) > 0:
            stepProgress = progress / len(paramList)

        for param in paramList:
            cmdStr = cmd % param
            if cmdStr in cmdEchoDict:
                cmdResDict = cmdEchoDict.get(cmdStr)
                flag, cli_ret, err_msg = (
                    cmdResDict.get(common.FLAG_KEY),
                    cmdResDict.get(common.CLI_RET_KEY),
                    cmdResDict.get(common.ERROR_MSG_KEY),
                )
            else:
                flag, cli_ret, err_msg = self.update_cmd_cache(
                    cli, cmdStr, cmdEchoDict, sn
                )
            if cliDirection == CLI_HOR_DIRECTION:
                objList = cliUtil.getHorizontalCliRet(cli_ret)
            elif cliDirection == CLI_HOR_DIRECTION_NO_STAND:
                objList = cliUtil.getHorizontalNostandardCliRet(cli_ret)
            else:
                objList = cliUtil.getVerticalCliRet(cli_ret)

            for obj_info in objList:
                self.set_condition_value(
                    obj_info, outPutKey, condition, resList)

            if currentProgress and len(paramList) > 0:
                currentProgress += stepProgress
                common.refreshProcess(self.env, currentProgress, self.logger)

        return resList

    def set_condition_value(self, obj_info, out_key, condition, res_list):
        """
        设置结果
        :param obj_info:
        :param out_key:
        :param condition:
        :param res_list:
        :return:
        """
        obj_attr = obj_info.get(out_key, "")
        if condition:
            key = condition[0]
            value = condition[1]
            if obj_info.get(key, "") == value:
                res_list.append(obj_attr)
        else:
            if obj_attr:
                res_list.append(obj_attr)

    def getPreCheckFileLocation(self):
        """
        @summary: get the pre - check result file base location dir
        """
        basePath = os.path.abspath(common.DIR_RELATIVE_CMD)
        pid = str(self.env.get("pid"))
        if pid:
            basePath += common.HYPERMETRO_DATA + os.sep + pid
        else:
            basePath += common.HYPERMETRO_DATA
        self.logger.logInfo("filePath is %s" % basePath)
        return basePath

    def saveObjToFile(self, devSn, objDict):
        """
        @summary: 将收集的数据序列化到文件
        """

        filePath = self.getPreCheckFileLocation()
        if not os.path.exists(filePath):
            try:
                os.makedirs(filePath)
            except Exception:
                self.logger.logError(str(traceback.format_exc()))
        filePath = filePath + os.sep + devSn
        self.logger.logInfo("file name is %s" % filePath)

        if os.path.exists(filePath):
            try:
                os.remove(filePath)
                self.logger.logInfo("saveObjToFile remove %s" % filePath)
            except Exception:
                self.logger.logError(str(traceback.format_exc()))
        try:
            with codecs.open(filePath, 'w', encoding='utf-8') as f:
                f.write(str(objDict))
        except Exception:
            self.logger.logError(str(traceback.format_exc()))

    def update_cmd_cache(self, cli, cmd, cmd_dict, sn):
        """
        执行命令并缓存数据库
        :param cli: 连接
        :param cmd: 命令
        :param cmd_dict: 回文缓存
        :param sn:
        """
        context = self.env.get("objectForPy")
        flag, cli_ret, err_msg = execute_cmd_in_cli_mode(
            context, cli, cmd, self.lang, sn, self.logger
        )
        cmd_dict[cmd] = {
            common.FLAG_KEY: flag,
            common.CLI_RET_KEY: cli_ret,
            common.ERROR_MSG_KEY: err_msg,
        }
        return flag, cli_ret, err_msg
