# -*- coding:utf-8 -*-
import resource
import cliResource
from com.huawei.ism.exception import IsmException
import re
import config

RESULT_NOCHECK = "NOCHECK"
RESULT_NOSUPPORT = "NOSUPPORT"
RESULT_WARNING = "WARNING"


def getMsg(lang, msg, args="", resource=resource.MESSAGES_DICT):
    '''
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @param resource: 消息字典
    @return: 经过国际化处理后的消息
    '''
    errMsg = "\n--"

    try:
        if not resource.has_key(msg):
            return errMsg

        localeDict = resource.get(msg)
        if not localeDict.has_key(lang):
            return errMsg

        localeMsg = localeDict.get(lang)
        if "%s" in localeMsg or "%i" in localeMsg:
            return localeMsg % args
        else:
            return localeMsg

    except:
        return errMsg

def execCmdHasEndWithSign(cli, cmd, endWithSign):
    """
    @summary: 查询执行命令时 是否含有指定结束符
    """
    if endWithSign:
        cliRet = cli.execCmd(cmd, endWithSign)
    else:
        cliRet = cli.execCmd(cmd)
    return cliRet

def isChinese(lang):
    '''
    @summary: 判断lang是否为中文
    @param lang: 语言lang
    @return: 
        True: 中文
        False: 非中文
    '''
    if lang == "zh":
        return True
    return False


def checkLineInBlackList(cliLine, lang):
    '''
    @summary: 判断cli语句行是否在黑名单中
    @param cliLine: cli语句行
    @param lang: 语言lang
    @return: 
        True: cli语句行在黑名单中
        False: cli语句行不在黑名单中
    '''
    errMsg = ""

    for dictItems in cliResource.BLACKLIST_DICT:
        if dictItems.get("key_word") in cliLine:
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (True, errMsg)
    return (False, errMsg)

def excuteCmdInCliMode(cli, cmd, isHasLog, lang, endWithSign=None):
    '''
    @summary: 获取cli模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @return: (falg, cliRet, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        cliRet: cli回显
        errMsg: 错误消息
    '''
    errMsg = ""
    cliRet = ""

    if cli is None:
        errMsg = getMsg(lang, "device.connect.abnormal")
        return (RESULT_NOCHECK, "", errMsg)

    if isHasLog:
        cliRet = execCmdHasEndWithSign(cli, cmd, endWithSign)
    else:
        if endWithSign:
            cliRet = cli.execCmdNoLog(cmd, endWithSign)
        else:
            cliRet = cli.execCmdNoLog(cmd)

    if len(cliRet) == 0:
        errMsg = getMsg(lang, "cli.result.is.empty")
        return (RESULT_NOCHECK, cliRet, errMsg)  # 修改备注：回显长度为零，说明命令执行失败，由False改为未检查

    lineList = cliRet.splitlines()
    for line in lineList:
        checkRet = checkLineInBlackList(line, lang)
        if checkRet[0]:
            return (RESULT_NOCHECK, cliRet, checkRet[1])

    return (True, cliRet, errMsg)

def parsingVerticalHorizontalCliRet(cliRet):
    '''
          解析特定的命令回显，最终解析成字典列表
         回显例子：
        -----------------------------------------------------------
        FID: 128
        SwitchType: DS
        DomainID: 25
        SwitchName: SNS5192_126
        FabricName: U7-Bigfabric
        -----------------------------------------------------------
        FID: 110
        SwitchType: BS
        DomainID: 1
        SwitchName: switch_110
        FabricName:
        -----------------------------------------------------------
          解析结果：
     [{"FID": "128", "SwitchType": "DS", "DomainID": "25","SwitchName": "SNS5192_126", "FabricName": "U7-Bigfabric"},{...}]
    
    PS：不支持正文中含有“:.>”的回显解析。
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    for line in cliRetList:
        if re.search(config.REG_USER_LOGIN_FLAG, line):
            continue
        if re.search("^-+\r*\n*$", line) and len(lineDict) > 0:
            dictList.append(lineDict.copy())
            lineDict.clear()
        fields = line.split(":")
        if len(fields) < 2:
            continue
        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")
        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)
    if len(lineDict) > 0:
        dictList.append(lineDict.copy())

    return dictList

def parsingVerticalCliRet(cliRet):
    '''
          解析特定的命令回显，最终解析成字典
         回显例子：
        FC Routing service:             enabled
        Virtual Fabric:                 enabled
          解析结果：
     {"FC Routing service": "enabled", "Virtual Fabric": "enabled"}
    PS：不支持正文中含有“:.*>”的回显解析。
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    lineDict = {}
    for line in cliRetList:
        if re.search(config.REG_USER_LOGIN_FLAG, line):
            continue
        fields = line.split(":")
        if len(fields) < 2:
            continue
        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")
        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)
    return lineDict

def getHorizontalCliRet(cliRet):
    '''
          功能：解析以===分割表头和内容的水平格式CLI回显，生成字典列表
        回显例子：
        SNS5192_126:FID128:admin> switchshow
        switchName:     SNS5192_126
        Allow XISL Use: OFF
        LS Attributes:  [FID: 128, Base Switch: No, Default Switch: Yes, Ficon Switch: No, Address Mode 0]
        
        Index Slot Port Address Media  Speed        State    Proto
        ============================================================
         256    3    0   ------   id    16G            N/A    FC  Disabled
        SNS5192_126:FID128:admin>
         返回：
    [{'Speed': '16G', 'Address': '------', 'Port': '0', 'Media': 'id', 'Index': '256', 'Proto': 'FC', 'State': 'N/A', 'Slot': '3'}, ....]
        备注：解析局限性，表头和内容为空格或中间有空格，会解析成多个。
    PS：不支持正文中含有“:.*>”的回显解析。
    '''
    i = 0
    headline = ''
    cliRetList = cliRet.encode("utf8").splitlines()
    for line in cliRetList:
        reg_headline = re.compile("^\s*=+(\s+=+)*\s*$")
        match_headline = reg_headline.search(line)
        if match_headline:
            headline = match_headline.group()
            break
        i += 1
    if headline == "" or i == 0 or i >= len(cliRetList) - 1:
        return []
    
    title = cliRetList[i - 1]
    field_words = cliRetList[(i + 1):]
    keys = title.split()
    requiredLineLen = len(keys)
    dictList = []
    for line in field_words:
        if re.search(config.REG_USER_LOGIN_FLAG, line):
            break
        #标题换行的场景
        if re.search("^-+(\s+-+)*\s*$", line):
            continue
    
        if len(line.strip()) == 0:
            continue
    
        if len(line) <= requiredLineLen:
            continue

        vals = line.split()[:requiredLineLen]
        dictList.append(dict(zip(keys, vals)))
    return dictList
