# -*- coding:utf-8 -*-
import os
import logger
import cliUtil
import re
import traceback
import threading
import time

FILE_SUFFIX = "."
INSPECT_USER = ['admin', 'root']


def getLang(py_java_env):
    '''
    @summary: 从上下文中获取lang
    @param py_java_env: 上下文对象
    @return: lang
    '''
    return py_java_env.get("lang")


def getLogger(loggerInstance, pyFilePath):
    '''
    @summary: 获取日志类
    @param loggerInstance: logger实例
    @param pyFilePath: py文件路径
    '''
    pyFileName = getBaseName(pyFilePath)
    return logger.Logger(loggerInstance, pyFileName)


def getBaseName(file_path):
    '''
    @summary: 返回文件路径的文件名，不包含后缀
    @param file_path:文件路径
    @return: 返回不包含后缀的文件名字符串
    '''
    file_name = os.path.basename(file_path)
    if FILE_SUFFIX in file_name:
        dot_index = file_name.rindex(FILE_SUFFIX)
        return file_name[0:dot_index]
    else:
        return file_name


def getMsg(lang, msg, args = ""):
    '''
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @return: 经过国际化处理后的消息
    '''
    return cliUtil.getMsg(lang, msg, args)

def isVirtualSwitchesStarted(cli, lang):
    '''
    @summary: 检查虚拟交换机是否启用
    @return: (检查是否成功, 虚拟交换机是否启用, CLI回显, 错误信息)
    '''
    errMsg = ""
    cmd = "fosconfig --show"
    (flag, cliRet, errMsg) = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag != True:
        return (False, False, cliRet, errMsg)

    cliRetDict = cliUtil.parsingVerticalCliRet(cliRet)
    if cliRetDict.get("Virtual Fabric") == "enabled":
        return (True, True, cliRet, errMsg)
    else:
        return (True, False, cliRet, errMsg)
    

def isDefaultSwitchLogin(cli, lang):
    '''
    @summary: 检查是否默认登陆的为Default switch
    @return: (检查是否成功, 虚拟交换机是否启用, CLI回显, 错误信息)
    '''
    cmd = "switchshow"
    (flag, cliRet, errMsg) = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag != True:
        return (False, False, cliRet, errMsg)
    if "Default Switch: Yes" in cliRet:
        return (True, True, cliRet, errMsg)
    else:
        return (True, False, cliRet, errMsg)

def SwitchToSpecVirtualSwitch(cli, lang, fid):
    cmd = "setcontext %s" % fid
    (flag, cliRet, errMsg) = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag != True:
        return (False, cliRet, errMsg)
    if "FID%s:" % fid in cliRet:
        return (True, cliRet, errMsg)
    else:
        errMsg = cliUtil.getMsg(lang, "switch.to.default.switch", fid)
        return (False, cliRet, errMsg)

def switchToDefaultSwitch(cli, lang):
    return SwitchToSpecVirtualSwitch(cli, lang, "128")

def getSwitchFidList(cli, lang, LOGGER):
    '''
    @summary:确认是否有虚拟交换机，如果有切换到default交换机，并返回交换机列表。
    @return: (检查是否成功, 虚拟交换机是否启用, CLI回显, 错误信息)
    '''
    allCliRet = ""
    errMsg = ""
    switchFidList = []
    (checkResult, isStarted, cliRet, errMsg) = isVirtualSwitchesStarted(cli, lang)
    allCliRet = joinLines(allCliRet, cliRet)
    if not checkResult:
        return (False, switchFidList, allCliRet, errMsg)
    if not isStarted:
        return (True, switchFidList, allCliRet, errMsg)
    (checkResult, isDefaultSwtich, cliRet, errMsg) = isDefaultSwitchLogin(cli, lang)
    allCliRet = joinLines(allCliRet, cliRet)
    if not checkResult:
        LOGGER.logInfo("Check if the default switch for logon failed.")
    if not isDefaultSwtich:
        (flag, cliRet, errMsg) = switchToDefaultSwitch(cli, lang)
        allCliRet = joinLines(allCliRet, cliRet)
        if flag != True:
            return (flag, switchFidList, allCliRet, errMsg)
    (flag, switchFidList, cliRet, errMsg) = getSwitchFidListsOneCommand(cli, lang)
    allCliRet = joinLines(allCliRet, cliRet)
    if flag != True:
        return (flag, switchFidList, allCliRet, errMsg)
            
    return (True, switchFidList, allCliRet, "")        


def getSwitchFidListsOneCommand(cli, lang):
    '''
    @summary:确认是否有虚拟交换机，如果有切换到default交换机，并返回交换机列表。
    @return: (检查是否成功, 虚拟交换机是否启用, CLI回显, 错误信息)
    '''
    allCliRet = ""
    errMsg = ""
    switchFidList = []
    cmd = "lscfg --show -n"
    (flag, cliRet, errMsg) = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    allCliRet = joinLines(allCliRet, cliRet)
    if flag != True:
        return (flag, switchFidList, allCliRet, errMsg)
    switchCfgInfoDictList = cliUtil.parsingVerticalHorizontalCliRet(cliRet)
    for switchCfgInfoDict in switchCfgInfoDictList:
        fid = switchCfgInfoDict.get("FID", "")
        if fid:
            switchFidList.append(fid)

    return (True, switchFidList, allCliRet, errMsg)


def getSwitchType(cli):
    """
    获取交换机型号公共方法
    :param cli:
    :return:
    """
    cmd = "switchshow"
    cliRet = cli.execCmdWithTimout(cmd, 300)
    retList = cliRet.splitlines()
    switchType = ''
    for ret in retList:
        if "switchType:" not in ret:
            continue

        tmpRes = re.match(r"switchType:\s+(\d+)\.", ret, re.I)
        if tmpRes:
            switchType = tmpRes.group(1)
        break

    return switchType, cliRet


def getSwitchName(cli):
    """
    获取交换机名称公共方法
    :param cli:
    :return:
    """
    cmd = "switchshow"
    cliRet = cli.execCmdWithTimout(cmd, 300)
    retList = cliRet.splitlines()
    switchName = ''
    for ret in retList:
        if "switchName:" not in ret:
            continue

        tmpRes = re.match(r"switchName:(.*)", ret, re.I)
        if tmpRes:
            switchName = tmpRes.group(1).strip()
        break

    return switchName, cliRet


def getNamalVerticalCliRetDict(cliRet):
    dataDict = {}
    retList = cliRet.splitlines()
    for line in retList:
        tmpLines = line.split(":")
        if len(tmpLines) >= 2:
            key = tmpLines[0].strip()
            value = tmpLines[1].strip()
            dataDict[key] = value
    return dataDict

def checkIsAdmin(cli):
    cmd = "userconfig --show"
    cliRet = cli.execCmdWithTimout(cmd, 300)
    dataDict = getNamalVerticalCliRetDict(cliRet)
    role = dataDict.get("Role")
    if role in INSPECT_USER:
        return True, cliRet, ''

    if role:
        return False, cliRet, ''

    role = dataDict.get("Home LF Role")
    chassisRole = dataDict.get("Chassis Role")
    if role in INSPECT_USER and chassisRole in INSPECT_USER:
        return True, cliRet, ''

    return False, cliRet, ''

def joinLines(originLines, postLines):
    """
    @summary: 将postLines追加originLines后
    """
    if not (originLines or postLines):
        return ""
    
    if not originLines:
        return postLines
    
    if not postLines:
        return originLines
    
    return "\n".join([originLines, postLines])

def refreshProcess(py_java_env, percentNumber, LOGGER):
    """
    @summary: 设置巡检当前进度
    """
    observer = py_java_env.get("progressObserver")
    try:
        if observer != None:
            observer.updateProgress(int(percentNumber))
    except:
        LOGGER.logInfo(str(traceback.format_exc()))

class AsynProgress(threading.Thread):
    """
    异步刷新线程
    当某个命令执行需要很久时，考虑使用。刷新时间定为0.1秒能更好、更快的反应关掉线程。
    可选参数说明：
    currentProgress 当前起始进度。
    endProgress：预计结束进度。
    """
    def __init__(self, PY_JAVA_ENV, LOGGER, currentProgress = 1.0, endProgress = 99, sleepTime = 0.1):
        threading.Thread.__init__(self)
        self.stopFlg = False
        self.endProgress = endProgress
        self.currentProgress = currentProgress
        self.PY_JAVA_ENV = PY_JAVA_ENV
        self.LOGGER = LOGGER
        self.sleepTime = sleepTime
    def run(self):
        self.LOGGER.logInfo("start threading progress!")
        tmp = 1
        while self.currentProgress <= self.endProgress and not self.stopFlg:
            if tmp >= 1:
                refreshProcess(self.PY_JAVA_ENV, self.currentProgress, self.LOGGER)
                tmp = 0
            self.currentProgress += 0.1
            tmp += 0.1
            time.sleep(self.sleepTime)
    def setStopFlag(self, stopFlag):
        self.stopFlg = stopFlag


def switchshowCliRetAnalysisMethod(cliRet, switchTypeHigh, LOGGER):
    """
    :param cliRet:
    :param switchType: True 高端、False 中低端
    :return:
    """
    # 交换机信息
    switchInfoDict = {}
    # 端口信息
    portInfoDictList = []

    # 先解析垂直
    verticalCliRet = cliRet[0:cliRet.index("========")]
    rex = re.compile(r"(.*?):(.*)")
    verLines = verticalCliRet.splitlines()
    for line in verLines:
        res = rex.match(line)
        if res:
            switchInfoDict[res.group(1).strip()] = res.group(2).strip()

    # 解析横向
    horCliRet = cliRet[cliRet.index("========"):]
    titleKey = ["Index", "Port", "Address", "Media", "Speed", "State", "Proto"]
    tmpLines = cliRet.splitlines()
    titleLine = ''
    for line in tmpLines:
        for key in titleKey:
            if key not in line:
                break
            titleLine = line

    titleKeys = titleLine.split()
    LOGGER.logInfo("titleKeys:%s" % str(titleKeys))

    horCliRet = cliRet[cliRet.index("========"):]
    resLines = horCliRet.splitlines()
    if switchTypeHigh:
        for line in resLines:
            portInfoDict = {}
            valuesList = line.split()
            # 高端的值都是大于等于8的，小于的不关注
            if len(valuesList) < 8:
                LOGGER.logInfo("no need to care line:%s" % str(valuesList))
                continue
            for i in range(len(titleKeys)):
                portInfoDict[titleKeys[i]] = valuesList[i]

            if len(valuesList) > 8:
                portInfoDict["otherInfo"] = " ".join(valuesList[8:])
            else:
                portInfoDict["otherInfo"] = ""
            portInfoDictList.append(portInfoDict)
    else:
        for line in resLines:
            portInfoDict = {}
            valuesList = line.split()
            # 中低端的值都是大于等于7的，小于的不关注
            if len(valuesList) < 7:
                LOGGER.logInfo("no need to care line:%s" % str(valuesList))
                continue
            for i in range(len(titleKeys)):
                portInfoDict[titleKeys[i]] = valuesList[i]

            if len(valuesList) > 7:
                portInfoDict["otherInfo"] = " ".join(valuesList[7:])
            else:
                portInfoDict["otherInfo"] = ""

            portInfoDict["Slot"] = "--"
            portInfoDictList.append(portInfoDict)

    return portInfoDictList, switchInfoDict

"""解析switchshow回文的接口：根据回文自动判断是否为高端设备，回文解析中自适应"""
def switchshowCliRetAnalysis(cliRet, LOGGER):
    """
    :param cliRet:
    :return:
    """
    # 交换机信息
    switchInfoDict = {}
    # 端口信息
    portInfoDictList = []

    # 先解析垂直
    verticalCliRet = cliRet[0:cliRet.index("========")]
    rex = re.compile(r"(.*?):(.*)")
    verLines = verticalCliRet.splitlines()
    for line in verLines:
        res = rex.match(line)
        if res:
            switchInfoDict[res.group(1).strip()] = res.group(2).strip()

    # 解析横向
    horCliRet = cliRet[cliRet.index("========"):]
    titleKeys = set(["Index", "Port", "Address", "Media", "Speed", "State", "Proto"])
    tmpLines = cliRet.splitlines()
    titleLine = ''
    for line in tmpLines:
        if titleKeys.issubset(set(line.split())):
            titleLine = line
            break

    #更新keys
    titleKeys = titleLine.split()
    LOGGER.logInfo("titleKeys:%s" % str(titleKeys))

    horCliRet = cliRet[cliRet.index("========"):]
    resLines = horCliRet.splitlines()

    # 中低端的值都是大于等于7的，小于的不关注
    # 高端的值都是大于等于8的，小于的不关注
    if "Index Slot" in cliRet:
        validValueCount = 8
    else:
        validValueCount = 7

    for line in resLines:
        portInfoDict = {}
        #部分设备无Slot字段
        portInfoDict["Slot"] = "--"
        valuesList = line.split()
        if len(valuesList) < validValueCount:
            LOGGER.logInfo("no need to care line:%s" % str(valuesList))
            continue
        for i in range(len(titleKeys)):
            portInfoDict[titleKeys[i]] = valuesList[i]

        if len(valuesList) > validValueCount:
            portInfoDict["otherInfo"] = " ".join(valuesList[validValueCount:])
        else:
            portInfoDict["otherInfo"] = ""
        portInfoDictList.append(portInfoDict)

    return portInfoDictList, switchInfoDict


def analysisNsshowTMethod(cliRet, LOGGER):
    cliRetList = cliRet.splitlines()
    index = 0
    headLineIndexList = []
    for line in cliRetList:
        reg_headline = re.compile("^\s*N(\s+)\w*;\s+\w+;")
        match_headline = reg_headline.search(line)
        if match_headline:
            headline = match_headline.group()
            headLineIndexList.append(index)
        index += 1

    dataDictList = []
    for i in range(len(headLineIndexList)):
        dataDictInfo = {}
        if i == len(headLineIndexList) - 1:
            cliInfoRetList = cliRetList[headLineIndexList[i]:]
        else:
            cliInfoRetList = cliRetList[headLineIndexList[i] + 1:headLineIndexList[i + 1] - 1]
        for line in cliInfoRetList:
            line = line.strip()
            if not line:
                continue

            regx = re.compile(r"(.*?):(.*)")
            tmpList = regx.findall(line)
            for data in tmpList:
                if len(data) < 2:
                    continue

                key = data[0].strip()
                value = data[1].strip()
                dataDictInfo[key] = value

        if not dataDictInfo:
            continue

        dataDictList.append(dataDictInfo)

    return dataDictList

def getResult(result1, result2):
    resultList = [False, cliUtil.RESULT_NOCHECK, cliUtil.RESULT_WARNING, True, cliUtil.RESULT_NOSUPPORT]
    for r in resultList:
        if result1 is r:
            return result1

        if result2 is r:
            return result2
    #默认返回未检查
    return cliUtil.RESULT_NOCHECK

class UnCheckException(Exception):
    """
    @summary: 未检查异常自定义类
    """
    def __init__(self, errorMsg, cliRet, flag=cliUtil.RESULT_NOCHECK):
        self.errorMsg = errorMsg
        self.cliRet = cliRet
        self.flag = flag