# coding: UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil
from com.huawei.ism.exception import IsmException

import common
from ds_rest_util import CommonRestService
from utils import Products

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "checkBucketTierDeletionPolicy"

NAMESPACE_COUNT_UPPER_LIMIT = 2400


def get_count_from_rsp_json(rsp_json):   
    count = rsp_json.get("data").get("count")
    return count


def query_product_ver(rest, base_uri):
    LOGGER.logInfo("start to query hotpatch version.")
    cmd_str = "{}/api/v2/cluster/product".format(base_uri)
    product_result = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    return product_result.get("data", {}).get("hotpatch_version", ""), product_result.get("data", {}).get("version", "")


def check_storage_version(rest, base_uri):
    version_815 = "8.1.5"
    version_820 = "8.2.0"
    hotpatch_version, version = query_product_ver(rest, base_uri)
    LOGGER.logInfo("hotpatch_version is: {} version is: {}".format(hotpatch_version, version))
    if not hotpatch_version and (version == version_820 or version == version_815):
        return False, version

    if hotpatch_version.startswith(version_815):
        compare = Products.compareVersion(hotpatch_version, "8.1.5.SPH052")
        LOGGER.logInfo("hotpatch_version is {}, compare result is {}".format(hotpatch_version, compare))
        if compare >= 0:
            return True, hotpatch_version
        else:
            return False, hotpatch_version
    elif hotpatch_version.startswith(version_820):
        compare = Products.compareVersion(hotpatch_version, "8.2.0.SPH056")
        LOGGER.logInfo("hotpatch_version is {}, compare result is {}".format(hotpatch_version, compare))
        if compare >= 0:
            return True, hotpatch_version
        else:
            return False, hotpatch_version
    return True, hotpatch_version


def check_node_in_pools(services_on_pool_nodes):
    for services_on_pool_node in services_on_pool_nodes:
        for node_ip in services_on_pool_node:
            obj_service = services_on_pool_node[node_ip].get("object_service")
            if obj_service != -1:
                return True
    return False


def check_block_obs_service(rest, base_uri):
    is_block = False
    is_obs = False
    cmd_str = "{}/api/v2/cluster_service/converged_eds_service".format(base_uri)
    rsp_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    service_of_pools = rsp_json.get("data").get("service_of_pools")
    if not service_of_pools:
        return True, is_obs
    for service_of_pool in service_of_pools:
        is_obs_and_hdfs = service_of_pool.get("is_obs_and_hdfs")
        is_obs_and_file = service_of_pool.get("is_obs_and_file")
        if is_obs_and_file == 1 or is_obs_and_hdfs == 1:
            return is_block, True
        services_on_pool_nodes = service_of_pool.get("services_on_pool_nodes")
        if check_node_in_pools(services_on_pool_nodes):
            return is_block, True
    return is_block, False


def execute(rest):
    """
    检查桶和分级特性删除策略数量
    """
    LOGGER.logInfo("Start to check number of buckets and number of tier deletion policies")
    info_list = []
    err_list = []
    dev_node = py_java_env.get("devInfo")
    observer = py_java_env.get("progressObserver")
    progress_map = {}
    return_result = common.INSPECT_PASS
    try:
        progress_map[ITEM_ID] = 1
        observer.updateProgress(progress_map)
        base_uri = RestUtil.getDstorageUrlHead(dev_node)

        # 1、检查补丁版本是否是
        #   OceanStor Pacific 8.1.5 SPH052之前的所有8.1.5版本
        #   OceanStor Pacific 8.2.0 SPH056之前的所有8.2.0版本
        check_version_result, version = check_storage_version(rest, base_uri)
        # 软件版本不在风险版本列表内，没有风险
        if check_version_result:
            return common.INSPECT_NOSUPPORT, "Software version not involved", ""

        # 2、检查是否是块池、打开对象服务
        is_block, is_obs = check_block_obs_service(rest, base_uri)
        if is_block:
            msg = common.get_err_msg(LANG, "block.not.involved")
            return common.INSPECT_NOSUPPORT, msg, msg
        if not is_obs:
            return common.INSPECT_PASS, "Object service is not enabled.", ""

        # 3、查询命名空间个数（即桶数量）
        cmd_str = "{}/api/v2/converged_service/namespaces_count".format(base_uri)
        rsp_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
        namespace_count = get_count_from_rsp_json(rsp_json)

        # 4、查询分级特性删除策略个数
        cmd_str = "{}/api/v2/tier_service/tier_deletion_policy_count".format(base_uri)
        rsp_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
        policy_count = get_count_from_rsp_json(rsp_json)

        info_list.append("Number of buckets(namespaces)   : {}".format(namespace_count))
        info_list.append("Number of tier deletion policies: {}".format(policy_count))
        if namespace_count > NAMESPACE_COUNT_UPPER_LIMIT and policy_count > 0:
            return_result = common.INSPECT_UNNORMAL
            err_list.append("The number of buckets(namespaces) is more than {}".format(NAMESPACE_COUNT_UPPER_LIMIT))
            err_list.append("The number of tier deletion policies is more than 0")
            err_list.append(
                "In the current version {}, there may be too many buckets and tiering deletion policies."\
                    .format(version)
            )
        progress_map[ITEM_ID] = 80
        observer.updateProgress(progress_map)
        
        return (
            return_result,
            "\n".join(info_list),
            "\n".join(err_list)
        )
    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        return (
            common.INSPECT_UNNORMAL,
            "",
            common.get_err_msg(LANG, "query.result.abnormal"),
        )