# coding: UTF-8
import re
import string

import common
import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil
from com.huawei.ism.exception import IsmException
from ds_rest_util import CommonRestService

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "checkFtpServiceStatus"


def execute(rest):
    """
    步骤1:检查产品的版本,如果不是815或820,则检查结果为不涉及,结束;否则存在风险,继续;
    步骤2:检查补丁版本,如果无补丁,或者补丁是815SPH036或820SPH026之前的版本,有风险,继续;
    步骤3:通过rest接口检查FTP服务状态,如果开启,则有风险,检查不通过;否则检查通过。
    :param env:
    :return:
    """
    dev_node = py_java_env.get("devInfo")
    try:
        # 块池不涉及该问题
        if is_block_pool(rest, dev_node):
            return common.INSPECT_NOSUPPORT, "block not involved", ""

        if not is_open_converged_service(rest, dev_node):
            return common.INSPECT_NOSUPPORT, "not open converged service, not involved", ""

        product_version = dev_node.getProductVersion()
        if "8.1.5" not in product_version and "8.2.0" not in product_version:
            LOGGER.logInfo("product version is not 815 or 820,no need to check")
            return common.INSPECT_NOSUPPORT, "Software version: {}".format(product_version), ""

        patch_version = dev_node.getDistributedHotPatchVersion()
        software_version = get_version(product_version, patch_version)
        if check_is_risk_version(patch_version):
            if is_ftp_servive_status_on(rest, dev_node):
                return common.INSPECT_UNNORMAL, \
                       "The FTP service of the current version {} may be abnormal.".format(software_version), ""
            else:
                return common.INSPECT_PASS, \
                       "CHECK PASS! FTP service off. Software version: {}".format(software_version), ""
        else:
            return common.INSPECT_PASS, "CHECK PASS! Software version: {}".format(software_version), ""
    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        return (
            common.INSPECT_UNNORMAL, "", common.get_err_msg(LANG, "query.result.abnormal"),
        )


def get_version(product_version, patch_version):
    """
    :function: 获取存储软件版本
    :param version: 产品版本,补丁版本
    :return: 存储软件版本
    """
    if patch_version in product_version:
        return product_version
    elif product_version in patch_version:
        return patch_version
    elif not patch_version:
        return product_version
    else:
        return product_version + "." + patch_version


def check_is_risk_version(patch_version):
    """
    :function: 检查是否是有风险的补丁版本,815SPH036或820SPH026之前的版本有风险
    :param: 补丁版本
    :return: 是否有风险
    """
    if not patch_version:
        LOGGER.logInfo("GA version, no hotpatch")
        return True

    LOGGER.logInfo("patch version %s" % patch_version)
    patch = re.search(r"SPH(\d+)", patch_version)
    if not patch:
        return False

    patch_num = int(patch.group(1))
    if "8.1.5" in patch_version and patch_num < 36:
        return True
    if "8.2.0" in patch_version and patch_num < 26:
        return True
    return False


def is_ftp_servive_status_on(rest, dev_node):
    base_uri = RestUtil.getDstorageUrlHead(dev_node)
    cmd_str = "{}/api/v2/nas_protocol/ftp_service".format(base_uri)
    ftp_cfg_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    LOGGER.logInfo("ftp cfg json %s" % ftp_cfg_json)
    ftp_status = ftp_cfg_json.get("data").get("status")
    LOGGER.logInfo("ftp status %s " % ftp_status)
    if "off" in ftp_status:
        # 没有开启ftp服务
        return False
    else:
        return True


def is_block_pool(rest, dev_node):
    base_uri = RestUtil.getDstorageUrlHead(dev_node)
    cmd_str = "{}/dsware/service/cluster/storagepool/queryStoragePool".format(base_uri)
    pool_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    for para in pool_json['storagePools']:
        service_type = para.get('serviceType')
        if service_type == 1:
            return True
    return False


def is_open_converged_service(rest, dev_node):
    base_uri = RestUtil.getDstorageUrlHead(dev_node)
    cmd_str = "{}/dsware/service/resource/queryStoragePool".format(base_uri)
    file_proto_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    for para in file_proto_json['storagePools']:
        services_created = para.get('servicesCreated')
        services_num = len(services_created)
        if services_num == 0:
            return False
        for service_type in services_created:
            if service_type == 1:
                return False
    return True
