# coding: UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

import common
from ds_rest_util import CommonRestService
from com.huawei.ism.exception import IsmException
import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "checkMultiDCVersion"
ERR_CODE_SINGLE_DC_IGNORE = "33561761"


def get_storage_software(dev_node):
    """
    :function: 获取存储软件版本
    :param version: 节点信息
    :return: 存储软件版本
    """
    product_version = dev_node.getProductVersion()
    patch_version = dev_node.getDistributedHotPatchVersion()

    if patch_version in product_version:
        return product_version
    elif product_version in patch_version:
        return patch_version
    elif not patch_version:
        return product_version
    else:
        return product_version + "." + patch_version


def check_storage_software(version):
    """
    :function: 存储软件风险配套检查
    :param version: 当前存储软件版本
    :return: True:检查通过; False:检查不通过
    """
    black_product_ver = (
        "8.1.5", "8.1.5.SPH001", "8.1.5.SPH005", "8.1.5.SPH010", "8.1.5.SPH015", "8.1.5.SPH017", "8.1.5.SPH019",
        "8.1.5.SPH021",
    )
    # 非风险版本与存储软件配套检查
    return version not in black_product_ver


def check_object_service(rest, base_uri):
    """
    :function: 检查是否开启对象服务
    :param base_uri: 请求的URI
    :param rest: 集群rest链接
    :return:
    """
    LOGGER.logInfo("start to check object service.")
    cmd_str = "{}/dfv/service/obsCommonCfg/initSystemOperationRecord".format(base_uri)
    response_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    result = response_json.get("result")
    if result != 0:
        raise Exception("initSystemOperationRecord.result.abnormal[{}]".format(result))

    status = response_json.get("data", {}).get("status", "")
    if "finished" == status:
        LOGGER.logInfo("False status is %s" % status)
        return False
    else:
        LOGGER.logInfo("True status is %s" % status)
        return True


def execute(rest):
    """
    检查多DC版本
    :param env:
    :return:
    """
    LOGGER.logInfo("checkMultiDCVersion begin ...")
    dev_node = py_java_env.get("devInfo")
    try:
        product_version = get_storage_software(dev_node)
        software_result = check_storage_software(product_version)
        if software_result:
            LOGGER.logInfo('checkMultiDCVersion end, not involved ! product_version = {}'.format(product_version))
            return common.INSPECT_NOSUPPORT, "Not Involved ! product_version = {}".format(
                product_version), ""  # 软件版本不在风险版本列表内，没有风险

        base_uri = RestUtil.getDstorageUrlHead(dev_node)
        object_service = check_object_service(rest, base_uri)
        if object_service:
            return common.INSPECT_NOSUPPORT, "The object service is not enable.", "" # 未使用对象服务，没有风险

        cmd_str = "{}/api/v2/rgm/topo_sites".format(base_uri)
        LOGGER.logInfo('request cmd_str:%s', cmd_str)

        response_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
        LOGGER.logInfo('response_json:%s', str(response_json))

        data_len = len(response_json.get("data", []))
        LOGGER.logInfo('data_len {}'.format(data_len))
        if data_len >= 2:
            # 环境是多DC环境，需要整改，巡检不通过
            LOGGER.logInfo('checkMultiDCVersion end, check fail')
            return common.INSPECT_UNNORMAL, "Check Fail! The environment is multi-DC", ""
        else:
            # 环境是单集群环境，没有风险，巡检通过
            LOGGER.logInfo('checkMultiDCVersion end, check success')
            return common.INSPECT_PASS, "Check Pass! The environment is single cluster", ""
    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        err_code = exception.getErrorId()
        # ERR_CODE_SINGLE_DC_IGNORE表示环境是单集群，没有风险，巡检通过
        if ERR_CODE_SINGLE_DC_IGNORE in str(err_code):
            return common.INSPECT_PASS, "Check Pass! The environment is single cluster", "" 
        return (
            common.INSPECT_UNNORMAL, "", common.get_err_msg(LANG, "query.result.abnormal"),
        )
