# coding:utf-8
# Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

import common
import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil

from com.huawei.ism.exception import IsmException
from ds_rest_util import CommonRestService
from utils import Products

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "checkMutiClusterAndObsService"


def execute(rest):
    """
    检查是否是多集群并开启了对象服务
    :param rest: 集群rest链接
    :return:
    """
    LOGGER.logInfo('checkMutiClusterAndObsService start')
    ret_list = []
    dev_node = py_java_env.get("devInfo")
    observer = py_java_env.get("progressObserver")
    try:
        base_uri = RestUtil.getDstorageUrlHead(dev_node)
        # 检查此巡检项涉及的存储版本
        software_result = check_storage_software(dev_node, rest, base_uri)
        if software_result:
            return common.INSPECT_NOSUPPORT, "Software version not involved", ""  # 软件版本不在风险版本列表内，没有风险
        object_service = check_object_service(rest, base_uri)
        if object_service:
            return common.INSPECT_NOSUPPORT, "The object service is not enable.", ""  # 未使用对象服务，没有风险

        cmd_str = "{}/dfv/service/LS/lz/".format(
            base_uri
        )
        # 打印日志 请求
        LOGGER.logInfo("get_cluster_information, request cmd_str:%s", cmd_str)

        response_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
        response_json_str = str(response_json)
        # 打印日志 响应
        LOGGER.logInfo("get_cluster_information, response_json:%s", response_json_str)

        result = response_json.get("result")
        if result != 0:
            ret_list.append("get_cluster_information error, please contact technical support engineers! result_code :")
            ret_list.append(result)
            return common.INSPECT_UNNORMAL, "get_cluster_information error", ""

        datalist = response_json.get("data", [])
        datalist_len = len(datalist)
        if datalist_len >= 2:
            ret_list.append("The environment is multiple clusters, the number is :")
            ret_list.append(datalist_len)
            # 环境是多集群环境, 需要整改，巡检不通过
            return common.INSPECT_UNNORMAL, "The environment is multiple clusters", ""
        else:
            # 环境为单集群环境, 没有风险，巡检通过
            return common.INSPECT_PASS, "The environment is single cluster", ""
    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        return (
            common.INSPECT_UNNORMAL,
            exception,
            common.get_err_msg(LANG, "query.result.abnormal"),
        )


def check_object_service(rest, base_uri):
    """
    :function: 检查是否开启对象服务
    :param base_uri: 请求的URI
    :param rest: 集群rest链接
    :return:
    """
    LOGGER.logInfo("start to check object service.")
    cmd_str = "{}/dfv/service/obsCommonCfg/initSystemOperationRecord".format(base_uri)
    response_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    result = response_json.get("result")
    if result != 0:
        raise Exception("initSystemOperationRecord.result.abnormal[{}]".format(result))

    status = response_json.get("data", {}).get("status", "")
    if "finished" == status:
        LOGGER.logInfo("False status is %s" % status)
        return False
    else:
        LOGGER.logInfo("True status is %s" % status)
        return True


def check_storage_software(dev_node, rest, base_uri):
    """
    :function: 检查存储软件版本，813版本，815SPH036之前，820SPH005之前的版本为风险版本
    :param dev_node: 集群信息
    :param base_uri: 请求的URI
    :param rest: 集群rest链接
    :return:
    """
    product_version = dev_node.getProductVersion()
    patch_version = query_product_hotpatch_ver(rest, base_uri)
    LOGGER.logInfo("product_version is:%s/%s" % (product_version, patch_version))

    # 813版本均涉及
    black_product_ver_813 = ("8.1.RC5", "8.1.RC6", "8.1.3",)
    # 815SPH036及之后的版本不涉及
    black_product_ver_815 = ("8.1.RC7", "8.1.5",)
    # 820SPH005及之后的版本不涉及
    black_product_ver_820 = ("8.2.0",)

    for black_ver in black_product_ver_813:
        if product_version.startswith(black_ver):
            LOGGER.logInfo("product_version is 813")
            return False

    for black_ver in black_product_ver_815:
        if product_version.startswith(black_ver):
            compare = Products.compareVersion(patch_version, "8.1.5.SPH036")
            LOGGER.logInfo("product_version is 815, compare is %s" % compare)
            if compare >= 0:
                return True
            else:
                return False

    for black_ver in black_product_ver_820:
        if product_version.startswith(black_ver):
            compare = Products.compareVersion(patch_version, "8.2.0.SPH005")
            LOGGER.logInfo("product_version is 820, compare is %s" % compare)
            if compare >= 0:
                return True
            else:
                return False
    return True


def query_product_hotpatch_ver(rest, base_uri):
    cmd_str = "{}/api/v2/cluster/product".format(base_uri)
    product_result = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    return product_result.get("data", {}).get("hotpatch_version", "")