# coding: UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

import time
import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil
from com.huawei.ism.exception import IsmException

import common
from ds_rest_util import CommonRestService
from utils import Products

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "checkTierDeletionPolicyPath"

abnomal_policies = []


def get_count_from_rsp_json(rsp_json):
    result = rsp_json.get("result")
    if not result:
        return False, 0
    if "code" in result:
        code = result.get("code")
        if not isinstance(code, int) or code != 0:
            return False, 0
    else:
        return False, 0
    
    data = rsp_json.get("data")
    if not data:
        return False, 0
    if "count" in data:
        count = data.get("count")
        if not isinstance(count, int):
            return False, 0
    else:
        return False, 0
    return True, count


def query_product_ver(rest, base_uri):
    LOGGER.logInfo("start to query hotpatch version.")
    cmd_str = "{}/api/v2/cluster/product".format(base_uri)
    product_result = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    return product_result.get("data", {}).get("hotpatch_version", ""), product_result.get("data", {}).get("version", "")


def check_storage_version(rest, base_uri):
    version_813 = "8.1.3"
    version_815 = "8.1.5"
    version_820 = "8.2.0"
    hotpatch_version, version = query_product_ver(rest, base_uri)
    LOGGER.logInfo("hotpatch_version is: {} version is: {}".format(hotpatch_version, version))
    
    if version == version_813:
        return False, version
    
    if not hotpatch_version and (version == version_820 or version == version_815):
        return False, version
    
    if hotpatch_version.startswith(version_815):
        compare = Products.compareVersion(hotpatch_version, "8.1.5.SPH036")
        LOGGER.logInfo("hotpatch_version is {}, compare result is {}".format(hotpatch_version, compare))
        if compare >= 0:
            return True, hotpatch_version
        else:
            return False, hotpatch_version
    elif hotpatch_version.startswith(version_820):
        compare = Products.compareVersion(hotpatch_version, "8.2.0.SPH026")
        LOGGER.logInfo("hotpatch_version is {}, compare result is {}".format(hotpatch_version, compare))
        if compare >= 0:
            return True, hotpatch_version
        else:
            return False, hotpatch_version

    return True, hotpatch_version


def check_policies_path(rsp_json):
    policies = rsp_json.get("data")
    for policy in policies:
        path = policy.get("path_name")
        LOGGER.logInfo("policy path: {}".format(path))
        if len(path) >= 8:
            name = policy.get("name")
            LOGGER.logInfo("policy name: {}".format(name))
            abnomal_policies.append("Policy Name: {}\nPolicy Path: {}".format(name, path))


def check_long_path(rest, base_uri, policy_count):
    offset = 0
    limit = 400
    while offset < policy_count:
        cmd_str = "{}/api/v2/tier_service/tier_deletion_policies?range=%7B%22offset%22%3A{}%2C%22limit%22%3A{}%7D"\
            .format(base_uri, offset, limit)
        rsp_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
        LOGGER.logInfo('get deletion policies info result:{}'.format(rsp_json))
        check_policies_path(rsp_json)
        offset = offset + limit
        time.sleep(0.1)


def check_block_service(rest, base_uri):
    cmd_str = "{}/api/v2/cluster_service/converged_eds_service".format(base_uri)
    rsp_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    service_of_pools = rsp_json.get("data").get("service_of_pools")
    if not service_of_pools:
        return True
    return False


def execute(rest):
    """
    检查分级特性删除策略的路径长度
    """
    LOGGER.logInfo("Start to check number of buckets and number of tier deletion policies")
    dev_node = py_java_env.get("devInfo")
    observer = py_java_env.get("progressObserver")
    progress_map = {}
    try:
        progress_map[ITEM_ID] = 1
        observer.updateProgress(progress_map)
        base_uri = RestUtil.getDstorageUrlHead(dev_node)

        # 1、检查补丁版本是否是
        #   OceanStor Pacific 8.1.3
        #   OceanStor Pacific 8.1.5 SPH036之前的所有8.1.5版本
        #   OceanStor Pacific 8.2.0 SPH026之前的所有8.2.0版本
        check_version_result, version = check_storage_version(rest, base_uri)
        # 软件版本不在风险版本列表内，没有风险
        if check_version_result:
            return common.INSPECT_NOSUPPORT, "Software version not involved", ""

        # 1.1 检查是否块池，块池直接跳过
        is_block = check_block_service(rest, base_uri)
        if is_block:
            msg = common.get_err_msg(LANG, "block.not.involved")
            return common.INSPECT_NOSUPPORT, msg, msg

        # 2、查询分级特性删除策略个数
        cmd_str = "{}/api/v2/tier_service/tier_deletion_policy_count".format(base_uri)
        rsp_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
        LOGGER.logInfo('get deletion policy count info result:{}'.format(rsp_json))
        ret, policy_count = get_count_from_rsp_json(rsp_json)
        if not ret:
            return (
                common.INSPECT_UNNORMAL,
                "get namespace count info failed\n",
                common.get_err_msg(LANG, "query.result.abnormal")
            )
        
        progress_map[ITEM_ID] = 30
        observer.updateProgress(progress_map)

        # 3、循环检查路径长度大于等于 8 的删除策略
        check_long_path(rest, base_uri, policy_count)
        
        progress_map[ITEM_ID] = 80
        observer.updateProgress(progress_map)
        if not abnomal_policies:
            return (
                common.INSPECT_PASS,
                "CHECK PASS",
                ""
            )
        return (
            common.INSPECT_UNNORMAL,
            "\n".join(abnomal_policies),
            "In the current version {}, the deletion policy of tier service may be abnormal.".format(version) +
                "\nThe eds process will restart during the patch upgrade, which may trigger this problem. Therefore, " +
                "you are advised to delete the abnormal tier deletion policy before the upgrade."
        )
        
    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        return (
            common.INSPECT_UNNORMAL,
            "",
            common.get_err_msg(LANG, "query.result.abnormal"),
        )