# coding:utf-8
# Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

import common
import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil

from com.huawei.ism.exception import IsmException
from ds_rest_util import CommonRestService
from utils import Products

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "checkTierPolicyForHeterogeneousDevices"


def execute(rest):
    LOGGER.logInfo('checkTierPolicyForHeterogeneousDevices start')
    dev_node = py_java_env.get("devInfo")

    try:
        base_uri = RestUtil.getDstorageUrlHead(dev_node)
        check_version_result = check_storage_version(rest, base_uri)
        # 软件版本不在风险版本列表内，没有风险
        if check_version_result:
            return common.INSPECT_NOSUPPORT, "Software version not involved", ""

        # 查看是否有节点开启对象服务
        object_service = check_object_service(rest, base_uri)
        if object_service:
            return common.INSPECT_NOSUPPORT, "The object service is not enable in this environment.", ""

        # 发起rest请求查询分级特性迁移策略
        check_tier_result = query_tier_migration_policies(rest, base_uri)
        if not check_tier_result:
            return common.INSPECT_NOSUPPORT, "Tier migrate service is not enable in this environment.", ""

        for strategy in check_tier_result:
            if strategy == 3:
                return (common.INSPECT_UNNORMAL, "Tier migrate policy is heterogeneous devices",
                        common.get_err_msg(LANG, "check.tier.policy.for.heterogeneous.devices.not.pass"))
        return common.INSPECT_PASS, "Tier migration not involved for heterogeneous devices", ""
    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        return (
            common.INSPECT_UNNORMAL,
            exception,
            common.get_err_msg(LANG, "query.result.abnormal"),
        )


def check_storage_version(rest, base_uri):
    version_815 = "8.1.5"
    hotpatch_version, version = query_product_ver(rest, base_uri)
    LOGGER.logInfo("hotpatch_version is: %s version is: %s" % (hotpatch_version, version))
    if not hotpatch_version and version_815 == version:
        return False

    # 热补丁仅涉及8.1.5.SPH025以前的版本
    if hotpatch_version.startswith(version_815):
        compare = Products.compareVersion(hotpatch_version, "8.1.5.SPH025")
        LOGGER.logInfo("hotpatch_version is 815, compare is %s" % compare)
        if compare >= 0:
            return True
        else:
            return False
    return True


def check_object_service(rest, base_uri):
    """
    :function: 检查是否开启对象服务
    :param base_uri: 请求的URI
    :param rest: 集群rest链接
    :return:
    """
    LOGGER.logInfo("start to check object service.")
    cmd_str = "{}/dfv/service/obsCommonCfg/initSystemOperationRecord".format(base_uri)
    response_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    result = response_json.get("result")
    if result != 0:
        LOGGER.logInfo("check object service result abnormal!")
        raise Exception("initSystemOperationRecord.result.abnormal[{}]".format(result))

    status = response_json.get("data", {}).get("status", "")
    if "finished" == status:
        LOGGER.logInfo("False status is %s" % status)
        return False
    else:
        LOGGER.logInfo("True status is %s" % status)
        return True


def query_product_ver(rest, base_uri):
    LOGGER.logInfo("start to query hotpatch version.")
    cmd_str = "{}/api/v2/cluster/product".format(base_uri)
    product_result = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    return product_result.get("data", {}).get("hotpatch_version", ""), product_result.get("data", {}).get("version", "")


def query_tier_migration_policies(rest, base_uri):
    LOGGER.logInfo("start to query smart tier migration policies.")
    cmd_str = "{}/api/v2/tier_service/tier_migrate_policies".format(base_uri)
    tier_result = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    data = tier_result.get("data", [])
    if not data:
        return []
    else:
        strategy_list = [item.get("strategy") for item in data if "strategy" in item]
        return strategy_list
