# coding: UTF-8

import time
import traceback

import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil
from com.huawei.ism.exception import IsmException

import common
from ds_rest_util import CommonRestService

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "check_cache_watermark_for_scale"
CHECK_TIMES = 10 # 水位检查次数
WATER_MARK_THRESHOLD_BLOCK_POOL = 80
WATER_MARK_THRESHOLD_CONVERGED_POOL = 60

ret_list = [] # 用于记录所有的请求信息 出现异常时打印
water_mark_dic = {} # 用于记录水位查询结果 累计值 key为节点lsid


def get_ip_lsid_mapping(rest, base_uri, pool_id):
    """ 查询lsid与Ip的对应关系  """
    ip_lsid_map = {}
    ip_lsid_mapping_url = "{}/api/v2/data_service/eds_service?poolId={}" \
        .format(base_uri, pool_id) # 对应cli命令 show eds_service status
    ret_list.append(ip_lsid_mapping_url)
    rsp_json = CommonRestService.exec_get_gor_big_by_ds(rest, ip_lsid_mapping_url)
    ret_list.append(rsp_json)
    result = rsp_json.get("result")
    if result != 0:
        LOGGER.logWarning("get eds_service status failed. " + str(ret_list))
        return False, {}
    datas = rsp_json.get("details")
    if not datas:
        LOGGER.logWarning("get eds_service status failed." + str(ret_list))
        return False, {}
    for data in datas:
        lsid = data.get("lsId")
        node_ip = data.get("nodeIp")
        if not lsid or not node_ip:
            LOGGER.logWarning("eds_service status: lsid or nodeId is null. " + str(ret_list))
            continue
        ip_lsid_map[str(lsid)] = node_ip
    return True, ip_lsid_map


def get_node_cache_watermarks(rest, base_uri, pool_id):
    if pool_id not in water_mark_dic:
        water_mark_dic[pool_id] = {}
    watermark_show_url = "{}/api/v2/data_service/cache_perf_batch?type=16&param_type=31&pool_id={}" \
        .format(base_uri, pool_id) # 对应命令 show cache water_mark
    ret_list.append(watermark_show_url)
    rsp_json = CommonRestService.exec_get_gor_big_by_ds(rest, watermark_show_url)
    ret_list.append(str(rsp_json))
    result = rsp_json.get("result")
    if result != 0:
        LOGGER.logWarning("get water_mark failed. " + str(ret_list))
        return False
    data = rsp_json.get("data")
    if not data:
        LOGGER.logWarning("get water_mark no data. " + str(ret_list))
        return False

    for d in data:
        watermark = int(d.get("max"))
        lsid = d.get("value")
        if str(lsid) not in water_mark_dic[pool_id]:
            water_mark_dic[pool_id][str(lsid)] = watermark
        else:
            water_mark_dic[pool_id][str(lsid)] += watermark
    return True


def get_product_version(dev_node):
    """
    :function: 获取版本
    :param: 节点信息
    :return: 版本信息
    """
    product_version = dev_node.getProductVersion()
    patch_version = dev_node.getHotPatchVersion()
    if not patch_version:
        return product_version
    if patch_version in product_version:
        return product_version
    elif product_version in patch_version:
        return patch_version
    else:
        return product_version + "." + patch_version


def check_version(dev_node):
    """ Ocean Pacific 8.2.1RC2及之后 OceanStor A310 1.0.1RC1及之后才做此检查   """
    product_model = dev_node.getProductModel().lower()
    version = get_product_version(dev_node).lower()
    ret_list.append(product_model)
    ret_list.append(version)
    if "a310" in product_model:
        if "1.0.0" in version:  # 只不支持A310 1.0.0 从1.0.1之后均支持
            return False
    elif "pacific" in product_model:
        if "8.2.0" in version or "8.2.1rc1" in version or "l15y01" in version:  # l15y01为820L
            return False
    return True


def check_lsid(ip_lsid_map, pool_id, watermark_threshold):
    result_list_temp = []
    inspect_result = True

    water_mark_dic_by_pool = water_mark_dic.get(pool_id)
    if not water_mark_dic_by_pool:
        LOGGER.logInfo("watermark dic has no pool_id({})".format(pool_id))
        return result_list_temp, False
    for lsid in water_mark_dic_by_pool:
        watermark = water_mark_dic_by_pool[lsid]
        node_ip = ip_lsid_map.get(str(lsid))
        if not node_ip:  # 可以查到水位，但是对应的ip查不到，一般不会出现
            LOGGER.logWarning("watermark's lsid is not in ip_lsid_maps. lsid_watermark: " +
                              str(water_mark_dic_by_pool) + "ip_lsid_map " + str(ip_lsid_map))
            return result_list_temp, False
        if int(watermark) <= watermark_threshold:
            result_list_temp.append(common.get_err_msg(LANG, "check.cache.watermark.result.pass")
                                    .format(str(node_ip), str(watermark)))
        else:
            inspect_result = False
            result_list_temp.append(common.get_err_msg(LANG, "check.cache.watermark.result.not.pass")
                                    .format(str(node_ip), str(watermark)))
    return result_list_temp, inspect_result


def is_need_check_watetmark(storage_pool):
    # 新增存储池不检查
    if storage_pool.isExpansionCreate():
        LOGGER.logInfo("storage pool is new create. not involved. name({})".format(storage_pool.getName()))
        return False
    # 新增硬盘池不检查（判断条件：现有硬盘池池且存在待扩节点）
    disk_pools = storage_pool.getDiskPools()
    for disk_pool in disk_pools:
        if disk_pool.isExpansionCreate():
            LOGGER.logInfo("disk pool is new create. not involved. name({})".format(disk_pool.getName()))
            continue
        if disk_pool.getExpansionNodeList() or disk_pool.getExpansionClusterNode():
            LOGGER.logInfo("disk pool has new node. diskpool name({})".format(disk_pool.getName))
            return True
    return False


# 获取待检存储池
def get_pool_should_checked(storage_pools):
    pools_of_checked = []
    for storage_pool in storage_pools:
        # 判断是否为待扩存储池
        if not is_need_check_watetmark(storage_pool):
            continue
        pools_of_checked.append(storage_pool)
    return pools_of_checked


def get_all_water_marks(rest, base_uri, pools_of_checked):
    # 10s内每秒查询一次水位 然后取平均值
    for _ in range(0, CHECK_TIMES):
        for storage_pool in pools_of_checked:
            pool_id = storage_pool.getId()
            result = get_node_cache_watermarks(rest, base_uri, pool_id)
            if not result:
                return result
        time.sleep(1)
    # 水位取平均值
    for storage_pool in pools_of_checked:
        pool_id = storage_pool.getId()
        lsids_watermark_map = water_mark_dic.get(pool_id)
        for lsid in lsids_watermark_map:
            lsids_watermark_map[lsid] /= 10
    return True


def execute(rest):
    result_list = []
    inspect_result = common.INSPECT_PASS
    result_info = ""
    try:
        dev_node = py_java_env.get("devInfo")
        storage_pools = dev_node.getStoragePools()
        base_uri = RestUtil.getDstorageUrlHead(dev_node)
        check_version_ret = check_version(dev_node)
        if not check_version_ret:
            return common.INSPECT_PASS, "This version does not support this feature.", ""

        pools_of_checked = get_pool_should_checked(storage_pools)

        if not pools_of_checked:
            return common.INSPECT_PASS, "", ""

        ret = get_all_water_marks(rest, base_uri, pools_of_checked)
        if not ret:
            return common.INSPECT_UNNORMAL, "get cache watermark failed\n", \
                common.get_err_msg(LANG, "query.result.abnormal")
        for storage_pool in pools_of_checked:
            pool_id = storage_pool.getId()
            is_block = storage_pool.isBlock()
            if is_block:
                watermark_threshold = WATER_MARK_THRESHOLD_BLOCK_POOL
            else:
                watermark_threshold = WATER_MARK_THRESHOLD_CONVERGED_POOL
            result, ip_lsid_map = get_ip_lsid_mapping(rest, base_uri, pool_id)
            if not result:
                return common.INSPECT_UNNORMAL, "get ip_lsid_map failed\n", \
                    common.get_err_msg(LANG, "query.result.abnormal")

            single_result_list, temp_result = check_lsid(ip_lsid_map, pool_id, watermark_threshold)
            result_list.extend(single_result_list)
            if not temp_result:
                inspect_result = common.INSPECT_UNNORMAL
        if inspect_result != common.INSPECT_PASS:
            result_info = common.get_err_msg(LANG, "check.cache.watermark.abnormal.tips")
        return inspect_result, "\n".join(result_list), result_info

    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        return common.INSPECT_UNNORMAL, "", common.get_err_msg(LANG, "query.result.abnormal")
