# coding: UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

"""
描述：依赖预巡检脚本 preCheckCapForExpansion.sh 的本地巡检脚本，检查扩容对集群内硬盘池的影响
预巡检脚本返回的字符串格式：
{"code": 101, "result": "exist pool expansion cap risk", "disk_pools": 
[{"pool_id": 1, "security_level": 1, "data_num": 4, "parity_num": 2, 
"max used cap ip":7.225.22.98, "exceeded cap":1000}]
code为101的表示有风险，disk_pools为对应硬盘池信息
"""

import json
import common

from com.huawei.ism.exception import IsmException

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "check_cap_for_expansion"
PRE_ITEM_ID = "preCheckCapForExpansion"

SECURITY_LEVEL_RACK = 0
SECURITY_LEVEL_SERVER = 1

# 预巡检脚本错误码
NORMAL = 0
ABNORMAL = 1
HAS_RISK = 101
NONE_POOL = 102
NOT_MASTER_MDC = 103


class DiskPool:
    def __init__(self, pool_id, security_level, protect_mode, data_num, parity_num, \
    expansion_num, max_used_cap_ip, exceeded_cap):
        self.pool_id = pool_id
        self.security_level = security_level
        self.protect_mode = protect_mode
        self.data_num = data_num
        self.parity_num = parity_num
        self.expansion_num = expansion_num
        self.max_used_cap_ip = max_used_cap_ip
        self.exceeded_cap = exceeded_cap


def get_info_from_master_mdc():
    handle = py_java_env.get("preInspectHandle")
    # 获取所有的节点
    dev_node = py_java_env.get("devInfo")
    for cluster_node in dev_node.getClusterNodes():
        """
            筛选节点类型
            isSelected() 初始化时勾选的节点

            includedStorageService() 包含存储服务
            includeManagementRole() 包含管理节点角色
            isOnlyComputeNode() 纯计算节点

            isPacificNode() 太平洋设备
            isAtlanticNode() 大西洋设备
            isEastSeaSingleNode() 东海单控设备
            isEastSeaDoubleNode() 东海双控设备
            isHGNode() 海光服务器
            isProprietaryHardware() 专有硬件服务器，包括太平洋、大西洋、北冰洋、东海服务器
        """
        if not cluster_node.isSelected() or not cluster_node.includedStorageService():
            continue
        node_ip = common.get_node_ip(cluster_node)
        # 调用框架api，获取节点对应的收集结果
        ret_json_str = handle.getPreInspectResult(node_ip, PRE_ITEM_ID)
        if not ret_json_str or len(ret_json_str) == 0:
            LOGGER.logError("not exist pre inspect result. node ip:{} pre item id:{}".format(node_ip, PRE_ITEM_ID))
            continue
        ret_json_str = ret_json_str.replace('\r', '').replace('\n', '\\n')
        LOGGER.logInfo("pre inspect result:{}".format(ret_json_str))
        try:
            ret_json = json.loads(ret_json_str)
            error_code = ret_json.get("code")
            if error_code != NOT_MASTER_MDC:
                return ret_json
        except (IsmException, Exception) as exception:
            LOGGER.logError("loads json str failed. str:{}".format(ret_json_str))
            LOGGER.logException(exception)
    # 执行到这里说明没有主MDC的巡检结果
    return None


def get_pre_inspect_disk_pools(ret_json):
    pre_inspect_disk_pools_dict = {}
    disk_pools = ret_json.get("disk_pools")
    for disk_pool in disk_pools:
        pool_id = disk_pool.get("pool_id")
        pool_id_str = str(pool_id)
        security_level = disk_pool.get("security_level")
        protect_mode = disk_pool.get("protect_mode")
        data_num = disk_pool.get("data_num")
        parity_num = disk_pool.get("parity_num")
        max_used_cap_ip = disk_pool.get("max_used_cap_ip")
        exceeded_cap = disk_pool.get("exceeded_cap")

        pre_inspect_disk_pools_dict[pool_id_str] = \
            DiskPool(pool_id, security_level, protect_mode, data_num, parity_num, 0, max_used_cap_ip, exceeded_cap)
    return pre_inspect_disk_pools_dict


def get_expansion_num(nodes):
    num = 0
    for node in nodes:
        num += sum(dict(common.get_expansion_main_storage_disk(node)).values())
    return num


# 是否仅为扩盘
def is_only_expansion_disk(disk_pool):
    # 设备实际池内节点
    joined_cluster_node_list = disk_pool.getJoinedClusterNode()
    # 加入了集群的扩容新增节点
    expansion_cluster_node_list = disk_pool.getExpansionClusterNode()
    # 未加入集群的扩容新增节点
    expansion_node_list = disk_pool.getExpansionNodeList()

    # 池内节点的盘数量不为0，新增节点盘数量为0，则认为是扩容硬盘场景
    return get_expansion_num(joined_cluster_node_list) != 0 and \
        get_expansion_num(expansion_cluster_node_list) == 0 and \
        get_expansion_num(expansion_node_list) == 0


# 对齐: (data_num+parity_num)/node_num 或者 node_num/(data_num+parity_num) 为整数
def is_server_aligned_expansion(pre_disk_pool_info, disk_pool):
    # 设备实际池内节点
    joined_cluster_node_len = len(disk_pool.getJoinedClusterNode())
    # 加入了集群的扩容新增节点
    expansion_cluster_node_len = len(disk_pool.getExpansionClusterNode())
    # 未加入集群的扩容新增节点
    expansion_node_len = len(disk_pool.getExpansionNodeList())

    LOGGER.logInfo("check after expansion pool whether is aligned. "
                   "joined node:{}, expansion cluster node:{}, expansion node:{}" \
                   .format(joined_cluster_node_len, expansion_cluster_node_len, expansion_node_len))
    node_nums = joined_cluster_node_len + expansion_cluster_node_len + expansion_node_len
    return (pre_disk_pool_info.data_num + pre_disk_pool_info.parity_num) % node_nums == 0 or \
        node_nums % (pre_disk_pool_info.data_num + pre_disk_pool_info.parity_num) == 0


def execute_inspection(dev_node, ret_json):
    # 从预巡检结果中获取容量有扩容风险的硬盘池
    pre_inspect_disk_pools_dict = get_pre_inspect_disk_pools(ret_json)
    storage_pools = dev_node.getStoragePools()
    for storage_pool in storage_pools:
        # 新建池不涉及该问题, 不做巡检, 存储池内无节点，表现为空池
        if storage_pool.isExpansionCreate():
            continue
        disk_pools = storage_pool.getDiskPools()
        for disk_pool in disk_pools:
            # 扩容新建的池与空池不做巡检
            if disk_pool.isExpansionCreate() or disk_pool.isNullPool():
                continue
            pre_disk_pool_info = pre_inspect_disk_pools_dict.get(str(disk_pool.getId()))
            # 该硬盘池在预巡检中没有风险
            if pre_disk_pool_info is None:
                LOGGER.logInfo("pool:{} is no risk in pre check.".format(disk_pool.getId()))
                continue

            # 扩容硬盘不做检查
            if is_only_expansion_disk(disk_pool):
                LOGGER.logInfo("pool:{} is only expand disk.".format(disk_pool.getId()))
                continue

            # 节点级扩容到对齐场景
            if pre_disk_pool_info.security_level == SECURITY_LEVEL_SERVER and \
                    is_server_aligned_expansion(pre_disk_pool_info, disk_pool):
                LOGGER.logError("server security pool:{} has cap risk during expansion.".format(disk_pool.getId()))
                return (common.INSPECT_UNNORMAL, "pool:{} has cap risk during expansion,node:{},exceeded cap:{} MB."\
                .format(disk_pool.getId(), pre_disk_pool_info.max_used_cap_ip, pre_disk_pool_info.exceeded_cap), "")
            # 获取不到新扩节点的机柜id，机柜级安全无法判断
            LOGGER.logInfo("pool:{} security level:{} check pass." \
                           .format(disk_pool.getId(), pre_disk_pool_info.security_level))

    return common.INSPECT_PASS, "check pass.", ""


# 801版本没有硬盘池，直接用存储池进行判断
def execute_inspection_without_disk_pool(dev_node, ret_json):
    # 从预巡检结果中获取容量有扩容风险的硬盘池
    pre_inspect_disk_pools_dict = get_pre_inspect_disk_pools(ret_json)
    storage_pools = dev_node.getStoragePools()
    for storage_pool in storage_pools:
        # 新建池不涉及该问题, 不做巡检, 存储池内无节点，表现为空池
        if storage_pool.isExpansionCreate():
            continue
        # 扩容新建的池与空池不做巡检
        if storage_pool.isExpansionCreate() or storage_pool.isNullPool():
            continue
        pre_pool_info = pre_inspect_storage_pools_dict.get(str(storage_pool.getId()))
        # 该硬盘池在预巡检中没有风险
        if pre_pool_info is None:
            LOGGER.logInfo("pool:{} is no risk in pre check.".format(storage_pool.getId()))
            continue

        # 扩容硬盘不做检查
        if is_only_expansion_disk(storage_pool):
            LOGGER.logInfo("pool:{} is only expand disk.".format(storage_pool.getId()))
            continue

        # 节点级扩容到对齐场景
        if pre_pool_info.security_level == SECURITY_LEVEL_SERVER and \
                is_server_aligned_expansion(pre_pool_info, storage_pool):
            LOGGER.logError("server security pool:{} has cap risk during expansion.".format(storage_pool.getId()))
            return (common.INSPECT_UNNORMAL, "pool:{} has cap risk during expansion,node:{},exceeded cap:{} MB."\
            .format(storage_pool.getId(), pre_pool_info.max_used_cap_ip, pre_pool_info.exceeded_cap), "")
        # 获取不到新扩节点的机柜id，机柜级安全无法判断
        LOGGER.logInfo("pool:{} security level:{} check pass." \
                        .format(storage_pool.getId(), pre_pool_info.security_level))

    return common.INSPECT_PASS, "check pass.", ""


def execute(rest):
    """
    test
    :param rest:
    :return:
    """

    dev_node = py_java_env.get("devInfo")
    try:
        LOGGER.logInfo("check_pool_cap_for_expand_node start. product_version:{}" \
                       .format(dev_node.getProductVersion() + "," + dev_node.getHotPatchVersion()))

        # 从预巡检脚本传回的信息获取有风险的硬盘池
        ret_json = get_info_from_master_mdc()
        if ret_json is None:
            return common.INSPECT_UNNORMAL, ret_json, common.get_err_msg(LANG, "query.result.abnormal")

        ret_code = ret_json.get("code")
        ret_msg = ret_json.get("result")
        if ret_code == NORMAL or ret_code == NONE_POOL:
            return common.INSPECT_PASS, ret_msg, ""
        if ret_code == ABNORMAL or ret_code != HAS_RISK:
            return common.INSPECT_UNNORMAL, ret_msg, common.get_err_msg(LANG, "query.result.abnormal")

        # 集群中存在扩容风险的硬盘池
        if "8.0.1" in dev_node.getProductVersion():
            return execute_inspection_without_disk_pool(dev_node, ret_json)
        return execute_inspection(dev_node, ret_json)
    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        return common.INSPECT_UNNORMAL, "", common.get_err_msg(LANG, "query.result.abnormal")
