#!/usr/bin/env python
# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

import ast
import re
from com.huawei.ism.tool.obase.exception import ToolException

import common
from cpu_compatibility_matrix import CPU_COMPATIBILITY_BLOCK_ORIGIN_DICT, CPU_COMPATIBILITY_BLOCK_EXPANSION_DICT, \
    CPU_COMPATIBILITY_NOT_BLOCK_EXPANSION_DICT, CPU_COMPATIBILITY_NOT_BLOCK_ORIGIN_DICT, get_score

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
HANDLE = py_java_env.get("preInspectHandle")
ITEM_ID = "check_cpu_specifications_compatibility"
PRE_ITEM_ID = "pre_inspect_get_cpu_specifications"
ARCH_CMD = 'arch'
CPU_VER_CMD = "dmidecode -s processor-version"

# 保存检查结果为不通过的结果信息
err_msg = []
# 保存检查结果为不涉及的结果信息
not_support_msg = []


class DiskPoolInfo:
    """  硬盘池信息类  """

    def __init__(self, name, main_storage_disk_type):
        # 池名称
        self.name = name
        # 主存盘类型（全闪/混存）
        self.main_storage_disk_type = main_storage_disk_type
        # 原有节点信息列表
        self.origin_info = []
        # 待扩节点信息列表
        self.expansion_info = []


class NodeInfo:
    """  节点信息类  """

    def __init__(self, ip, arch, cpu_info):
        # 节点的管理IP
        self.ip = ip
        # 节点的系统架构 eg：ARM架构：aarch64,X86架构：x86_64
        self.arch = arch
        # 节点的CPU型号
        self.cpu_info = cpu_info.upper()

    def show(self):
        return "  ".join([self.ip, self.arch, self.cpu_info])


def execute(rest):
    dev_node = py_java_env.get("devInfo")
    storage_pools = dev_node.getStoragePools()
    if not storage_pools:
        return common.INSPECT_UNNORMAL, "No Storage Pools", common.get_err_msg(LANG, "query.result.abnormal")

    res = []
    for pool in get_pools(storage_pools):
        if not pool.getExpansionNodeList():
            continue
        res.append(get_infos_from_disk_pool(pool))
    LOGGER.logInfo("storage pool:[{}], server type is block:{}".format(
        storage_pools[0].name, storage_pools[0].isBlock()))
    if not storage_pools[0].isBlock():
        check_arch(res, dev_node)
    check_cpu_compatibility(res, storage_pools[0].isBlock())
    if err_msg:
        return common.INSPECT_UNNORMAL, format_res(res), "\n".join(err_msg)
    if not_support_msg:
        return common.INSPECT_NOSUPPORT, format_res(res), "\n".join(not_support_msg)
    return common.INSPECT_PASS, format_res(res), common.get_err_msg(LANG, "check.cpu.compatibility.pass")


def get_pools(storage_pools):
    """  获取所有硬盘池  """
    pool_list = []
    for storage_pool in storage_pools:
        pool_list.extend(storage_pool.getDiskPools())
    return pool_list


def format_res(res):
    """  格式化输出原始信息  """
    formatted_origin_infos = ""
    for pool_info in res:
        origin_nodes = "\n".join(node_info.show() for node_info in pool_info.origin_info)
        expansion_nodes = "\n".join(node_info.show() for node_info in pool_info.expansion_info)
        formatted_origin_infos += "{}:\norigin:\n{}\nexpansion:\n{}\n".format(pool_info.name, origin_nodes,
            expansion_nodes)
    return formatted_origin_infos


def get_infos_from_disk_pool(disk_pool):
    """
    从硬盘池获取信息
    :param disk_pool:  硬盘池
    :return: disk_pool_info: 硬盘池信息
    """
    main_storage_disk_type = "Full-Flash" if "ssd" in str(disk_pool.mainStorageDiskType).lower() else "Hybrid"
    disk_pool_info = DiskPoolInfo(disk_pool.getName(), main_storage_disk_type)
    disk_pool_info.origin_info = get_infos_from_nodes(disk_pool.getJoinedClusterNode())
    expansion_info = get_infos_from_nodes(disk_pool.getExpansionClusterNode())
    expansion_info.extend(get_infos_from_nodes(disk_pool.getExpansionNodeList(), is_cluster_node=False))
    disk_pool_info.expansion_info = expansion_info
    return disk_pool_info


def get_infos_from_nodes(nodes, is_cluster_node=True):
    """  从多个节点批量获取节点信息  """
    if is_cluster_node:
        return [get_info_by_pre_inspect(node) for node in nodes]
    return [get_info_by_ssh(node) for node in nodes]


def get_info_by_pre_inspect(node):
    """  通过预巡检获取集群内节点信息  """
    node_ip = common.get_node_ip(node)
    try:
        pre_inspect_result = HANDLE.getPreInspectResultWithLinePython(node_ip, PRE_ITEM_ID)
        arch, cpu_info = ast.literal_eval(pre_inspect_result)["result"].splitlines()
        return NodeInfo(node_ip, arch, cpu_info)
    except ToolException as e:
        LOGGER.logError(e)
        err_msg.append(common.get_err_msg(LANG, "get.cluster.node.info.failed").format(node_ip))
        return NodeInfo(node_ip, "", "")


def get_info_by_ssh(node):
    """  通过SSH获取集群外节点信息  """
    ssh = None
    arch_info = ''
    cpu_ver = ''
    node_ip = node.getIp()
    try:
        ssh = common.get_ssh_conn(node)
        arch_info = ssh.execCmdWithTimout(ARCH_CMD, common.HOST_CMD_SHORT_TIMEOUT).split("\r\n")[1]
        cpu_specifications_info = ssh.execCmdWithTimout(CPU_VER_CMD, common.HOST_CMD_SHORT_TIMEOUT)
        cpu_ver_info = re.findall(r'[A-Z]?\d{4}[A-Z]?', cpu_specifications_info)
        cpu_ver = cpu_ver_info[0] if cpu_ver_info else cpu_specifications_info
    except ToolException as e:
        LOGGER.logError("failed get node {} cpu ver info {}".format(node.getIp(), e))
        err_msg.append(common.get_err_msg(LANG, "get.not.cluster.node.info.failed").format(node_ip))
    finally:
        if ssh:
            common.release_ssh_conn(ssh)
    return NodeInfo(node_ip, arch_info, cpu_ver)


def check_cpu_compatibility(pool_infos, is_block):
    """  检查CPU规格的兼容性  """
    if not is_block:
        check_1620s_version_811(pool_infos)
        check_1620s_1620_not_in_one_disk_pool(pool_infos)
    for pool_info in pool_infos:
        pool_check(pool_info, is_block)


def pool_check(pool_info, is_block):
    """  检查硬盘池  """
    not_support_nodes = check_node_cpu_is_support(pool_info)
    if not_support_nodes:
        err_msg.append(common.get_err_msg(LANG, "check.cpu.node.not.support").format(pool_info.name,
                                                                                     ", ".join(not_support_nodes)))
    if not pool_info.origin_info:
        if not check_new_build_pool(pool_info.expansion_info, is_block):
            err_msg.append(common.get_err_msg(LANG, "check.cpu.build.pool.check.failed").format(pool_info.name))
        return
    support_list = get_expansion_support_list(
        get_origin_min_performance_cpu(pool_info),
        is_block,
        pool_info.main_storage_disk_type == "Full-Flash")
    if not support_list:
        # 所有节点的CPU都不在兼容性列表中，则报错：待扩节点的CPU不在兼容性列表中
        err_msg.append(common.get_err_msg(LANG, "check.cpu.all.not.support").format(pool_info.name))
        return
    not_support_expansion_node_list = expansion_node_cpu_that_not_in_support_list(pool_info.expansion_info,
                                                                                  support_list)
    if not_support_expansion_node_list:
        err_msg.append(common.get_err_msg(LANG, "check.cpu.expansion.not.match.support.dict").format(pool_info.name,
            str(support_list), "\n".join(not_support_expansion_node_list)))


def check_new_build_pool(expansion_node, is_block):
    """  新建硬盘池检查  """
    expansion_cpu_set = {node_info.cpu_info for node_info in expansion_node}
    support_list = CPU_COMPATIBILITY_BLOCK_EXPANSION_DICT if is_block else CPU_COMPATIBILITY_NOT_BLOCK_EXPANSION_DICT
    for value in support_list.values():
        if expansion_cpu_set.issubset(set(value)):
            return True
    return False


def check_node_cpu_is_support(pool_info):
    """  检查硬盘池的节点是否合法  """
    not_support_nodes = []
    for node_info in pool_info.origin_info + pool_info.expansion_info:
        if get_score(node_info.cpu_info) != 999:
            continue
        not_support_nodes.append(node_info.ip)
    return not_support_nodes


def get_origin_min_performance_cpu(pool_info):
    """  获取原有节点中性能最差的CPU型号  """
    origin_cpu_list = [node.cpu_info for node in pool_info.origin_info]
    origin_cpu_list.sort(key=lambda x: get_score(x), reverse=False)
    return origin_cpu_list[0]


def get_expansion_support_list(cpu_string, is_block, is_full_flush):
    """  获取支持扩容的CPU列表  """
    if is_block:
        for key, value in CPU_COMPATIBILITY_BLOCK_ORIGIN_DICT.items():
            if cpu_string not in value:
                continue
            return CPU_COMPATIBILITY_BLOCK_EXPANSION_DICT.get(key)
    for key, value in CPU_COMPATIBILITY_NOT_BLOCK_ORIGIN_DICT.items():
        if cpu_string not in value:
            continue
        # 5220R的CPU特殊处理
        if key == "3":
            key += "_HDD" if not is_full_flush else "_SSD"
        return CPU_COMPATIBILITY_NOT_BLOCK_EXPANSION_DICT.get(key)
    return []


def expansion_node_cpu_that_not_in_support_list(expansion_node, support_list):
    """  获取待扩容节点中不支持扩容的节点信息  """
    res = []
    for node_info in expansion_node:
        if node_info.cpu_info in support_list:
            continue
        res.append(node_info.show())
    return res


def check_1620s_1620_not_in_one_disk_pool(pool_infos):
    """  CPU：1620s和1620不能共硬盘池  """
    for pool_info in pool_infos:
        cpu_set = set()
        for node_info in pool_info.origin_info + pool_info.expansion_info:
            cpu_set.add(get_score(node_info.cpu_info))
        # 1620性能排行22,1620S性能排行17
        if 22 in cpu_set and 17 in cpu_set:
            err_msg.append(
                common.get_err_msg(LANG, "check.cpu.1620.1620s.can.not.in.same.disk.pool").format(pool_info.name))


def check_1620s_version_811(pool_infos):
    """  1620s要求升级到811及以后的版本  """
    product_version = py_java_env.get("devInfo").getProductVersion()
    if not product_version.startswith("8.1.0"):
        return
    for pool_info in pool_infos:
        for node_info in pool_info.origin_info + pool_info.expansion_info:
            # 1620S性能排行17
            if 17 != get_score(node_info.cpu_info):
                continue
            err_msg.append(common.get_err_msg(LANG, "check.cpu.1620s.need.811"))
            return


def check_arch(pool_infos, dev_node):
    """  8.1.0-8.1.1同一集群只能有一种架构 """
    product_version = py_java_env.get("devInfo").getProductVersion()
    # 8.1.0,8.1.1版本
    if not product_version.startswith("8.1.0") and not product_version.startswith("8.1.1"):
        return
    arch_set = set()
    cluster_nodes = dev_node.getClusterNodes()
    if cluster_nodes:
        arch_set.add(get_info_by_pre_inspect(cluster_nodes[0]).arch)
    for disk_pool_info in pool_infos:
        arch_set.update(
            {node_info.arch for node_info in disk_pool_info.origin_info + disk_pool_info.expansion_info})
    if len(arch_set) > 1:
        err_msg.append(common.get_err_msg(LANG, "check.cpu.810.811.cluster.not.support.mult.arch"))
