# coding: UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
描述：依赖预巡检脚本checkDiskPoolCapacityAndMainStorageDiskCnt.sh的本地巡检脚本，检查每个硬盘池扩盘总数量对GC的影响程度
预巡检脚本返回的字符串格式：
{"error_code": 0,"result": err_msg,"diskPools": [{"poolId": 0,"total_capacity": 675305,"free_capacity": 0,
"origin_disk_cnt": 0}], ...}(一行)
其中err_msg可能为多行，已经在巡检脚本中额外处理
"""
import json

import common
from com.huawei.ism.exception import IsmException
import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil
from ds_rest_util import CommonRestService

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "check_expand_disk_num_in_each_disk_pool"
PRE_ITEM_ID = "checkDiskPoolCapacityAndMainStorageDiskCnt"
PRECISION = 6


class StoragePoolCapacityExpansion:
    storage_pool_id = ""
    storage_pool_name = ""
    disk_pool_capacity_expansion_list = []

    def __init__(self, sid, name):
        self.storage_pool_id = sid
        self.storage_pool_name = name
        self.disk_pool_capacity_expansion_list = []


class DiskPoolCapacityExpand:
    disk_pool_id = ""
    disk_pool_name = ""
    recommended_expand_disks_cnt = 0
    pass_or_not = ""

    def __init__(self, pool_id, name, cnt, flag):
        self.disk_pool_id = pool_id
        self.disk_pool_name = name
        self.recommended_expand_disks_cnt = cnt
        self.pass_or_not = flag


class DiskPoolInfo:
    redundancy_policy = ""
    data_units = 0
    parity_units = 0
    free_ratio = 0.0
    origin_disk_cnt = 0

    def __init__(self, policy, data_num, parity_num, ratio, cnt):
        self.redundancy_policy = policy
        self.data_units = data_num
        self.parity_units = parity_num
        self.free_ratio = ratio
        self.origin_disk_cnt = cnt


# 预巡检脚本错误码
# 0 正常
# 10 集群主MDC异常，无法获取数据
# 11 /opt/dfv/oam/public/conf/product.ini配置文件未见版本信息
# 12 当前版本不涉及问题，不需要巡检
# 13 集群内不存在硬盘池
# 14 获取硬盘池容量和主存盘数量失败
# 15 当前节点非主MDC所在节点
NOT_MASTER_MDC = 15

ERROR_CODE_DICT = {
    10: common.INSPECT_UNNORMAL,
    11: common.INSPECT_UNNORMAL,
    12: common.INSPECT_NOSUPPORT,
    13: common.INSPECT_PASS,
    14: common.INSPECT_UNNORMAL
}

ERROR_MSG_DICT = {
    10: "pre.inspection.master.mdc.abnormal",
    11: "pre.inspection.production_config.abnormal",
    12: "",
    13: "",
    14: "pre.inspection.master.mdc.query.disk.pool.info.error"
}


def execute(rest):
    """
    检查扩容前硬盘池容量, 获取可扩容盘最大值
    : param rest:获取集群信息接口,见dm界面http请求
    : return 1:错误码 2: 原始信息 3: 用户信息
    """
    ret_list = common.get_err_msg(LANG, "expansion.config.info")
    dev_node = py_java_env.get("devInfo")
    observer = py_java_env.get("progressObserver")

    try:
        # 从预巡检脚本传回的信息获取硬盘池容量
        ret_json = get_info_from_master_mdc()
        if ret_json is None:
            return common.INSPECT_UNNORMAL, ret_json, \
                   common.get_err_msg(LANG, "pre.inspection.mdc.cluster.abnormal")
        error_code = ret_json.get("error_code")
        error_msg = ret_json.get("result")
        if error_code != 0:
            if error_code == 12 or error_code == 13:
                return ERROR_CODE_DICT[error_code], error_msg, ""
            else:
                return ERROR_CODE_DICT[error_code], error_msg, common.get_err_msg(LANG, ERROR_MSG_DICT[error_code])
        # 字典类型, key是string类型的disk_pool_id, value是(硬盘池空闲率, 硬盘池原有主存盘数量)
        pre_inspect_disk_pool_info_dict = get_pre_inspect_disk_pool_info(ret_json)
        if pre_inspect_disk_pool_info_dict is None:
            return common.INSPECT_UNNORMAL, "", common.get_err_msg(LANG, "query.result.abnormal")

        # 使用get命令https://${ip}:${port}/api/v2/data_service/diskpool查询硬盘池冗余策略相关信息
        base_uri = RestUtil.getDstorageUrlHead(dev_node)
        cmd_str = "{}/api/v2/data_service/diskpool".format(base_uri)
        pools_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
        if not pools_json.get("diskPools"):
            # 获取的json文件中没有池
            return common.INSPECT_PASS, pools_json, ""
        disk_pools = pools_json.get("diskPools", [])
        # 字典类型, key是string类型的disk_pool_id, value是类DiskPoolInfo实例化
        disk_pool_info_dict = get_disk_pool_info(disk_pools, pre_inspect_disk_pool_info_dict)

        (inspection_ok, capacity_expansion_list) = execute_inspection(dev_node, observer, disk_pool_info_dict)
        if inspection_ok:
            return common.INSPECT_PASS, ret_list, ""
        else:
            return common.INSPECT_WARNING, ret_list, generate_warning_message(capacity_expansion_list)

    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        return common.INSPECT_UNNORMAL, ret_list, common.get_err_msg(LANG, "query.result.abnormal")


def get_info_from_master_mdc():
    handle = py_java_env.get("preInspectHandle")
    # 获取所有的节点
    dev_node = py_java_env.get("devInfo")
    for cluster_node in dev_node.getClusterNodes():
        """
            筛选节点类型
            isSelected() 初始化时勾选的节点

            includedStorageService() 包含存储服务
            includeManagementRole() 包含管理节点角色
            isOnlyComputeNode() 纯计算节点

            isPacificNode() 太平洋设备
            isAtlanticNode() 大西洋设备
            isEastSeaSingleNode() 东海单控设备
            isEastSeaDoubleNode() 东海双控设备
            isHGNode() 海光服务器
            isProprietaryHardware() 专有硬件服务器，包括太平洋、大西洋、北冰洋、东海服务器
        """
        if not cluster_node.isSelected() or not cluster_node.includedStorageService():
            continue
        node_ip = common.get_node_ip(cluster_node)
        # 调用框架api，获取节点对应的收集结果
        ret_json_str = handle.getPreInspectResult(node_ip, PRE_ITEM_ID).replace('\r', '').replace('\n', '\\n')
        ret_json = json.loads(ret_json_str)
        error_code = ret_json.get("error_code")
        if error_code != NOT_MASTER_MDC:
            return ret_json
    # 执行到这里说明全都不是主MDC节点
    return None


def get_pre_inspect_disk_pool_info(ret_json):
    pre_inspect_disk_pool_info_dict = {}
    disk_pools = ret_json.get("disk_pools")
    for disk_pool in disk_pools:
        disk_pool_id = str(disk_pool.get("poolId"))
        total_capacity = disk_pool.get("total_capacity")
        free_capacity = disk_pool.get("free_capacity")
        origin_disk_cnt = disk_pool.get("origin_disk_cnt")
        try:
            # 硬盘池空闲率, 空闲容量/总容量
            free_ratio = free_capacity * 1.0 / total_capacity
            pre_inspect_disk_pool_info_dict[disk_pool_id] = (free_ratio, origin_disk_cnt)
        except ZeroDivisionError:
            LOGGER.logInfo("Failed to calculate the free_ratio")
            return None
    return pre_inspect_disk_pool_info_dict


def get_disk_pool_info(disk_pools, disk_pool_free_ratio_dict):
    disk_pool_info_dict = {}
    for disk_pool in disk_pools:
        disk_pool_id = str(disk_pool.get("poolId"))
        # 冗余策略: EC或副本
        redundancy_policy = disk_pool.get("redundancyPolicy")
        # ec的数据块个数和校验块个数
        data_units = disk_pool.get("numDataUnits")
        parity_units = disk_pool.get("numParityUnits")
        # 硬盘池空闲率, 空闲容量/总容量
        free_ratio, origin_disk_cnt = disk_pool_free_ratio_dict[disk_pool_id]

        disk_pool_info_dict[disk_pool_id] = DiskPoolInfo(redundancy_policy, data_units,
                                                         parity_units, free_ratio, origin_disk_cnt)

    return disk_pool_info_dict


def execute_inspection(dev_node, observer, disk_pool_info_dict):
    inspection_ok = True
    # 列表类型, 元素是类StoragePoolCapacityExpansion实例化
    capacity_expansion_list = list()

    progress_map = {ITEM_ID: 1}
    observer.updateProgress(progress_map)

    # 810版本及以后getStoragePools()获取的是storage pool列表
    storage_pools = dev_node.getStoragePools()
    for storage_pool in storage_pools:
        # 新建池不涉及该问题, 不做巡检, 存储池内无节点，表现为空池
        if storage_pool.isExpansionCreate():
            continue
        storage_pool_result = StoragePoolCapacityExpansion(str(storage_pool.getId()), str(storage_pool.getName()))

        disk_pools = storage_pool.getDiskPools()
        for disk_pool in disk_pools:
            # 扩容新建的池与空池不做巡检
            if disk_pool.isExpansionCreate() or disk_pool.isNullPool():
                continue
            # 通过硬盘池id从disk_pool_info_dict字典中获取硬盘池信息
            disk_pool_info = disk_pool_info_dict[str(disk_pool.getId())]
            # 副本池不做巡检
            if disk_pool_info.redundancy_policy == "replication":
                continue
            expand_disk_cnt = get_expand_disk_cnt(disk_pool)
            recommended_expand_disks_cnt = get_recommended_disks_cnt(disk_pool_info.free_ratio,
                                                                     disk_pool_info.data_units,
                                                                     disk_pool_info.parity_units,
                                                                     disk_pool_info.origin_disk_cnt)
            pass_or_not = "pass"
            if recommended_expand_disks_cnt < expand_disk_cnt:
                inspection_ok = False
                pass_or_not = "warning"
            disk_pool_result = DiskPoolCapacityExpand(str(disk_pool.getId()), str(disk_pool.getName()),
                                                      recommended_expand_disks_cnt, pass_or_not)
            storage_pool_result.disk_pool_capacity_expansion_list.append(disk_pool_result)
        capacity_expansion_list.append(storage_pool_result)

    return inspection_ok, capacity_expansion_list


def get_expand_disk_cnt(disk_pool):
    # 设备实际池内节点
    joined_cluster_node_list = disk_pool.getJoinedClusterNode()
    # 加入了集群的扩容新增节点
    expansion_cluster_node_list = disk_pool.getExpansionClusterNode()
    # 未加入集群的扩容新增节点
    expansion_node_list = disk_pool.getExpansionNodeList()

    expand_disk_cnt = get_expansion_num(joined_cluster_node_list) + get_expansion_num(expansion_cluster_node_list)\
                         + get_expansion_num(expansion_node_list)
    return expand_disk_cnt


def get_recommended_disks_cnt(free_ratio, data_units, parity_units, disks_cnt):
    try:
        k = round(data_units ** 2 * 1.0 / (data_units + parity_units) ** 2 * free_ratio, PRECISION)
        result = round(k / (1 - k) * disks_cnt, PRECISION)
        recommended_disks_cnt = int(result)
        return recommended_disks_cnt
    except ZeroDivisionError:
        LOGGER.logInfo("Failed to calculate the recommended_disks_cnt.")
        return None


def generate_warning_message(capacity_expansion_list):
    result = ""
    storage_pool_solution = ""
    for storage_pool_result in capacity_expansion_list:
        disk_pool_solution = ""
        for disk_pool_result in storage_pool_result.disk_pool_capacity_expansion_list:
            disk_pool_solution += common.get_err_msg(LANG, "expand.disk.pool.capacity.expansion.solution",
                                                     (disk_pool_result.disk_pool_id, disk_pool_result.disk_pool_name,
                                                      disk_pool_result.recommended_expand_disks_cnt,
                                                      disk_pool_result.pass_or_not))
        storage_pool_solution += common.get_err_msg(LANG, "expand.storage.pool.capacity.expansion.solution",
                                                    (storage_pool_result.storage_pool_id,
                                                     storage_pool_result.storage_pool_name, disk_pool_solution))
    result += common.get_err_msg(LANG, "expand.disk.expansion.cnt.exceed.threshold", storage_pool_solution)
    return result


def get_expansion_num(nodes):
    num = 0
    for node in nodes:
        num += sum(dict(common.get_expansion_main_storage_disk(node)).values())
    return num
