#!/usr/bin/env python
# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

import os
import re

from com.huawei.ism.exception import IsmException
from com.huawei.ism.tool.obase.exception import ToolException
from com.huawei.ism.tool.protocol.utils import RestUtil

import common
from ds_rest_util import CommonRestService

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "check_fips_service"

# 结果信息：保存结果为失败的信息
err_msgs = []
# 结果信息：保存结果为不涉及的信息
not_support_msgs = []
# 原始信息，保存执行的指令和得到的回显
origin_infos = []


def execute(rest):
    # 首先获取连接
    dev_node = py_java_env.get("devInfo")
    # 然后获取存储池
    storage_pools = dev_node.getStoragePools()
    # 获取存储池信息失败，判断为异常情况，不做后续检查
    if not storage_pools:
        return common.INSPECT_UNNORMAL, "No Storage Pools", common.get_err_msg(LANG, "query.result.abnormal")

    # 先判断是否需要检查：设备版本不低于[8.2.1]且集群开启了fips
    query_is_need_check(rest, dev_node)
    if not_support_msgs:
        return common.INSPECT_NOSUPPORT, os.linesep.join(origin_infos), os.linesep.join(not_support_msgs)
    if err_msgs:
        return common.INSPECT_UNNORMAL, os.linesep.join(origin_infos), os.linesep.join(err_msgs)

    # 查询集群外的每个待扩容节点是否支持开启fips：架构是arm，且欧拉版本不低于eulerosv2r12
    check_all_expansion_node(get_all_expansion_node(storage_pools))
    if err_msgs:
        return common.INSPECT_UNNORMAL, os.linesep.join(origin_infos), os.linesep.join(err_msgs)
    if not_support_msgs:
        return common.INSPECT_NOSUPPORT, os.linesep.join(origin_infos), os.linesep.join(not_support_msgs)
    return common.INSPECT_PASS, os.linesep.join(origin_infos), common.get_err_msg(LANG, "check.node.fips.pass")


def get_all_expansion_node(storage_pools):
    """
    从存储池中获取所有集群外的待扩节点
    :param storage_pools:存储池的列表
    :return: expansion_nodes 待扩节点的列表
    """
    expansion_nodes = []
    # 集群中存在0个或多个存储池，每个存储池存在0个或多个硬盘池，每个硬盘池存在0个或多个集群外待扩节点
    for storage_pool in storage_pools:
        for disk_pool in storage_pool.getDiskPools():
            expansion_nodes.extend(disk_pool.getExpansionNodeList())
    return expansion_nodes


def get_node_arch_and_os_version_from_ssh(node):
    """
    通过SSH连接查询集群外待扩节点的系统架构和欧拉版本信息
    :param node: 集群外待扩节点
    :return:node_arch_info, node_os_version_info：节点架构信息，节点欧拉版本信息
    """
    ssh = None
    try:
        ssh = common.get_ssh_conn(node)
        node_arch_info = ssh.execCmdWithTimout("arch", common.HOST_CMD_SHORT_TIMEOUT)
        node_os_version_info = ssh.execCmdWithTimout("uname -a", common.HOST_CMD_SHORT_TIMEOUT)
    except ToolException as e:
        LOGGER.logError("failed get node {} info:[{}]".format(node.getIp(), e))
        node_arch_info, node_os_version_info = "arch query error\n", "os version query error"
    finally:
        if ssh:
            common.release_ssh_conn(ssh)
    origin_infos.append("{}:[{} {}]\n".format(node.getIp(), node_arch_info, node_os_version_info))
    return node_arch_info, node_os_version_info


def check_node_support_fips(expansion_node):
    """
    查询集群外的每个待扩容节点是否支持开启fips：架构是arm，且欧拉版本不低于eulerosv2r12
    :param expansion_node: 单个待扩节点
    :return: 是/否
    """
    node_arch_info, node_os_version_info = get_node_arch_and_os_version_from_ssh(expansion_node)
    # 架构是arm
    if "aarch64" not in node_arch_info:
        return False
    # 欧拉版本不低于eulerosv2r12
    os_version = re.search("eulerosv2r(\d+)", node_os_version_info)
    if not os_version:
        return False
    return int(os_version.group(1)) >= 12


def check_all_expansion_node(expansion_nodes):
    """  检查所有的集群外待扩节点  """
    if not expansion_nodes:
        not_support_msgs.append(common.get_err_msg(LANG, "cluster.fips.not.node.outside.cluster"))
        return

    node_not_pass = [node.getIp() for node in expansion_nodes if not check_node_support_fips(node)]
    if not node_not_pass:
        return
    err_msgs.append(common.get_err_msg(LANG, "check.fips.not.pass.node").format(os.linesep.join(node_not_pass)))


def query_is_need_check(rest, dev_node):
    """先判断是否需要检查：设备版本为8.2.1及以后版本且集群开启了fips"""
    if not common.is_product_version_ge_than_821(dev_node):
        not_support_msgs.append(common.get_err_msg(LANG, "product.version.not.involve.check"))
    try:
        if not query_cluster_fips_status(rest, dev_node):
            not_support_msgs.append(common.get_err_msg(LANG, "cluster.fips.not.open"))
    except IsmException as e:
        LOGGER.logError("query cluster fips status failed:[{}]".format(str(e)))
        # 存储版本低，不支持查询fips状态，导致查询失败时，视为fips未开启
        not_support_msgs.append(common.get_err_msg(LANG, "cluster.fips.not.open"))


def query_cluster_fips_status(rest, dev_node):
    """
    检查设备集群的fips服务是否开启
    :param rest: 集群rest链接
    :param dev_node: 集群设备
    :return: True/False：fips服务状态为开启/关闭
    """
    base_uri = RestUtil.getDstorageUrlHead(dev_node)
    cmd_str = "{}/api/v2/cluster_service/queryfips".format(base_uri)
    fips_info_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    origin_infos.append(str(fips_info_json))
    LOGGER.logInfo("query cluster fips info:[{}]".format(str(fips_info_json)))
    return fips_info_json.get("data", {}).get("configValue", "") == "enable"
