# coding:utf-8
# Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

"""
预巡检查询结果存储格式:
节点不涉及:
not involved result
查询失败:
query failed: err_result
查询成功:
pool_id:xxx,total_capacity:xxx,available_free:xxx
"""

import re
import os
import common
from common import get_err_msg

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
HANDLE = py_java_env.get("preInspectHandle")
ITEM_ID = "check_free_ratio"
PRE_ITEM_ID = "pre_inspect_get_free_ratio"
QUERY_FAILED_PREFIX = "query failed: "
NO_SUPPORT_PREFIX = "pre_inspect_get_free_ratio-3"
NOT_INVOLVED = "not.involved"

# 15%
FREE_RATIO_THRESHOLD = 15000

COMMON_WORK_PATH = "/opt/dfv/oam/public/script/inspect"
# 获取工作目录，如果是正常的节点，返回空，如果是自定义了opt目录的dpc节点，返回自定义的目录
CURRENT_PATH = os.path.realpath(__file__).split("/opt/dfv/oam/public/script/inspect")[0] \
    if COMMON_WORK_PATH in os.path.realpath(__file__) else ""

# 保存结果为不通过时的错误信息
err_msg = []
# 保存查询到的原始信息,当报错时,方便定位
res_msg = []


def is_involved_version(product_version):
    """
    检查版本信息:801至821版本支持该巡检项
    """
    version = product_version.replace('.', "")
    if "801" <= version[0:3] <= "821":
        return True
    return False


def parse_result(pre_inspect_result, node_ip_str):
    query_res = re.search(r'pool_id:(\d+),total_capacity:(\d+),available_free:(\d+)', pre_inspect_result)
    if not query_res:
        # 解析失败，查询结果有误，巡检不通过
        LOGGER.logError("pre inspect query failed." + node_ip_str + "pre item id:{}".format(PRE_ITEM_ID))
        err_msg.append(node_ip_str + get_err_msg(LANG, "query.storage.pool.error"))
        return False

    pool_id = int(query_res.group(1))
    total_capacity = int(query_res.group(2))
    available_free = int(query_res.group(3))
    LOGGER.logInfo(
        node_ip_str + "pool_id:{},total_capacity:{},available_free:{}".format(pool_id, total_capacity,
                                                                              available_free))
    res_msg.append(
        node_ip_str + "pool_id:{},total_capacity:{},available_free:{}".format(pool_id, total_capacity,
                                                                              available_free))
    if total_capacity == 0:
        LOGGER.logError("pool_id:{},total_capacity is zero.".format(pool_id))
        err_msg.append("pool_id:{},total_capacity is zero.".format(pool_id))
        return False
    free_ratio = available_free * 100000 / total_capacity
    if free_ratio <= FREE_RATIO_THRESHOLD:
        LOGGER.logError(get_err_msg(LANG, "check.storage.pool.water.level.not.pass").format(pool_id))
        err_msg.append(get_err_msg(LANG, "check.storage.pool.water.level.not.pass").format(pool_id))
        return False

    return True


def execute(rest):
    # 获取节点
    dev_node = py_java_env.get("devInfo")

    # 判断是否是涉及版本
    product_version = str(dev_node.getProductVersion())
    if not is_involved_version(product_version):
        LOGGER.logInfo("product version({}) is not involved.".format(product_version))
        return common.INSPECT_NOSUPPORT, get_err_msg(LANG, "query.version.na", product_version), get_err_msg(
            LANG, "query.version.na", product_version)

    dev_nodes = dev_node.getClusterNodes()
    if not dev_nodes or len(dev_nodes) == 0:
        return common.INSPECT_NOSUPPORT, "No Nodes", ""

    is_pass = True
    is_checked = False
    for node in dev_nodes:
        node_ip = common.get_node_ip(node)
        node_ip_str = "node ip:{} ".format(node_ip)

        # 获取查询结果
        pre_inspect_result = HANDLE.getPreInspectResult(node_ip, PRE_ITEM_ID)
        # 不涉及，检查其他节点
        if not pre_inspect_result:
            LOGGER.logInfo("not exist pre inspect result. node ip:{} pre item id:{}".format(node_ip, PRE_ITEM_ID))
            res_msg.append(node_ip_str + get_err_msg(LANG, NOT_INVOLVED))
            continue
        if NO_SUPPORT_PREFIX in pre_inspect_result:
            LOGGER.logInfo(
                get_err_msg(LANG, NOT_INVOLVED) + ". node ip:{} pre item id:{}".format(node_ip, PRE_ITEM_ID))
            res_msg.append(node_ip_str + get_err_msg(LANG, NOT_INVOLVED))
            continue

        is_checked = True
        # 查询失败，巡检不通过
        if QUERY_FAILED_PREFIX in pre_inspect_result:
            is_pass = False
            LOGGER.logError("pre inspect query failed. node ip:{} pre item id:{}".format(node_ip, PRE_ITEM_ID))
            err_msg.append(node_ip_str + get_err_msg(LANG, "query.storage.pool.error") + "({})".format(
                pre_inspect_result[
                pre_inspect_result.find(QUERY_FAILED_PREFIX) + len(QUERY_FAILED_PREFIX):pre_inspect_result.find("]")]))
            continue

        # 解析查询结果
        is_pass = is_pass and parse_result(pre_inspect_result, node_ip_str)

    if is_checked:
        if not is_pass:
            return common.INSPECT_UNNORMAL, '\n'.join(res_msg), '\n'.join(err_msg)
        return common.INSPECT_PASS, '\n'.join(res_msg), ""
    else:
        return common.INSPECT_NOSUPPORT, '\n'.join(res_msg), ""
