# coding: UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import json
import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil
from com.huawei.ism.exception import IsmException
from check_pool_num import CheckPoolNumUtil

import common
from ds_rest_util import CommonRestService

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
HANDLE = py_java_env.get("preInspectHandle")

ITEM_ID = "check_iSCSI_link_balance"
PRE_ITEM_ID = "pre_inspect_check_iSCSI_link_balance"


def execute(rest):
    """
    检查存储节点iSCSI链路是否均衡
    步骤一:检查存储池的节点是否开启了iSCSI服务。
    如果没有节点开启iSCSI服务,结束;否则继续执行步骤二。
    步骤二:检查存储池中开启了iSCSI服务的节点,统计其中未配置iSCSI链路的节点数量。
    如果未配置iSCSI链路的节点数量大于等于3个节点,结束;否则继续执行步骤三。
    步骤三:检查未配置iSCSI链路的节点数量在开启iSCSI服务节点数占比。
    通过“未配置链路节点数/存储池开启iSCSI节点数”计算占比。
    :param env:
    :return:
    """
    # 用作详细信息，rest接口请求信息
    LOGGER.logInfo("check_iSCSI_link_balance begin ...")
    ret_list = []
    balance_res_all = ""
    dev_node = py_java_env.get("devInfo")
    observer = py_java_env.get("progressObserver")

    progress_map = {}
    try:
        progress_map[ITEM_ID] = 1
        observer.updateProgress(progress_map)

        # 812之前的版本不涉及该巡检项
        product_version = str(dev_node.getProductVersion())
        if not is_involve_product_version(product_version):
            LOGGER.logInfo("check_iSCSI_link_balance not support ...")
            return common.INSPECT_NOSUPPORT, "", ""
        
        # 获取存储节点
        storage_node_ip_list = get_storage_node(rest, dev_node)
        vbs_client_list = get_vbs_node(rest, dev_node)
        # 过滤出含VBS服务的存储节点
        vbs_storage_nodes = get_vbs_storage_node(vbs_client_list, storage_node_ip_list)

        balance_res_pools = check_pool_iscsi_link_balance(dev_node, rest, ret_list, vbs_storage_nodes)
        if len(balance_res_pools) != 0:
            balance_res_all = "ESN : " + str(dev_node.getDeviceSerialNumber()) + "\n" + balance_res_pools

        progress_map[ITEM_ID] = 80
        observer.updateProgress(progress_map)

        if len(balance_res_all) != 0:
            return common.INSPECT_UNNORMAL, "\n".join(ret_list), balance_res_all
        return common.INSPECT_PASS, "\n".join(ret_list), ""
    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        return (
            common.INSPECT_UNNORMAL,
            "\n".join(ret_list),
            common.get_err_msg(LANG, "query.result.abnormal"),
        )


def is_involve_product_version(product_version):
    version = product_version.replace('.', "")
    if version[0:2] > "81":
        return True
    if version[0:2] == "81" and version[2].isdigit() and version[2] >= "2":
        return True
    if len(version) >= 5:
        if version[0:2] == "81" and version[2:4] == "RC" and version[4] >= "3":
            return True
    return False


def get_iscsi_link_map(dev_node):
    # 理论上只有主节点会收集预收集信息，如果没有收集信息，通过
    iscsi_link_map = {}
    for cluster_node in dev_node.getClusterNodes():
        if not cluster_node.isSelected() or not cluster_node.includedStorageService():
            continue
        node_ip = common.get_node_ip(cluster_node)
        # 调用框架api，获取节点对应的收集结果
        pre_inpect_result = HANDLE.getPreInspectResult(node_ip, PRE_ITEM_ID)
        LOGGER.logInfo(node_ip + " result:" + str(pre_inpect_result) + "\n")
        if not pre_inpect_result or pre_inpect_result == "Not involved":
            continue
        for item in pre_inpect_result.split(';'):
            LOGGER.logInfo("node_ip:" + node_ip + "item:" + item)
            if item is None or item == "" or item.count('-') == 0:
                continue
            key, value = item.split('-')
            iscsi_link_map[key] = int(value)
    LOGGER.logInfo("iscsi_link_map = " + str(iscsi_link_map))
    return iscsi_link_map


def is_storage_node(role_list):
    """
    判断是否是存储节点
    """
    for role in role_list:
        if role == "storage":
            return True
    return False


def get_storage_node(rest, dev_node):
    """
    获取存储节点
    """
    base_uri = RestUtil.getDstorageUrlHead(dev_node)
    cmd_str = "{}/api/v2/cluster/servers".format(base_uri)
    cluster_nodes_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    LOGGER.logInfo(cluster_nodes_json)

    result = cluster_nodes_json.get("result")
    LOGGER.logInfo(result)
    storage_node_ip_list = list()
    if result == 0:
        return storage_node_ip_list
    # 过滤出存储节点
    cluster_nodes = cluster_nodes_json.get("data")
    for cluster_node in cluster_nodes:
        role_list = cluster_node.get("role", {})
        if is_storage_node(role_list):
            storage_node_ip_list.append(cluster_node.get("management_ip"))
    LOGGER.logInfo("storage_node_ip_list:" + json.dumps(storage_node_ip_list))
    return storage_node_ip_list


def get_vbs_node(rest, dev_node):
    """
    获取VBS节点
    """
    vbs_client_list = {}
    base_uri = RestUtil.getDstorageUrlHead(dev_node)
    cmd_str = "{}/dsware/service/cluster/dswareclient/queryDSwareClientHost".format(base_uri)
    vbs_nodes_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    LOGGER.logInfo(vbs_nodes_json)
    result = vbs_nodes_json.get("result")
    if result != 0:
        return vbs_client_list
    vbs_client_list = vbs_nodes_json.get("clients")
    return vbs_client_list


def get_vbs_storage_node(vbs_client_list, storage_node_ip_list):
    """
    过滤出含VBS服务的存储节点
    """
    vbs_storage_nodes = list()
    for vbs_node in vbs_client_list:
        manage_ip = vbs_node.get("manageIp")
        if manage_ip in storage_node_ip_list:
            vbs_storage_nodes.append(vbs_node)
    LOGGER.logInfo("vbs_storage_nodes:" + json.dumps(vbs_storage_nodes))
    return vbs_storage_nodes


def get_iscsi_switch(vbs_storage_nodes, open_iscsi_server_manager_ip_list):
    """
    查看节点的iscsi开关是否开启
    """
    if vbs_storage_nodes is None:
        return
    for vbs_storage_node in vbs_storage_nodes:
        if vbs_storage_node is None:
            continue
        iscsi_switch = vbs_storage_node.get("iscsiSwitch")
        if iscsi_switch == "open":
            warning_manager_ip_list.append(vbs_storage_node.get("manageIp"))


def get_iscsi_status(rest, warning_manager_ip_list, dev_node, manager_ip_list, ret_list):
    """
    查看iscsi状态以及IP是否配置
    """
    base_uri = RestUtil.getDstorageUrlHead(dev_node)
    cmd_str = "{}/dsware/service/cluster/dswareclient/queryIscsiPortal".format(base_uri)
    ret_list.append(cmd_str)
    param = {
        "nodeMgrIps": json.dumps(manager_ip_list)
    }
    iscsi_nodes_json = CommonRestService.execute_post_request(rest, cmd_str, param)
    ret_list.append(str(iscsi_nodes_json).encode("utf-8").decode("unicode_escape"))
    iscsi_node_list = iscsi_nodes_json.get("nodeResultList")
    LOGGER.logInfo("iscsi_node_list:" + json.dumps(iscsi_node_list))
    for iscsi_node in iscsi_node_list:
        LOGGER.logInfo("iscsi_node:" + json.dumps(iscsi_node))
        node_mgr_ip = iscsi_node.get("nodeMgrIp")
        if node_mgr_ip in warning_manager_ip_list:
            LOGGER.logInfo("node_mgr_ip:{} already in warning_manager_ip_list".format(node_mgr_ip))
            continue
        iscsi_status = iscsi_node.get("status")
        LOGGER.logInfo("status:{}".format(iscsi_status))
        if iscsi_status == "failed":
            warning_manager_ip_list.append(node_mgr_ip)


def check_one_pool_iscsi_link_balance(ret_list, pool_node_ips, iscsi_link_map):
    # 存储池打开了iscsi开关节点数
    pool_iscsi_open_num = 0
    # 存储池打开了iscsi开关，但是没有链路的节点数
    pool_iscsi_link_null_num = 0
    LOGGER.logInfo("iscsi_link_map = " + json.dumps(iscsi_link_map))
    for manager_ip, storage_ip in pool_node_ips.items():
        LOGGER.logInfo("manager_ip = " + manager_ip + " storage_ip = " + storage_ip)
        if storage_ip in iscsi_link_map:
            link_num = iscsi_link_map[storage_ip]
            ret_list.append("node manager ip: " + str(manager_ip) + \
            " node storage ip: " + str(storage_ip) + " iscsiLinkNum: " + str(link_num))
            pool_iscsi_open_num += 1
            if link_num == 0:
                pool_iscsi_link_null_num += 1
        else:
            ret_list.append("node manager ip: " + str(manager_ip) + \
            " node storage ip: " + str(storage_ip) + " iscsiLinkNum: None")
    LOGGER.logInfo("pool_iscsi_open_num = " + str(pool_iscsi_open_num) + \
        "  pool_iscsi_link_null_num = " + str(pool_iscsi_link_null_num))
    return pool_iscsi_open_num, pool_iscsi_link_null_num


def check_pool_iscsi_link_balance(dev_node, rest, ret_list, vbs_storage_nodes):
    # 获取集群存储池信息
    pool_iscsi_res_all = ""
    base_uri = RestUtil.getDstorageUrlHead(dev_node)
    cmd_str = "{}/dsware/service/resource/queryStoragePool".format(
        base_uri
    )
    pools_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    if not pools_json.get("storagePools"):
        LOGGER.logInfo("not exist storage pool.")
        return balance_res_pools
    storage_pools = pools_json.get("storagePools", [])
    #预收集信息解析
    iscsi_link_map = get_iscsi_link_map(dev_node)
    ret_list.append("Resolve the iSCSI NodeIp-LinkNum:\n" + json.dumps(iscsi_link_map))

    # 遍历集群存储池获取信息
    for record in storage_pools:
        ret_list.append("Start checking pool " + str(record.get("poolId")) + ": " + str(record.get("poolId")))
        pool_node_ips = get_pool_node_mgr_ips(dev_node, rest, record)
        if pool_node_ips is None:
            LOGGER.logInfo("pool_node_ips is None.")
            ret_list.append("None")
            continue
        ret_list.append("Pool Node IP: " + json.dumps(pool_node_ips))

        # 存储池节点数
        pool_nodes_num = len(pool_node_ips)
        pool_iscsi_open_num, pool_iscsi_link_null_num = \
        check_one_pool_iscsi_link_balance(ret_list, pool_node_ips, iscsi_link_map)
        log_res = "pool_nodes_num:" + str(pool_nodes_num) + "  pool_iscsi_open_num:" + \
            str(pool_iscsi_open_num) + "  pool_iscsi_link_null_num:" + str(pool_iscsi_link_null_num)
        LOGGER.logInfo(log_res)
        ret_list.append(log_res)
        pool_iscsi_res = get_pool_iscsi_res(pool_nodes_num, pool_iscsi_open_num, pool_iscsi_link_null_num, record)
        pool_iscsi_res_all = pool_iscsi_res_all + pool_iscsi_res
    return pool_iscsi_res_all


def get_pool_node_mgr_ips(dev_node, rest, record):
    pool_node_ips = {}
    # 前端存储IP
    pool_id = record.get("poolId")
    if pool_id is None:
        LOGGER.logInfo("pool id is null.")
        return pool_node_ips
    LOGGER.logInfo("pool_id:" + str(pool_id))
    # 通过池ID获取池节点信息
    base_uri = RestUtil.getDstorageUrlHead(dev_node)
    cmd_str = (
        "{}/dsware/service/cluster/storagepool/"
        "queryNodeDiskInfo?poolId={}".format(base_uri, pool_id)
    )
    node_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    if not node_json.get("nodeInfo"):
        LOGGER.logInfo("nodeInfo is null.")
        return pool_node_ips
    
    cmd_str = (
        "{}/dsware/service/cluster/dswareclient/"
        "queryDSwareClientHost?status=1".format(base_uri)
    )
    client_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    if not client_json.get("clients"):
        LOGGER.logInfo("clientInfo is null.")
        return pool_node_ips

    for storage_node in node_json.get("nodeInfo"):
        manage_ip = storage_node.get("nodeMgrIp")
        for client in client_json.get("clients"):
            if manage_ip == client.get("manageIp"):
                pool_node_ips[manage_ip] = client.get("clusterIp")
                break

    LOGGER.logInfo("pool_node_ips:" + json.dumps(pool_node_ips))
    return pool_node_ips


def get_pool_iscsi_res(pool_nodes_num, pool_iscsi_open_num, pool_iscsi_link_null_num, record):
    # 如果没有节点开启iSCSI开关，结束
    pool_iscsi_res = ""
    if pool_iscsi_open_num == 0:
        return pool_iscsi_res
    # 如果未开启iSCSI开关节点数大于等于3或者未开启iSCSI开关节点在存储池节点数占比大于等于20%
    if pool_iscsi_link_null_num >= 3 or round(float(pool_iscsi_link_null_num) / float(pool_iscsi_open_num), 2) >= 0.2:
        # 池名称、池ID
        pool_iscsi_res = pool_iscsi_res + "Storage Pool Name : " + str(record.get("poolName")) + "\n"
        pool_iscsi_res = pool_iscsi_res + "Storage Pool ID : " + str(record.get("poolId")) + "\n"

        pool_pool_iscsi_exit_num = pool_iscsi_open_num - pool_iscsi_link_null_num
        pool_iscsi_res = pool_iscsi_res + "Number of Storage Nodes With iSCSI Links : " + \
            str(pool_pool_iscsi_exit_num) + "\n"
        pool_iscsi_res = pool_iscsi_res + "Number of Storage Nodes Without iSCSI Links : " + \
            str(pool_iscsi_link_null_num) + "\n"
    else:
        LOGGER.logInfo("PoolID " + str(record.get("poolId")) + " no need rebalance iSCSI links!")
    return pool_iscsi_res