#!/usr/bin/env python
# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

import ast
import re
from com.huawei.ism.tool.obase.exception import ToolException

import common

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
HANDLE = py_java_env.get("preInspectHandle")
ITEM_ID = "check_memory_capacity"
PRE_ITEM_ID = "pre_inspect_get_capacity_info"
QUERY_FAILED = 0

# 保存结果为不通过时的错误信息
err_msg = []
# 保存查询到的原始信息，当报错时，方便定位
res_msg = []


class NodeMemoryInfo:
    """  节点内存信息类，保存节点的IP，内存容量   """

    def __init__(self, ip, memory):
        self.ip = ip
        self.memory = memory

    def show(self):
        return "{} - {} GB".format(self.ip, self.memory)


class DiskPoolMemoryInfo:
    """  硬盘池信息类，保存硬盘池名称和池中的节点的信息   """

    def __init__(self, disk_pool_name):
        self.name = disk_pool_name
        self.origin_node_list = []
        self.expansion_node_list = []

    def get_not_pass_expansion_node(self):
        """
        获取硬盘池中检查失败的扩容节点
        :return int 原有节点总内存的最小值, list 检查失败的节点的列表
        """
        if not self.expansion_node_list or not self.origin_node_list:
            return 0, []
        min_memory = min(node.memory for node in self.origin_node_list)
        return min_memory, [node.show() for node in self.expansion_node_list if node.memory < min_memory]

    def get_query_fail_node(self):
        """  获取硬盘池中查询失败的节点  """
        return [node.ip for node in self.origin_node_list + self.expansion_node_list if node.memory == QUERY_FAILED]

    def show(self):
        """原始信息格式化输出
        池名称:[****]
        原有节点信息：
        节点IP - 节点内存容量
        ***.***.***.*** - XX GB
        待扩节点信息：
        节点IP - 节点内存容量
        ***.***.***.*** - XX GB
        """
        origin_node_info = "\n".join(node.show() for node in self.origin_node_list)
        expansion_node_info = "\n".join(node.show() for node in self.expansion_node_list)
        return common.get_err_msg(LANG, "check.memory.disk.pool.info").format(
            self.name, origin_node_info, expansion_node_info)


def execute(rest):
    # 首先获取连接
    dev_node = py_java_env.get("devInfo")
    # 然后获取硬盘池（8.0.X版本为存储池）
    pool_list = common.get_disk_pool_by_dev_node(dev_node)
    # 扩容时必定有池，没有池，是异常场景
    if not pool_list:
        return common.INSPECT_UNNORMAL, "No Pools", common.get_err_msg(LANG, "query.result.abnormal")
    check(pool_list)
    if err_msg:
        return common.INSPECT_UNNORMAL, '\n'.join(res_msg), '\n'.join(err_msg)
    return common.INSPECT_PASS, '\n'.join(res_msg), common.get_err_msg(LANG, "check.node.memory.pass")


def check(pool_list):
    pool_infos = (creat_pool_info(pool) for pool in pool_list)
    for pool_info in pool_infos:
        # 保存原始信息
        res_msg.append(pool_info.show())
        # 扩容检查失败的节点
        min_memory, not_pass_node = pool_info.get_not_pass_expansion_node()
        if not_pass_node:
            err_msg.append(common.get_err_msg(LANG, "check.memory.not.pass.node").format(
                pool_info.name, min_memory, "\n".join(not_pass_node)))

        # 查询内存容量信息失败的节点
        query_fail_node = pool_info.get_query_fail_node()
        if query_fail_node:
            err_msg.append(common.get_err_msg(LANG, "check.memory.query.fail.node").format(
                pool_info.name, "\n".join(query_fail_node)))


def creat_pool_info(pool):
    """
    获取池的内存信息

    :param pool:池的node
    :return: memory_info_list 存储硬盘池内存信息的列表
    """
    pool_info = DiskPoolMemoryInfo(pool.getName())
    pool_info.origin_node_list = [creat_node_info_by_pre_inspect(node) for node in pool.getJoinedClusterNode()]
    pool_info.expansion_node_list.extend(
        creat_node_info_by_pre_inspect(node) for node in pool.getExpansionClusterNode())
    pool_info.expansion_node_list.extend(creat_node_info_by_ssh(node) for node in pool.getExpansionNodeList())
    return pool_info


def creat_node_info_by_ssh(node):
    """  通过ssh获取节点信息，适用于集群外节点  """
    cmd_str = r"dmidecode -t memory|egrep 'Size:\s[0-9]+\s\w+'|grep -v 'Volatile'"
    ssh = None
    try:
        ssh = common.get_ssh_conn(node)
        node_size_info = ssh.execCmdWithTimout(cmd_str, common.HOST_CMD_SHORT_TIMEOUT)
        sum_size = get_sum_size_from_info(node_size_info.encode("utf8"))
    except ToolException as e:
        LOGGER.logError("failed get node {} memery info {}".format(node.getIp(), e))
        sum_size = QUERY_FAILED
    finally:
        if ssh:
            common.release_ssh_conn(ssh)
    return NodeMemoryInfo(node.getIp(), sum_size)


def creat_node_info_by_pre_inspect(node):
    """  通过预巡检方式获取节点信息，适用于集群内节点  """
    node_ip = common.get_node_ip(node)
    try:
        pre_info = HANDLE.getPreInspectResultWithLinePython(node_ip, PRE_ITEM_ID)
        node_size_info = ast.literal_eval(pre_info)["result"]
        sum_size = get_sum_size_from_info(node_size_info)
    except ToolException as e:
        LOGGER.logError("failed get node {} memery info {}".format(node_ip, e))
        sum_size = QUERY_FAILED
    return NodeMemoryInfo(node_ip, sum_size)


def get_sum_size_from_info(node_size_info):
    """
    从获得的信息中计算总内存
    :param node_size_info: 节点内存的信息
    :return: sum_memory_capacity 总内存
    """

    sum_memory_capacity = 0
    lines = re.findall(r'Size: \d*\s\w\w', node_size_info)
    for line in lines:
        sum_memory_capacity += trans_to_gb(line.strip().split(' '))
    return sum_memory_capacity


def trans_to_gb(line):
    """
    把[“Size：”，“数字”，“单位”]的容量信息转换为统一为gb为单位浮点数
    :param line: [“Size：”，“数字”，“单位”]
    :return: num 代表容量为 num GB
    """

    num = float(line[-2])
    unit = line[-1].lower()
    if unit in ('g', 'gb'):
        return num
    elif unit in ('m', 'mb'):
        return num / 1024
    elif unit in ('k', 'kb'):
        return num / (1024 * 1024)
    elif unit in ('t', 'tb'):
        return num * 1024
    # 未知的容量单位，返回0
    else:
        return 0
