# coding:utf-8
# Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

"""
预巡检查询结果存储格式:
节点不涉及:
not involved result
查询失败:
query failed: err_result
查询成功:
db_id:xxx, key_num:xxx db_id:xxx, key_num:xxx db_id:xxx, key_num:xxx
"""

import re
import os
import common
from common import get_err_msg

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
HANDLE = py_java_env.get("preInspectHandle")
ITEM_ID = "check_metadata_for_scale_out"
PRE_ITEM_ID = "pre_inspect_get_metadata_num"
QUERY_FAILED_PREFIX = "query failed: "
NO_SUPPORT_PREFIX = "pre_inspect_get_metadata_num-3"
NOT_INVOLVED = "not.involved"

# 80亿
KEY_NUM_THRESHOLD = 8000000000

COMMON_WORK_PATH = "/opt/dfv/oam/public/script/inspect"
# 获取工作目录，如果是正常的节点，返回空，如果是自定义了opt目录的dpc节点，返回自定义的目录
CURRENT_PATH = os.path.realpath(__file__).split("/opt/dfv/oam/public/script/inspect")[0] \
    if COMMON_WORK_PATH in os.path.realpath(__file__) else ""

# 保存结果为不通过时的错误信息
err_msg = []
# 保存查询到的原始信息,当报错时,方便定位
res_msg = []


def is_involved_version(product_version):
    """
    检查版本信息:801至版本支持该巡检项,或者1.0.x版本也支持
    """
    version = product_version.replace('.', "")
    if "801" <= version[0:3] <= "821" or version.startswith("1"):
        return True
    return False


def parse_result(pre_result, node_ip_str, cur_err_msg):
    cur_node_is_pass = True
    cur_res_msg = node_ip_str
    pattern = r'db_id:(\d+),key_num:(\d+)'
    query_res = pre_result.split(" ")
    for item in query_res:
        match = re.search(pattern, item)
        if not match:
            continue
        db_id = int(match.group(1))
        key_num = int(match.group(2))
        LOGGER.logInfo(node_ip_str + "db_id:{}, key_num:{}".format(db_id, key_num))
        cur_res_msg = cur_res_msg + "\n\tdb_id:{}, key_num:{}".format(db_id, key_num)
        if key_num >= KEY_NUM_THRESHOLD:
            cur_node_is_pass = False
            LOGGER.logError(get_err_msg(LANG, "check.metadata.not.pass").format(db_id, key_num))
            cur_err_msg.append(get_err_msg(LANG, "check.metadata.not.pass").format(db_id, key_num))

    res_msg.append(cur_res_msg)
    return cur_node_is_pass


def execute(rest):
    # 获取节点
    dev_node = py_java_env.get("devInfo")

    # 判断是否是涉及版本
    product_version = str(dev_node.getProductVersion())
    if not is_involved_version(product_version):
        LOGGER.logInfo("product version({}) is not involved.".format(product_version))
        return common.INSPECT_NOSUPPORT, get_err_msg(LANG, "query.version.na", product_version), get_err_msg(
            LANG, "query.version.na", product_version)

    dev_nodes = dev_node.getClusterNodes()
    if not dev_nodes or len(dev_nodes) == 0:
        return common.INSPECT_NOSUPPORT, "No Nodes", ""

    is_pass = True
    is_checked = False
    for node in dev_nodes:
        node_ip = common.get_node_ip(node)
        node_ip_str = "node ip:{}".format(node_ip)

        # 获取查询结果
        pre_inspect_result = HANDLE.getPreInspectResult(node_ip, PRE_ITEM_ID)

        # 不涉及
        if not pre_inspect_result:
            LOGGER.logInfo("not exist pre inspect result. node ip:{} pre item id:{}".format(node_ip, PRE_ITEM_ID))
            res_msg.append(node_ip_str + get_err_msg(LANG, NOT_INVOLVED))
            continue
        if NO_SUPPORT_PREFIX in pre_inspect_result:
            LOGGER.logInfo(
                get_err_msg(LANG, NOT_INVOLVED) + ". node ip:{} pre item id:{}".format(node_ip, PRE_ITEM_ID))
            res_msg.append(node_ip_str + get_err_msg(LANG, NOT_INVOLVED))
            continue

        is_checked = True
        # 查询失败，巡检不通过
        if QUERY_FAILED_PREFIX in pre_inspect_result:
            is_pass = False
            LOGGER.logError("pre inspect query failed. node ip:{} pre item id:{}".format(node_ip, PRE_ITEM_ID))
            err_msg.append(node_ip_str + get_err_msg(LANG, "query.storage.pool.error") + "({})".format(
                pre_inspect_result[
                pre_inspect_result.find(QUERY_FAILED_PREFIX) + len(QUERY_FAILED_PREFIX):pre_inspect_result.find("]")]))
            continue

        # 解析查询结果
        cur_err_msg = []
        cur_node_is_pass = parse_result(pre_inspect_result, node_ip_str, cur_err_msg)

        if not cur_node_is_pass:
            is_pass = False
            LOGGER.logError("check metadata not pass.")
            err_msg.append(node_ip_str + '\n\t' + '\n\t'.join(cur_err_msg))

    if is_checked:
        if not is_pass:
            return common.INSPECT_UNNORMAL, '\n'.join(res_msg), '\n'.join(err_msg)
        return common.INSPECT_PASS, '\n'.join(res_msg), ""
    else:
        return common.INSPECT_NOSUPPORT, '\n'.join(res_msg), ""
