# coding: UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.
import json
import re

from com.huawei.ism.exception import IsmException

import cli_util
import common
import hardware_util
import redfish_util

ITEM_ID = "check_os_version"
LANG = common.getLang(py_java_env)
HANDLE = py_java_env.get("preInspectHandle")
RES_KEY = "OS_Image_Ver"


def execute(rest_conn):
    """
    检查OS版本
    :param rest_conn: 集群rest连接
    :return:
    """
    item = common.ItemEntity(py_java_env, ITEM_ID, PY_LOGGER, __file__)
    item.update_progress(1)
    try:
        check_os_version(item.build_param_dict(), None, item.err_info_list, item.ret_list)
        return item.get_check_result()
    except (IsmException, Exception) as exception:
        item.logger.logException(exception)
        return item.get_failed_result(common.get_err_msg(
            LANG, "query.result.abnormal"))


@common.check_all_ibmc_node
def check_os_version(param_dict, cluster_dev_node, err_info_list, ret_list):
    cluster_node_ip = common.get_node_ip(cluster_dev_node)
    redfish = common.get_redfish_conn(cluster_dev_node)
    ret_by_node = []
    logger = param_dict.get("logger")
    context = param_dict.get("env")
    try:
        ret = HANDLE.getTargetCheckResult(cluster_node_ip, ITEM_ID)
        logger.logInfo("FusionStorage OS origin: {}".format(ret))
        data = json.loads(ret)
        ret_by_node.append(data.get("original_info"))

        version_ret = data.get("result")
        version = parse_os_version(version_ret)
        logger.logInfo("FusionStorage OS Version: {}".format(version))
        uid, _ = redfish_util.get_product_unique_id(redfish, logger)
        mapping_version = hardware_util\
            .get_mapping_fw_version(context, uid, RES_KEY)
        os_little_version = parse_os_little_version(version_ret)
        if version not in mapping_version or os_little_version \
                < parse_os_little_version(mapping_version):
            msg = common.get_err_msg(LANG, "version.mapping.not.pass",
                                     (cluster_node_ip, version_ret, mapping_version))
            err_info_list.append(msg)
    except Exception as exception:
        logger.logException(exception)
        err_info_list.append(common.get_err_msg(
            LANG, "item.check.abnormal", cluster_node_ip))
    finally:
        ret_list.append(cli_util.get_format_header_ret(
            cluster_node_ip, "\n".join(ret_by_node)))


def parse_os_little_version(version_info):
    match = re.findall(r".h(\d{3}).", version_info)
    if match:
        return int(match[0])
    raise Exception("parsed os little version failed")


def parse_os_version(version_info):
    match = re.findall(r".(eulerosv\d+r\d+).", version_info)
    if match:
        return match[0]
    raise Exception("parsed os version failed")
