# coding: UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2023. All rights reserved.
import re

from com.huawei.ism.tool.obase.exception import ToolException

import cli_util
import com.huawei.ism.tool.protocol.utils.RestUtil as rest_util
import common
from ds_rest_util import CommonRestService

LANG = common.getLang(py_java_env)
ITEM_ID = "check_raid_card_compatibility"
NO_FILE = "No such file or directory"
LOGGER = common.getLogger(PY_LOGGER, __file__)

PY_JAVA_ENV = py_java_env
VERSION_REG = re.compile("([.\d]+)")


def execute(rest_conn):
    """
    检查raid卡与存储版本兼容性
    :param rest_conn: 集群rest连接
    :return:
    """
    auto_update_process()
    return RaidCardCompatibilityChecker(rest_conn, PY_JAVA_ENV.get("devInfo")).check()


def auto_update_process():
    observer = PY_JAVA_ENV.get("progressObserver")
    progress_map = {ITEM_ID: 1}
    observer.updateProgress(progress_map)


def query_product_hotpatch_ver(dev_node, rest_conn):
    base_uri = rest_util.getDstorageUrlHead(dev_node)
    cmd_str = "{}/api/v2/cluster/product".format(base_uri)
    product_result = CommonRestService.exec_get_gor_big_by_ds(rest_conn, cmd_str)
    return build_origin_info(product_result, cmd_str), \
           product_result.get("data", {}).get("hotpatch_version", ""), \
           product_result.get("data", {}).get("version", ""), \
           product_result.get("data").get("patch_version", "")


def build_origin_info(product_result, url):
    return "\n".join([url, str(product_result)])


def patch_version_contains_product_version(patch_version):
    return VERSION_REG.findall(patch_version) and '.' in patch_version


class RaidCardCompatibilityChecker(object):
    def __init__(self, rest_conn, dev_node):
        self._rest_conn = rest_conn
        self._product_version = ""
        self._hot_patch_version = ""
        self._patch_version = ""
        self._dev_node = dev_node
        self._origin_info = []
        self._raid_card_type = {}
        self._err_msg = []
        self._init_product_version()
        self._raid_judge_executor = []
        self.init_executor()

    def init_executor(self):
        for sub_class in RaidJudgeExecutor.__subclasses__():
            self._raid_judge_executor.append(sub_class())

    def check(self):
        self._init_raid_card_type()
        if not self._raid_card_type:
            all_ret = common.save_cli_ret_to_file(self._origin_info, ITEM_ID, PY_JAVA_ENV, LOGGER)
            # 可能存在设备连不上造成没查到raid的场景
            if self._err_msg:
                return common.INSPECT_UNNORMAL, all_ret, "\n".join(self._err_msg)
            else:
                return common.INSPECT_PASS, all_ret, ""
        return self._check_raid_compatibility()

    def _init_raid_card_type(self):
        expansion_nodes = PY_JAVA_ENV.get("newExpansionNodes", [])
        if not expansion_nodes:
            LOGGER.logInfo("no expansion nodes found")
            return
        self._init_each_node_raid_card_type(expansion_nodes)

    def _init_each_node_raid_card_type(self, expansion_nodes):
        for expansion_node in expansion_nodes:
            self._query_one_node_raid_card_type(expansion_node)

    def _query_one_node_raid_card_type(self, dev_node):
        ssh = None
        try:
            ssh = common.get_ssh_conn(dev_node)
            self._query_raid_card_type(dev_node, ssh)
        except ToolException as exception:
            self._err_msg.append(exception.getDes())
        finally:
            if ssh:
                common.release_ssh_conn(ssh)

    def _query_raid_card_type(self, expansion_node, ssh):
        origin_info = []
        for sub_executor in self._raid_judge_executor:
            ssh_ret, raid_card_type = sub_executor.query_raid_card_type(ssh)
            LOGGER.logInfo("node:{} found raid_card_type :{}".format(expansion_node.getIp(), raid_card_type))
            origin_info.append(ssh_ret)
            if raid_card_type:
                self._raid_card_type.setdefault(raid_card_type, []).append(expansion_node)
                # 存储只可能有一张raid卡，找到了就中断循环
                break
        self._origin_info.append(cli_util.get_format_header_ret(expansion_node.getIp(), "\n\n".join(origin_info)))

    def _check_raid_compatibility(self):
        for raid_card, expansion_nodes in self._raid_card_type.items():
            for sub_executor in self._raid_judge_executor:
                LOGGER.logInfo("check raid :{}, target raid :{}".format(raid_card, sub_executor.get_target_raid_type()))
                if raid_card != sub_executor.get_target_raid_type():
                    continue
                if sub_executor.check_product_version_not_compatibility(self._product_version, self._hot_patch_version,
                                                                        self._patch_version):
                    self._append_error_msg(expansion_nodes, raid_card)
        all_ret = common.save_cli_ret_to_file(self._origin_info, ITEM_ID, PY_JAVA_ENV, LOGGER)
        if not self._err_msg:
            return common.INSPECT_PASS, all_ret, ""
        return common.INSPECT_UNNORMAL, all_ret, "\n".join(self._err_msg)

    def _append_error_msg(self, expansion_nodes, raid_card):
        error_ips = []
        for node in expansion_nodes:
            error_ips.append(node.getIp())
        self._err_msg.append(self._build_error_msg(error_ips, raid_card))

    def _build_error_msg(self, error_ips, raid_card):
        return common.get_err_msg(LANG, "raid.card.not.compatibility",
                                  (";".join(error_ips), raid_card, self.get_show_version()))

    def get_show_version(self):
        if self._hot_patch_version:
            return self.build_show_version(self._hot_patch_version)
        if self._patch_version:
            return self.build_show_version(self._patch_version)
        return self._product_version

    def build_show_version(self, patch_version):
        if patch_version_contains_product_version(patch_version):
            return patch_version
        return "{}.{}".format(self._product_version, patch_version)

    def _init_product_version(self):
        origin_info, self._hot_patch_version, self._product_version, self._patch_version = query_product_hotpatch_ver(
            self._dev_node, self._rest_conn)
        self._origin_info.append(origin_info)


class RaidJudgeExecutor(object):

    def query_raid_card_type(self, ssh):
        """
        获取raid卡类型
        :param ssh: ssh连接
        :return: raid卡类型
        """
        raise NotImplementedError

    def get_equal_black_version_list(self):
        """
        获取当前raid卡的黑名单校验，在名单中的版本为不支持的版本,用于==匹配
        :return: 目标热补丁版本
        """
        raise NotImplementedError

    def get_start_black_version_list(self):
        """
        获取当前raid卡的黑名单校验，在名单中的版本为不支持的版本，用于startswith匹配
        主要用于判断补丁版本不明确的大版本场景
        :return: 目标热补丁版本
        """
        raise NotImplementedError

    def get_target_raid_type(self):
        """
        返回当前判断类所判断的raid卡类型
        :return: raid类型
        """
        raise NotImplementedError

    def check_product_version_not_compatibility(self, product_version, hot_patch_version, patch_version):
        """
        检查当前raid卡是否与存储软件版本不兼容
        :param product_version: 当前集群的大版本
        :param hot_patch_version: 当前集群的热补丁版本
        :param patch_version: 当前集群的补丁版本
        :return: True/False
        """
        LOGGER.logInfo("check version :{} hot_patch_version :{} patch_version：{}"
                       .format(product_version, hot_patch_version, patch_version))
        if self._is_start_version(product_version):
            return True
        black_version = self.get_equal_black_version_list()
        LOGGER.logInfo("black version list :{}".format(black_version))
        if not isinstance(black_version, list):
            black_version = [black_version]
        if self._product_version_not_involve(black_version, product_version, hot_patch_version, patch_version):
            return True
        if self._patch_version_not_involve(black_version, product_version, hot_patch_version, patch_version):
            return True
        if self.hot_patch_version_not_involve(black_version, product_version, hot_patch_version):
            return True
        return False

    def _product_version_not_involve(self, black_version, product_version, hot_patch_version, patch_version):
        # 有补丁版本，就不判断软件大版本
        if hot_patch_version or patch_version:
            return False
        version = VERSION_REG.findall(product_version)
        return product_version in black_version or (version and version[0] in black_version)

    def _patch_version_not_involve(self, black_version, product_version, hot_patch_version, patch_version):
        if hot_patch_version:
            # 如果有热补丁版本，优先判断热补丁版本
            return False
        return self.judge_patch_version(black_version, product_version, patch_version)

    def hot_patch_version_not_involve(self, black_version, product_version, hot_patch_version):
        return self.judge_patch_version(black_version, product_version, hot_patch_version)

    def judge_patch_version(self, black_version, product_version, patch_version):
        if not patch_version:
            return False
        if patch_version_contains_product_version(patch_version):
            return patch_version in black_version
        # 判断存储软件大版本 +.补丁版本
        version = VERSION_REG.findall(product_version)
        LOGGER.logInfo("find reg version :{}".format(version))
        if not version:
            return False
        return "{}.{}".format(version[0], patch_version) in black_version

    def _is_start_version(self, product_version):
        start_black = self.get_start_black_version_list()
        if not isinstance(start_black, list):
            start_black = [start_black]
        for black_version in start_black:
            if product_version.startswith(black_version):
                return True
        return False


class Raid3152JudgeExecutor(RaidJudgeExecutor):
    UNION_CARD_REG = re.compile(r"Smart Storage.*Union Memory")
    # 不对外体现BP卡，3152BP卡更名为FBGF-RAD-R1-S1
    UNION_3152_RAID = "FBGF-RAD-R1-S1"

    def query_raid_card_type(self, ssh):
        ssh_ret = cli_util.execute_ssh_cmd_remove_color(ssh, "lspci -m |grep -i Adaptec", common.HOST_CMD_SHORT_TIMEOUT)
        if Raid3152JudgeExecutor.UNION_CARD_REG.findall(ssh_ret):
            return ssh_ret, Raid3152JudgeExecutor.UNION_3152_RAID
        return ssh_ret, ""

    def get_equal_black_version_list(self):
        return ["8.1.5", "8.1.5.SPH001", "8.1.5.SPH005", "8.1.5.SPH010", "8.1.5.SPH015", "8.1.5.SPH017",
         "8.1.5.SPH019", "8.2.0"]

    def get_start_black_version_list(self):
        return ["8.1.0", "8.1.1", "8.1.2", "8.1.3"]

    def get_target_raid_type(self):
        return Raid3152JudgeExecutor.UNION_3152_RAID


class Raid3908JudgeExecutor(RaidJudgeExecutor):
    CARD_REG = re.compile(r"\sSAS39xx\s?")
    RAID_3908 = "3908"

    def query_raid_card_type(self, ssh):
        ssh_ret = cli_util.execute_ssh_cmd_remove_color(ssh, "lspci | grep -i LSI", common.HOST_CMD_SHORT_TIMEOUT)
        if Raid3908JudgeExecutor.CARD_REG.findall(ssh_ret):
            return ssh_ret, Raid3908JudgeExecutor.RAID_3908
        return ssh_ret, ""

    def get_start_black_version_list(self):
        return ["8.1.0", "8.1.1"]

    def get_equal_black_version_list(self):
        return []

    def get_target_raid_type(self):
        return Raid3908JudgeExecutor.RAID_3908


class Raid1880JudgeExecutor(RaidJudgeExecutor):
    CARD_REG = re.compile(r"Device 3758")

    # 1880卡对外展示为 SP686C-M-40i,SP686C-M-16i
    def __init__(self):
        super(Raid1880JudgeExecutor, self).__init__()
        self._raid_1880 = "SP686C-M-40i{}SP686C-M-16i".format(common.get_err_msg(LANG, "or"))

    def query_raid_card_type(self, ssh):
        ssh_ret = cli_util.execute_ssh_cmd_remove_color(ssh, "lspci | grep -i LSI", common.HOST_CMD_SHORT_TIMEOUT)
        if Raid1880JudgeExecutor.CARD_REG.findall(ssh_ret):
            return ssh_ret, self._raid_1880
        ssh_ret = "{}\n\n{}".format(ssh_ret, cli_util.execute_ssh_cmd_remove_color(ssh, "lspci | grep -i RAID",
                                                                                   common.HOST_CMD_SHORT_TIMEOUT))
        if Raid1880JudgeExecutor.CARD_REG.findall(ssh_ret):
            return ssh_ret, self._raid_1880
        return ssh_ret, ""

    def get_equal_black_version_list(self):
        return ["8.1.2", "8.1.RC3", "8.1.2.SPH001", "8.1.2.SPH002", "8.1.2.SPC100", "8.1.2.SPH101", "8.1.2.SPH102",
                "8.1.2.SPH103", "8.1.2.SPH105", "8.1.3", "8.1.3.SPH001", "8.1.RC6"]

    def get_start_black_version_list(self):
        return ["8.1.0", "8.1.1"]

    def get_target_raid_type(self):
        return self._raid_1880
