#!/usr/bin/env python
# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import ast

from com.huawei.ism.exception import IsmException
from com.huawei.ism.tool.obase.exception import ToolException

import common

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
HANDLE = py_java_env.get("preInspectHandle")
ITEM_ID = "check_ssd_models"
PRE_ITEM_ID = "pre_inspect_get_trim_white_ls_and_ssd_models"


def execute(rest):
    return CheckSsdModel(rest).check()


class CheckSsdModel:
    def __init__(self, rest):
        self.rest = rest
        self.err_msgs = []
        self.res_ls = []
        self.trim_white_ls = []
        self.models_not_in_white_ls = set()
        self.msg_head = ""

    @staticmethod
    def is_product_version_gt_811(product_version):
        """  版本是否大于811  """
        version = product_version.replace('.', "")
        if version[0:2] > "81":
            return True
        if version[0:2] == "81" and version[2].isdigit() and version[2] >= "2":
            return True
        if len(version) >= 5 and version[:4] == "81RC" and version[4] >= "3":
            return True
        return False

    def check(self):
        """  检查SSD盘model号是否在trim白名单  """
        dev_node = py_java_env.get("devInfo")
        observer = py_java_env.get("progressObserver")
        progress_map = {}

        try:
            progress_map[ITEM_ID] = 1
            observer.updateProgress(progress_map)

            product_version = str(dev_node.getProductVersion())
            # 仅非结构化 【811及以前版本】涉及
            if self.is_product_version_gt_811(product_version):
                msg = common.get_err_msg(LANG, "query.version.na", product_version)
                return common.INSPECT_NOSUPPORT, msg, msg

            # 获取trim白名单
            self.get_trim_white_ls(dev_node)
            if not self.trim_white_ls:
                return common.INSPECT_UNNORMAL, "\n".join(self.res_ls), common.get_err_msg(LANG,
                                                                                           "get.trim.white.list.failed")

            # 获取存储池
            storage_pools = dev_node.getStoragePools()
            LOGGER.logInfo("storage_pools:{}".format(storage_pools))
            if not storage_pools:
                return common.INSPECT_UNNORMAL, str(dev_node), common.get_err_msg(LANG, "query.result.abnormal")

            for storage_pool in storage_pools:
                # 仅【非结构化】 811及以前版本涉及
                if storage_pool.isBlock():
                    msg = common.get_err_msg(LANG, "block.not.involved")
                    return common.INSPECT_NOSUPPORT, msg, msg

                # 当前接口限制（有硬盘池的 不能直接通过存储池获取节点信息）
                # com.huawei.ism.tool.inspect.fit.service.ExpansionImpl.expansionNode
                # addNewNodes的getFusionStoragePool有硬盘池获取硬盘池，否则存储池
                for disk_pool in storage_pool.getDiskPools():
                    self.msg_head = common.get_err_msg(LANG, "storage.pool.disk.pool.msg",
                                                       (storage_pool.getName(), disk_pool.getName()))
                    self.check_ssd_model(disk_pool)

            if self.err_msgs:
                return common.INSPECT_UNNORMAL, "\n".join(self.res_ls), "\n".join(self.err_msgs)

            return common.INSPECT_PASS, "\n".join(self.res_ls), ""
        except (IsmException, Exception) as exception:
            LOGGER.logException(exception)
            return common.INSPECT_UNNORMAL, "\n".join(self.res_ls), common.get_err_msg(LANG, "query.result.abnormal")

    def get_trim_white_ls(self, dev_node):
        for cluster_node in dev_node.getClusterNodes():
            node_ip = common.get_node_ip(cluster_node)
            pre_info = HANDLE.getPreInspectResultWithLinePython(node_ip, PRE_ITEM_ID)
            trim_white_list_info, _ = ast.literal_eval(pre_info)["result"].split("###")
            if trim_white_list_info:
                self.trim_white_ls = trim_white_list_info.split("=")[1].split(",")
                self.res_ls.append(str(["trim_white_ls:", self.trim_white_ls]))
                LOGGER.logInfo("trim_white_ls:{}".format(self.trim_white_ls))
                return

    def check_ssd_model(self, pool):
        self.check_ssd_model_in_cluster(pool)
        self.check_ssd_model_not_in_cluster(pool)
        if self.models_not_in_white_ls:
            self.err_msgs.append(
                common.get_err_msg(LANG, "has.ssd.models.not.in.trim.white.ls",
                                   (self.msg_head, str(self.models_not_in_white_ls))))

    def check_ssd_model_in_cluster(self, pool):
        cluster_nodes = pool.getJoinedClusterNode()
        cluster_nodes.extend(pool.getExpansionClusterNode())
        LOGGER.logInfo("pool_cluster_nodes:{}".format(cluster_nodes))
        for cluster_node in cluster_nodes:
            node_ip = common.get_node_ip(cluster_node)
            pre_info = HANDLE.getPreInspectResultWithLinePython(node_ip, PRE_ITEM_ID)
            _, model_and_type = ast.literal_eval(pre_info)["result"].split("###")
            self.check_model_in_white_ls(model_and_type, node_ip)

    def check_ssd_model_not_in_cluster(self, pool):
        nodes = pool.getExpansionNodeList()
        LOGGER.logInfo("ExpansionNodeList:{}".format(nodes))
        ssh = None
        for node in nodes:
            try:
                get_index_cmd = "awk -F '|' 'NR==1{for(i=1;i<=NF;i++){if($i~/VMR|Type/)print i}}' /proc/smio_host"
                ssh = common.get_ssh_conn(node)
                index_info = ssh.execCmdWithTimout(get_index_cmd, common.HOST_CMD_SHORT_TIMEOUT)
                # 去除ssh返回带的 命令字符串 和 [root@node76234 ~]# ]部分
                index_vmr, index_type = index_info.splitlines()[1:3]
                get_model_cmd = "cat /proc/smio_host | grep 'SSD' | awk -F '|' '{{print ${},${}}}'".format(
                    index_vmr, index_type)
                LOGGER.logInfo("get_model_cmd:{}".format(get_model_cmd))
                model_and_type = ssh.execCmdWithTimout(get_model_cmd, common.HOST_CMD_SHORT_TIMEOUT)
                node_ip = node.getIp()
                self.check_model_in_white_ls(model_and_type, node_ip)

            except ToolException as e:
                LOGGER.logInfo("ToolException:{}".format(e))
                self.err_msgs.append(e.getDes())
            finally:
                if ssh:
                    common.release_ssh_conn(ssh)

    def check_model_in_white_ls(self, model_and_type, node_ip):
        """  检查SSD盘的model号是否在trim白名单中  """
        # model_and_type如下：
        #                              VMR     Type
        #     HUAWEI,HSSD-D6223AV1T6N,1012  SAS_SSD
        #        ATA,ST4000NM0033-9ZM,SNC6 SATA_HDD
        model_and_type_lines = model_and_type.splitlines()
        for model_and_type_line in model_and_type_lines:
            if "SSD" not in model_and_type_line:
                continue
            model = model_and_type_line.split(",")[1]
            if model not in self.trim_white_ls:
                self.models_not_in_white_ls.add(str([node_ip, model]))
