# coding: UTF-8

import json
import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil
from com.huawei.ism.exception import IsmException

import common
from ds_rest_util import CommonRestService

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "check_storage_node_iscsi"


def execute(rest):
    """
    检查VBS节点是否开启ISCSI
    :param rest: 集群rest连接
    :return:检查结果
    """
    ret_list = []
    dev_node = py_java_env.get("devInfo")
    LOGGER.logInfo("check_storage_node_iscsi begin ...")
    try:
        # 812之前的版本不涉及该巡检项
        LOGGER.logInfo()
        product_version = str(dev_node.getProductVersion())
        if not is_involve_product_version(product_version):
            return common.INSPECT_NOSUPPORT, "", ""
        # 获取存储节点
        storage_node_ip_list = get_storage_node(rest, dev_node, ret_list)
        vbs_client_list = get_vbs_node(rest, dev_node, ret_list)
        # 过滤出含VBS服务的存储节点
        vbs_storage_nodes = get_vbs_storage_node(vbs_client_list, storage_node_ip_list)

        warning_manager_ip_list = list()
        # 查看iscsi开关是否开启
        get_iscsi_switch(vbs_storage_nodes, warning_manager_ip_list)

        # 构建manager_ip_list
        manager_ip_list = list()
        generate_manager_ip_list(vbs_storage_nodes, manager_ip_list)

        # 查看iscsi状态
        get_iscsi_status(rest, warning_manager_ip_list, dev_node, manager_ip_list, ret_list)

        if len(warning_manager_ip_list) > 0:
            warning_manager_ip_list = [str(warning_manager_ip) for warning_manager_ip in warning_manager_ip_list]
            LOGGER.logInfo("warning_manager_ip_list:" + json.dumps(warning_manager_ip_list))
            return (
                common.INSPECT_UNNORMAL,
                "\n".join(ret_list),
                common.get_err_msg(LANG, "check.storage.node.iscsi.not.pass").format(warning_manager_ip_list))
        return common.INSPECT_PASS, "\n".join(ret_list), ""

    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        return (
            common.INSPECT_UNNORMAL,
            "",
            common.get_err_msg(LANG, "query.result.abnormal"),
        )


def is_storage_node(role_list):
    """
    判断是否是存储节点
    """
    for role in role_list:
        if role == "storage":
            return True
    return False


def generate_manager_ip_list(vbs_storage_nodes, manager_ip_list):
    """
    构造管理节点IPList
    """
    if vbs_storage_nodes is None:
        return
    for vbs_storage_node in vbs_storage_nodes:
        manager_ip_list.append(vbs_storage_node.get("manageIp"))


def get_vbs_node(rest, dev_node, ret_list):
    """
    获取VBS节点
    """
    base_uri = RestUtil.getDstorageUrlHead(dev_node)
    cmd_str = "{}/dsware/service/cluster/dswareclient/queryDSwareClientHost".format(base_uri)
    ret_list.append(cmd_str)
    vbs_nodes_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    ret_list.append(str(vbs_nodes_json))
    ret_list.append("\n")
    LOGGER.logInfo(vbs_nodes_json)
    result = vbs_nodes_json.get("result")
    if result != 0:
        return common.INSPECT_UNNORMAL, "\n".join(ret_list), common.get_err_msg(LANG, "query.result.abnormal"),

    vbs_client_list = vbs_nodes_json.get("clients")
    return vbs_client_list


def get_storage_node(rest, dev_node, ret_list):
    """
    获取存储节点
    """
    base_uri = RestUtil.getDstorageUrlHead(dev_node)
    cmd_str = "{}/api/v2/cluster/servers".format(base_uri)
    ret_list.append(cmd_str)
    cluster_nodes_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    ret_list.append(str(cluster_nodes_json))
    ret_list.append("\n")
    LOGGER.logInfo(cluster_nodes_json)

    result = cluster_nodes_json.get("result")
    LOGGER.logInfo(result)
    if result == 0:
        return common.INSPECT_UNNORMAL, "\n".join(ret_list), common.get_err_msg(LANG, "query.result.abnormal"),
    # 过滤出存储节点
    cluster_nodes = cluster_nodes_json.get("data")
    storage_node_ip_list = list()
    for cluster_node in cluster_nodes:
        role_list = cluster_node.get("role", {})
        if is_storage_node(role_list):
            storage_node_ip_list.append(cluster_node.get("management_ip"))
    LOGGER.logInfo("storage_node_ip_list:" + json.dumps(storage_node_ip_list))
    return storage_node_ip_list


def get_vbs_storage_node(vbs_client_list, storage_node_ip_list):
    """
    过滤出含VBS服务的存储节点
    """
    vbs_storage_nodes = list()
    for vbs_node in vbs_client_list:
        manage_ip = vbs_node.get("manageIp")
        if manage_ip in storage_node_ip_list:
            vbs_storage_nodes.append(vbs_node)
    LOGGER.logInfo("vbs_storage_nodes:" + json.dumps(vbs_storage_nodes))
    return vbs_storage_nodes


def get_iscsi_switch(vbs_storage_nodes, warning_manager_ip_list):
    """
    查看节点的iscsi开关是否开启
    """
    if vbs_storage_nodes is None:
        return
    for vbs_storage_node in vbs_storage_nodes:
        if vbs_storage_node is None:
            continue
        iscsi_switch = vbs_storage_node.get("iscsiSwitch")
        if iscsi_switch == "close":
            warning_manager_ip_list.append(vbs_storage_node.get("manageIp"))


def get_iscsi_status(rest, warning_manager_ip_list, dev_node, manager_ip_list, ret_list):
    """
    查看iscsi状态以及IP是否配置
    """
    base_uri = RestUtil.getDstorageUrlHead(dev_node)
    cmd_str = "{}/dsware/service/cluster/dswareclient/queryIscsiPortal".format(base_uri)
    ret_list.append(cmd_str)
    param = {
        "nodeMgrIps": json.dumps(manager_ip_list)
    }
    iscsi_nodes_json = CommonRestService.execute_post_request(rest, cmd_str, param)
    ret_list.append(str(iscsi_nodes_json).encode("utf-8").decode("unicode_escape"))
    ret_list.append("\n")
    iscsi_node_list = iscsi_nodes_json.get("nodeResultList")
    LOGGER.logInfo("iscsi_node_list:" + json.dumps(iscsi_node_list))
    for iscsi_node in iscsi_node_list:
        LOGGER.logInfo("iscsi_node:" + json.dumps(iscsi_node))
        node_mgr_ip = iscsi_node.get("nodeMgrIp")
        if node_mgr_ip in warning_manager_ip_list:
            LOGGER.logInfo("node_mgr_ip:{} already in warning_manager_ip_list".format(node_mgr_ip))
            continue
        iscsi_status = iscsi_node.get("status")
        LOGGER.logInfo("status:{}".format(iscsi_status))
        if iscsi_status == "failed":
            warning_manager_ip_list.append(node_mgr_ip)


def is_involve_product_version(product_version):
    """
    检查版本信息：812之前的版本暂不支持该巡检项
    """
    version = product_version.replace('.', "")
    if version[0:2] > "81":
        return True
    if version[0:2] == "81" and version[2].isdigit() and version[2] >= "2":
        return True
    if len(version) >= 5:
        if version[0:2] == "81" and version[2:4] == "RC" and version[4] >= "3":
            return True
    return False