#!/usr/bin/env python
# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

import ast
from com.huawei.ism.exception import IsmException

import common

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
HANDLE = py_java_env.get("preInspectHandle")
ITEM_ID = "check_vbs_node_version"
PRE_ITEM_ID = "pre_inspect_get_node_versions_info"

PRODUCT_VERSION = 'product_version'
PATCH_VERSION = 'patch_version'
HOTPATCH_VERSION = 'hotpatch_version'


def execute(rest):
    dev_node = py_java_env.get("devInfo")
    # 获取设备集群节点信息
    nodes = dev_node.getClusterNodes()
    LOGGER.logInfo("Checking vbs node version start, nodes {}.".format(str(nodes)))

    try:
        # 管理节点的版本信息字典
        mgr_node_versions = {}

        # VBS节点列表, 列表元素为VBS节点的字典信息
        vbs_node_list = []

        # 版本不一致的VBS节点IP列表
        inconsistent_versions_vbs_node_ip = []

        for node in nodes:
            node_ip = common.get_node_ip(node)
            LOGGER.logInfo("Node ip is {}.".format(node_ip))
            # 未勾选的节点跳过
            if not node.isSelected():
                LOGGER.logInfo("Node is not selected, ignore.")
                continue
            # 获得预巡检结果
            pre_info = HANDLE.getPreInspectResultWithLine(node_ip, PRE_ITEM_ID)
            LOGGER.logInfo("Checking vbs node version pre-info is {}.".format(str(pre_info)))

            if "dpc_compute node" in pre_info:
                LOGGER.logInfo("The current node is a DPC node, not involved.")
                continue
            result = ast.literal_eval(pre_info)["result"]
            # 解析预巡检脚本的结果数据, node_dict Example:{'product_version': '8.1.2', 'role': 'storage', 'patch_version': 'SPC100'}
            node_dict = parse_pre_infos(result)

            # 该节点为管理节点
            if is_management_node(node_dict):
                mgr_node_versions = node_dict

            # 该节点为VBS节点
            if is_vbs_node(node_dict):
                # 添加到VBS节点列表
                node_dict['node_ip'] = node_ip
                vbs_node_list.append(node_dict)

            # 记录每个节点信息
            LOGGER.logInfo("Checking vbs node version, node dict is {}".format(node_dict))

        # 勾选的节点不包含VBS节点,返回不涉及
        if not vbs_node_list:
            LOGGER.logInfo("The selected inspection nodes do not contain VBS nodes.")
            return common.INSPECT_NOSUPPORT, "The selected nodes do not contain VBS nodes.", ""

        if not mgr_node_versions:
            # 不存在管理节点版本信息
            return common.INSPECT_UNNORMAL, "No management node.", common.get_err_msg(LANG, "query.result.abnormal")

        LOGGER.logInfo("Checking vbs node version, management node versions is {}".format(mgr_node_versions))

        # 遍历VBS节点列表, 检查版本是否与管理节点一致
        for vbs_node in vbs_node_list:
            if not check_vbs_versions(vbs_node, mgr_node_versions):
                # 如果VBS版本与管理节点不一致, 记录节点IP地址
                inconsistent_versions_vbs_node_ip.append(vbs_node['node_ip'])

        if not inconsistent_versions_vbs_node_ip:
            # 不存在不一致的VBS节点, 巡检通过
            LOGGER.logInfo("Checking vbs node version pass.")
            return common.INSPECT_PASS, "Checking pass.", ""
        else:
            # 存在不一致的VBS节点, 巡检不通过
            # 返回版本不一致的VBS节点的IP地址
            err_code = "check.vbs.node.version.not.pass"
            LOGGER.logInfo("Checking vbs node not pass,inconsistent ip {}.".format(inconsistent_versions_vbs_node_ip))
            return (common.INSPECT_UNNORMAL, str(inconsistent_versions_vbs_node_ip),
                    common.get_err_msg(LANG, err_code, ",".join(inconsistent_versions_vbs_node_ip)))

    except (IsmException, Exception) as e:
        LOGGER.logException(e)
        LOGGER.logError("Checking vbs node version Exception.")
        return (common.INSPECT_UNNORMAL, "Checking vbs node Exception.",
                common.get_err_msg(LANG, "query.result.abnormal"))


def parse_pre_infos(result):
    node_dict = {}
    node_infos = result.split("\n")
    for info in node_infos:
        if '=' in info:
            key, value = info.split('=')
            # 去除字符串前后的空格, 防止配置文件的格式不统一
            node_dict[key.strip()] = value.strip()
    return node_dict


def is_management_node(node_dict):
    # 节点可能有多种身份,如:‘management,storage’,因此用in判断身份
    if "management" in node_dict['role']:
        return True
    else:
        return False


def is_vbs_node(node_dict):
    # 严格判断节点是否为VBS节点
    if node_dict['role'] == "compute":
        return True
    else:
        return False


# 检查vbs与management节点的版本信息是否匹配
def check_vbs_versions(vbs_node_dict, mgr_node_dict):
    # 检查vbs与management节点是否具有product_version与patch_version信息
    if PRODUCT_VERSION not in vbs_node_dict or PRODUCT_VERSION not in mgr_node_dict:
        return False
    if PATCH_VERSION not in vbs_node_dict or PRODUCT_VERSION not in mgr_node_dict:
        return False

    # product_version 或 patch_version不一致, 检查不通过
    if (vbs_node_dict[PRODUCT_VERSION] != mgr_node_dict[PRODUCT_VERSION] or
            vbs_node_dict[PATCH_VERSION] != mgr_node_dict[PATCH_VERSION]):
        return False

    # 管理节点和VBS节点均不具备hotpatch_version, 检查通过
    if HOTPATCH_VERSION not in vbs_node_dict and HOTPATCH_VERSION not in mgr_node_dict:
        return True

    # 若管理节点和VBS节点均具有hotpatch_version字段, 检查该字段
    if HOTPATCH_VERSION in vbs_node_dict and HOTPATCH_VERSION in mgr_node_dict:
        return vbs_node_dict[HOTPATCH_VERSION] == mgr_node_dict[HOTPATCH_VERSION]

    # 管理节点和VBS节点其中一个具备hotpatch_version, 另外一个不具备该字段, 检查不通过
    return False












