# coding: UTF-8

import common
from ds_rest_util import CommonRestService
from com.huawei.ism.exception import IsmException
import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil

ENV = py_java_env
LANG = common.getLang(py_java_env)


def execute(rest):
    return CloseAllocationSwitch(rest).do_close()


class CloseAllocationSwitch:
    def __init__(self, rest):
        self.rest = rest
        self.ret_list = list()
        self.pools = self.query_pools()
        self.update_disk_pools()

    def do_close(self):
        # 1、检查存储池是否需要关闭超分配开关
        self.check_is_need_close_switch()
        # 2、查询参数值
        self.query_pools_thin_rate_value()
        # 3、关闭超分配开关
        self.close_switch()
        # 4、再次查询参数值
        self.query_pools_thin_rate_value()
        # 5、检查关闭失败的池
        self.check_close_failed()
        return self.handle_result()

    @staticmethod
    def get_base_url():
        return RestUtil.getDstorageUrlHead(common.get_device_node(ENV))

    def query_pools(self):
        """
        查询存储池信息：ID、总容量、已分配容量
        """
        cmd_str = "{}/dsware/service/resource/queryStoragePool".format(self.get_base_url())
        self.ret_list.append(cmd_str)
        pools_json = CommonRestService.exec_get_gor_big_by_ds(self.rest, cmd_str)
        if not pools_json.get("storagePools"):
            return list()
        pools = [PoolCapacityInfo(pool.get('poolId'), pool.get('physicalTotalCapacity'), pool.get('allocatedCapacity'))
                 for pool in pools_json.get("storagePools")]
        for pool in pools:
            self.ret_list.append("[pool:{}] physicalTotalCapacity:{} allocatedCapacity:{}".format(
                pool.pool_id, pool.physical_total_capacity, pool.allocated_capacity))
        return pools

    def update_disk_pools(self):
        """
        查询并更新记录硬盘池信息：ID,所属的存储池ID
        """
        try:
            cmd_str = "{}/dsware/service/resource/queryPoolBasicInfo".format(self.get_base_url())
            self.ret_list.append(cmd_str)
            pools_json = CommonRestService.exec_get_gor_big_by_ds(self.rest, cmd_str)
            if not pools_json.get("pools"):
                self.ret_list.append("not found disk pools.")
                return
            for pool in pools_json.get("pools"):
                disk_pool_id = pool.get('poolId')
                storage_pool_id = pool.get('storagePoolId')
                self.ret_list.append("[disk_pool:{}] storagePoolId:{}".format(disk_pool_id, storage_pool_id))
                storage_pool = self.find_storage_pool_info(storage_pool_id)
                if storage_pool:
                    storage_pool.add_disk_pool_id(disk_pool_id)
        except IsmException as exception:
            # 兜底处理，如果接口不存在时可忽略该逻辑
            self.ret_list.append("not query disk pool.")

    def find_storage_pool_info(self, storage_pool_id):
        for pool in self.pools:
            if pool.pool_id == storage_pool_id:
                return pool
        return None

    def check_is_need_close_switch(self):
        for pool in self.pools:
            if not pool.is_need_close_switch():
                pool.set_err_msg(common.get_err_msg(LANG, 'config.thin.rate.capacity.check.failed'))

    def check_close_failed(self):
        for pool in self.pools:
            if not pool.is_need_close_switch():
                continue
            if pool.is_switch_open():
                pool.set_err_msg(common.get_err_msg(LANG, 'config.thin.rate.failed'))

    def query_pools_thin_rate_value(self):
        """
        查询存储池p_capacity_thin_rate的值
        """
        cmd_str = "{}/dsware/service/cluster/storagepool/queryPoolParametersOperation".format(self.get_base_url())
        self.ret_list.append(cmd_str)
        for pool in self.pools:
            res_json = CommonRestService.exec_get_gor_big_by_ds(self.rest, cmd_str, params={"poolId": pool.pool_id})
            for para in res_json.get("parameter"):
                if para.get('paraName') != "p_capacity_thin_rate":
                    continue
                self.ret_list.append("[pool:{}] p_capacity_thin_rate: {}".format(pool.pool_id, para.get('value')))
                pool.set_thin_rate_value(para.get("value"))

    def close_switch(self):
        for pool in self.pools:
            if not pool.is_need_close_switch() or not pool.is_switch_open():
                continue
            for pool_id in pool.get_pool_ids():
                self.close_disk_pool_switch(pool, pool_id)

    def close_disk_pool_switch(self, pool, disk_pool_id):
        cmd_str = "{}/dsware/service/cluster/storagepool/poolParametersOperation".format(self.get_base_url())
        self.ret_list.append(cmd_str)
        param = {
            "poolId": str(disk_pool_id),
            "opType": "modify",
            "parameter": [{
                "paraName": "p_capacity_thin_rate",
                "value": "100"
            }]
        }
        res_json = CommonRestService.execute_post_request(self.rest, cmd_str, param)
        self.ret_list.append("[pool id:{}] close_result: {}".format(disk_pool_id, res_json.get("result")))
        if res_json.get('result') != 0:
            pool.set_err_msg(common.get_err_msg(LANG, 'config.thin.rate.failed'))

    def handle_result(self):
        err_msgs = [pool.err_msg for pool in self.pools if pool.err_msg]
        if err_msgs:
            return common.INSPECT_UNNORMAL, str("\n".join(self.ret_list)), "\n".join(err_msgs)
        return common.INSPECT_PASS, str("\n".join(self.ret_list)), ""


class PoolCapacityInfo:
    def __init__(self, pool_id, physical_total_capacity, allocated_capacity):
        self.pool_id = pool_id
        self.physical_total_capacity = physical_total_capacity
        self.allocated_capacity = allocated_capacity
        self.p_capacity_thin_rate = '0'
        self.err_msg = ''
        self.disk_pool_ids = list()

    def is_need_close_switch(self):
        """
        当前池是否需要关闭超分配开关：已分配容量是否小于总容量
        """
        return self.allocated_capacity < self.physical_total_capacity

    def is_switch_open(self):
        """
        超分配开关是否为打开状态：p_capacity_thin_rate的值为0
        """
        return str(self.p_capacity_thin_rate) == '0'

    def set_thin_rate_value(self, thin_rate_value):
        self.p_capacity_thin_rate = thin_rate_value

    def set_err_msg(self, err_msg):
        self.err_msg = err_msg

    def add_disk_pool_id(self, disk_pool_id):
        self.disk_pool_ids.append(disk_pool_id)

    def get_pool_ids(self):
        """
        获取硬盘池或存储池ID的列表
        :return: 如果有硬盘池返回硬盘池ID列表，没有则返回存储池ID
        """
        return self.disk_pool_ids if self.disk_pool_ids else [self.pool_id]
