#!/usr/bin/env python
# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

from com.huawei.ism.exception import IsmException

import common

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
HANDLE = py_java_env.get("preInspectHandle")
ITEM_ID = "data_cluster_modules"
PRE_ITEM_ID = "pre_inspect_get_backplane_no"


def execute(rest):
    return DataClusterModulesChecker(rest).check()


class DataClusterModulesChecker:
    def __init__(self, rest):
        self.rest = rest
        self.nodes_info = ""
        self.is_contain_primary_management_node = False
        self.has_failed_node = False
        self.offline_dcms = []
        self.bnos_of_node = {}
        self.bnos_of_dcm = {}
        self.dcms_without_another_dcm = []
        self.nodes_without_dcm = []

    def check(self):
        dev_node = py_java_env.get("devInfo")
        observer = py_java_env.get("progressObserver")
        progress_map = {}

        try:
            progress_map[ITEM_ID] = 1
            observer.updateProgress(progress_map)

            self.parse_pre_inspection_info(dev_node)
            LOGGER.logInfo("bnos_of_node: " + str(self.bnos_of_node))
            if not self.bnos_of_node:
                # 如果不存在9950节点，则巡检通过
                return common.INSPECT_PASS, self.nodes_info, ""
            elif not self.is_contain_primary_management_node:
                # 如果不包含主管理节点，则巡检不通过
                err_msg = common.get_err_msg(LANG, "data.cluster.modules.result.missing")
                LOGGER.logInfo("lost primary node")
                return common.INSPECT_UNNORMAL, self.nodes_info, err_msg
            elif self.offline_dcms:
                # 如果存在离线数据集群模块，则巡检不通过
                offline_ips = ", ".join(str(ip) for ip in self.offline_dcms)
                err_msg = common.get_err_msg(LANG, "data.cluster.modules.result.offline", offline_ips)
                LOGGER.logInfo("dcm offline: " + offline_ips)
                return common.INSPECT_UNNORMAL, self.nodes_info, err_msg
            elif self.has_failed_node:
                # 如果存在查询失败的情况，则巡检不通过
                LOGGER.logInfo("has failed node:")
                return common.INSPECT_UNNORMAL, self.nodes_info, common.get_err_msg(LANG, "query.result.busy")

            self.analyze_unadded_dcm()
            if self.dcms_without_another_dcm or self.nodes_without_dcm:
                # 存在未添加的数据集群模块，则巡检不通过
                output_str = self._gen_fault_desc()
                err_msg = common.get_err_msg(LANG, "data.cluster.modules.result.abnormal", output_str)
                return common.INSPECT_UNNORMAL, self.nodes_info, err_msg
            else:
                return common.INSPECT_PASS, self.nodes_info, ""
        except (IsmException, Exception) as exception:
            LOGGER.logException(exception)
            return common.INSPECT_UNNORMAL, self.nodes_info, common.get_err_msg(LANG, "query.result.busy")

    def parse_pre_inspection_info(self, dev_node):
        for cluster_node in dev_node.getClusterNodes():
            if not cluster_node.isSelected():
                continue
            node_ip = common.get_node_ip(cluster_node)
            bno_info = HANDLE.getPreInspectResult(node_ip, PRE_ITEM_ID)
            LOGGER.logInfo("node_ip: " + str(node_ip) + " bno_info: " + bno_info)
            self._gen_node_info(node_ip, bno_info)
            if "Fail" in bno_info:
                self.has_failed_node = True
            if "###" in bno_info:
                # 预巡检信息包含###，表示为主管理节点
                self.is_contain_primary_management_node = True
                bno_info, dcm_bno_info = bno_info.split("###")
                if dcm_bno_info:
                    self.parse_dcm_bno(dcm_bno_info)
            if bno_info and "Fail" not in bno_info:
                node_ips = self.bnos_of_node.get(bno_info, list())
                node_ips.append(node_ip)
                self.bnos_of_node[bno_info] = node_ips

    def parse_dcm_bno(self, dcm_bno_info):
        dcm_bnos = dcm_bno_info.split(",")
        for dcm_bno in dcm_bnos:
            bno, dcm_ip = dcm_bno.split("@")
            if "Offline" in bno:
                LOGGER.logInfo("Offline:" + dcm_ip)
                self.offline_dcms.append(dcm_ip)
            elif "Fail" in bno:
                LOGGER.logInfo("Fail:" + dcm_ip)
                self.has_failed_node = True
            else:
                dcm_ips = self.bnos_of_dcm.get(bno, list())
                dcm_ips.append(dcm_ip)
                self.bnos_of_dcm[bno] = dcm_ips

    def analyze_unadded_dcm(self):
        dcm_bnos = set(self.bnos_of_dcm.keys())
        node_bnos = set(self.bnos_of_node.keys())
        LOGGER.logInfo("dcm_bnos: " + str(dcm_bnos))
        LOGGER.logInfo("node_bnos: " + str(node_bnos))
        rest_bnos = list(node_bnos - dcm_bnos)
        self.nodes_without_dcm = [self.bnos_of_node.get(node_bno)[0] for node_bno in rest_bnos]
        for ips in self.bnos_of_dcm.values():
            if len(ips) == 1:
                self.dcms_without_another_dcm.append(ips[0])

    def _gen_fault_desc(self):
        dcm_ips_str = ", ".join(str(ip) for ip in self.dcms_without_another_dcm)
        dcm_ips_str = "data cluster modules: " + dcm_ips_str if dcm_ips_str else dcm_ips_str
        node_ips_str = ", ".join(str(ip) for ip in self.nodes_without_dcm)
        node_ips_str = " storage node: " + node_ips_str if node_ips_str else node_ips_str
        return dcm_ips_str + node_ips_str

    def _gen_node_info(self, node_ip, bno_info):
        if "###" in bno_info:
            node_bno, dcm_bno_info = bno_info.split("###")
            node_bno = node_bno if node_bno else common.get_err_msg(LANG, "not.involved")
            dcm_bno_info = dcm_bno_info if dcm_bno_info else "None"
            self.nodes_info += node_ip + "\nnode_bno:" + node_bno + "\ndcm_bno_set:" + dcm_bno_info + "\n\n"
        else:
            node_bno = "node_bno:{}".format(bno_info) if bno_info else common.get_err_msg(LANG, "not.involved")
            self.nodes_info += node_ip + "\nnode_bno:" + node_bno + "\n\n"
