# coding: UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.
import json

from com.huawei.ism.exception import IsmException

import cli_util
import common
import hardware_util
import redfish_util
import entity

ITEM_ID = "check_item_es3000_version"
LANG = common.getLang(py_java_env)
HANDLE = py_java_env.get("preInspectHandle")
TOOL_VERSION_KEY = "ES_3000_Tool_Ver"
DRIVER_VERSION_KEY = "ES_3000_Driver_Ver"
FIRMWARE_VERSION_KEY = "ES_3000_{}_FW_Ver"
FIRMWARE_MODEL = {"HSSD-D6|HWE5": 5, "HSSD-D7|HWE6": 6}
DRIVER_KEY = "driver"
TOOL_KEY = "tool"


def execute(rest_conn):
    """
    检查nvme版本
    :param rest_conn: 集群rest连接
    :return:
    """
    item = common.ItemEntity(py_java_env, ITEM_ID, PY_LOGGER, __file__)
    item.update_progress(1)
    try:
        check_nvme_version(item.build_param_dict(), None, item.err_info_list, item.ret_list)
        return item.get_check_result()
    except (IsmException, Exception) as exception:
        item.logger.logException(exception)
        return item.get_failed_result(common.get_err_msg(LANG, "query.result.abnormal"))


@common.check_all_ibmc_node
def check_nvme_version(param_dict, cluster_dev_node, err_info_list, ret_list):
    cluster_node_ip = common.get_node_ip(cluster_dev_node)
    redfish = common.get_redfish_conn(cluster_dev_node)
    ret_by_node = []
    logger = param_dict.get("logger")
    context = param_dict.get("env")
    try:
        ret = HANDLE.getTargetCheckResult(cluster_node_ip, ITEM_ID)
        data = json.loads(ret)
        origin_info = data.get("original_info")
        ret_by_node.append(origin_info)
        # 若没有nvme信息，则无需检查
        if 'nvme' not in origin_info:
            return
        version_list = data.get("result")
        uid, _ = redfish_util.get_product_unique_id(redfish, logger)
        for version_info in version_list:
            if isinstance(version_info, dict):
                check_driver_and_tool_version(version_info, (context, uid, cluster_node_ip), err_info_list, logger)
            else:
                # 检查固件版本
                check_firmware_version(version_info, (context, uid, cluster_node_ip), err_info_list, logger)
    except Exception as exception:
        logger.logException(exception)
        err_info_list.append(common.get_err_msg(LANG, "item.check.abnormal", cluster_node_ip))
    finally:
        ret_list.append(cli_util.get_format_header_ret(cluster_node_ip, "\n".join(ret_by_node)))


def check_driver_and_tool_version(version_info, args, err_info_list, logger):
    # 检查驱动版本
    if DRIVER_KEY == version_info.get("type"):
        check_driver_version(version_info, args, err_info_list, logger)
    # 检查工具版本
    elif TOOL_KEY == version_info.get("type"):
        check_tool_version(version_info, args, err_info_list, logger)


def check_driver_version(version_info, args, err_info_list, logger):
    """
    检查驱动版本
    :param version_info: 版本信息
    :param args: 固定参数，context,uid,cluster_node_ip
    :param err_info_list: 错误信息列表
    :param logger: 日志
    :return: 无
    """
    driver_version = version_info.get("version")
    mapping_version = hardware_util.get_mapping_fw_version(
        args[0], args[1], DRIVER_VERSION_KEY)
    logger.logInfo("[IP address:{}] FusionStorage es3000 driver Version: {}, mapping:{}"
                   .format(args[2], driver_version, mapping_version))
    if entity.Compare.compare_digital_version(
            driver_version, mapping_version) < 0:
        msg = common.get_err_msg(LANG, "es3000.driver.mapping.not.pass",
                                 (args[2], driver_version, mapping_version))
        err_info_list.append(msg)


def check_tool_version(version_info, args, err_info_list, logger):
    """
    检查工具版本
    :param version_info: 版本信息
    :param args: 固定参数，context,uid,cluster_node_ip
    :param err_info_list: 错误信息列表
    :param logger: 日志
    :return: 无
    """
    tool_version = version_info.get("version")
    mapping_version = hardware_util.get_mapping_fw_version(
        args[0], args[1], TOOL_VERSION_KEY)
    logger.logInfo("[IP address:{}] FusionStorage es3000 tool Version: {}, mapping:{}"
                   .format(args[2], tool_version, mapping_version))
    if entity.Compare.compare_digital_version(
            tool_version, mapping_version) < 0:
        msg = common.get_err_msg(LANG, "es3000.tool.mapping.not.pass",
                                 (args[2], tool_version, mapping_version))
        err_info_list.append(msg)


def check_firmware_version(version_info, args, err_info_list, logger):
    """
    检查固件版本
    :param version_info: 版本信息
    :param args: 固定参数，context,uid,cluster_node_ip
    :param err_info_list: 错误信息列表
    :param logger: 日志
    :return: 无
    """
    for firmware_info in version_info:
        firmware_version = firmware_info.get("version")
        model_filter = firmware_info.get("filter")
        esn = firmware_info.get("esn")
        model = FIRMWARE_MODEL.get(model_filter)
        mapping_version = hardware_util.get_mapping_fw_version(args[0], args[1],
                                                               FIRMWARE_VERSION_KEY.format(model))
        logger.logInfo("[IP address:{}] FusionStorage es3000 V{} firmware Version: {},mapping:{}"
                       .format(args[2], model, firmware_version, mapping_version))
        if entity.Compare.compare_digital_version(
                firmware_version, mapping_version) < 0:
            msg = common.get_err_msg(LANG, "es3000.fw.ver.not.match",
                                     (args[2], model, esn, firmware_version, mapping_version))
            err_info_list.append(msg)
