# coding: UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil
from com.huawei.ism.exception import IsmException
import common
from ds_rest_util import CommonRestService

# noinspection PyUnresolvedReferences
LANG = common.getLang(py_java_env)
# noinspection PyUnresolvedReferences
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "fsm_and_zk_install_area"
# noinspection PyUnresolvedReferences
PY_JAVA_ENV = py_java_env


class ResultException(Exception):
    def __init__(self, msg, param=None):
        self.msg = msg
        self.param = param


class CheckInstallArea:
    def __init__(self, rest, env):
        self.rest = rest
        self.env = env
        self.all_ret_list = []
        self.progressMap = {}

    def execute_check(self):
        dev_node = self.env.get("devInfo")
        observer = self.env.get("progressObserver")
        try:
            self.update_progress(observer, 1)
            self.base_uri = RestUtil.getDstorageUrlHead(dev_node)
            storage_pools = self.get_all_storage_pool()
            self.is_storage_pool_exist(storage_pools)
            self.is_storage_pool_name_trust(storage_pools)

            self.update_progress(observer, 10)
            pool_name_to_ip = self.get_pool_name_to_management_ips(storage_pools)
            all_ip = self.get_all_management_ips(pool_name_to_ip)

            self.update_progress(observer, 30)
            all_fsm_mgr_ip = self.get_fsm_management_ip()
            if len(all_fsm_mgr_ip) != 2:
                raise ResultException("fsm.node.count.not.pass")
            fsm_not_in_storage_pool_ip = []
            self.get_not_in_storage_pool_ip(all_ip, all_fsm_mgr_ip,
                                            fsm_not_in_storage_pool_ip)

            self.update_progress(observer, 50)
            zk_cluster_list = self.get_zookeeper_cluster_ip()
            zk_not_in_storage_pool_ip = []
            self.get_not_in_storage_pool_ip(all_ip, zk_cluster_list,
                                            zk_not_in_storage_pool_ip)
            self.is_ip_all_in_target_ips(fsm_not_in_storage_pool_ip,
                                         zk_not_in_storage_pool_ip)

            self.update_progress(observer, 90)
            self.execute_check_pool_name(pool_name_to_ip,
                                         all_fsm_mgr_ip,
                                         zk_cluster_list)
            return common.INSPECT_PASS, ""
        except ResultException as exception:
            return (
                common.INSPECT_UNNORMAL,
                common.get_err_msg(LANG, exception.msg, exception.param)
            )
        except (IsmException, Exception) as exception:
            LOGGER.logException(exception)
            return (
                common.INSPECT_UNNORMAL,
                common.get_err_msg(LANG, "query.result.abnormal")
            )

    def update_progress(self, observer, number):
        self.progressMap[ITEM_ID] = number
        observer.updateProgress(self.progressMap)

    def execute_check_pool_name(self, pool_name_to_ip,
                                all_fsm_mgr_ip, zk_cluster_list):
        fsm_not_trust_ip = []
        fsm_not_trust_pool_name = []
        for item_ip in all_fsm_mgr_ip:
            self.get_not_trust_pool_name_and_ip(pool_name_to_ip,
                                                item_ip,
                                                fsm_not_trust_ip,
                                                fsm_not_trust_pool_name)
        zk_not_trust_ip = []
        zk_not_trust_pool_name = []
        for item_ip in zk_cluster_list:
            self.get_not_trust_pool_name_and_ip(pool_name_to_ip,
                                                item_ip,
                                                zk_not_trust_ip,
                                                zk_not_trust_pool_name)
        self.is_pool_name_trust(fsm_not_trust_ip,
                                fsm_not_trust_pool_name,
                                zk_not_trust_ip,
                                zk_not_trust_pool_name)

    def get_all_storage_pool(self):
        """
        获取所有存储池
        :return:
        """
        cmd_str = (
            "{}/dsware/service/resource/"
            "queryStoragePool?baseInfo=true".format(self.base_uri)
        )
        self.all_ret_list.append(cmd_str)
        pools_json = CommonRestService.exec_get_gor_big_by_ds(
            self.rest, cmd_str
        )
        self.all_ret_list.append(str(pools_json))
        storage_pools = pools_json.get("storagePools", [])
        return storage_pools

    def is_storage_pool_exist(self, storage_pools):
        """
        检查是否有存储池
        :param storage_pools:
        :return:
        """
        if not storage_pools:
            raise ResultException("fsm.device.not.has.pool")

    def is_storage_pool_name_trust(self, storage_pools):
        """
        检查是否有可信存储池
        :param storage_pools:
        :return:
        """
        tag = 0
        for recode in storage_pools:
            pool_name = recode.get("poolName")
            if self.is_storage_pool_name_trust_zone(pool_name):
                tag = 1
        if not tag:
            raise ResultException("query.pool.name.not.pass")

    def get_pool_name_to_management_ips(self, storage_pools):
        """
        获取所有节点管理ip
        :param storage_pools:所有存储池
        :return:
        """
        pool_name_to_management_ips = {}
        for recode in storage_pools:
            pool_name = recode.get("poolName")
            pool_id = recode.get("poolId")
            cmd_str = ("{}/dsware/service/cluster/storagepool/"
                       "queryStorageNodeInfo?"
                       "poolId={}".format(self.base_uri, pool_id))
            self.all_ret_list.append(cmd_str)
            node_json = CommonRestService.exec_get_gor_big_by_ds(
                self.rest, cmd_str
            )
            self.all_ret_list.append(str(node_json))
            if not node_json.get("nodeInfo", []):
                continue
            nodes_info = node_json.get("nodeInfo", [])
            self.get_management_ip(
                nodes_info,
                pool_name,
                pool_name_to_management_ips
            )
        return pool_name_to_management_ips

    def get_management_ip(self, nodes_info, pool_name, pool_name_ip):
        # 8.1.RC1，存在问题，接口实现与OMRP不符，需要遍历两层nodeInfo
        # 8.1.RC2发布前及之后的8.1.0（TR6）修复了此问题，只有一层nodeInfo
        product_version = str(self.env.get("devInfo").getProductVersion())
        if product_version == "8.1.RC1":
            for node_info in nodes_info:
                nodes = node_info.get("nodeInfo", [])
                self.get_pool_name_to_node_ip(nodes, pool_name, pool_name_ip)
        else:
            self.get_pool_name_to_node_ip(nodes_info, pool_name, pool_name_ip)

    def get_pool_name_to_node_ip(self, nodes, pool_name, pool_name_ip):
        for node in nodes:
            node_mgr_ip = node.get("nodeMgrIp")
            if pool_name_ip.get(pool_name):
                pool_name_ip[pool_name].append(node_mgr_ip)
            else:
                pool_name_ip[pool_name] = [node_mgr_ip, ]

    def get_all_management_ips(self, pool_name_to_management_ips):
        """
        获取所有存储池的ip列表
        :param pool_name_to_management_ips:
        :return:
        """
        all_ip = []
        for value in pool_name_to_management_ips.values():
            all_ip.extend(value)
        return all_ip

    def get_fsm_management_ip(self):
        """
        获取fsm的管理ip
        :return:
        """
        cmd_str = "{}/api/v2/cluster/servers".format(self.base_uri)
        self.all_ret_list.append(cmd_str)
        fsm_mgr_ip_json = CommonRestService.exec_get_gor_big_by_ds(
            self.rest,
            cmd_str
        )
        self.all_ret_list.append(str(fsm_mgr_ip_json))
        fsm_mgr_ip_list = fsm_mgr_ip_json.get("data", [])
        all_fsm_mgr_ip = []
        for fsm_mgr_ip_info in fsm_mgr_ip_list:
            if "management" not in fsm_mgr_ip_info.get("role", []):
                continue
            fsm_mgr_ip = fsm_mgr_ip_info.get("management_ip")
            all_fsm_mgr_ip.append(fsm_mgr_ip)
        return all_fsm_mgr_ip

    def get_zookeeper_cluster_ip(self):
        """
        获取zk集群ip
        :return:
        """
        cmd_str = (
            "{}/dsware/service/resource/"
            "queryDswareAllNodeIpInfo".format(self.base_uri)
        )
        self.all_ret_list.append(cmd_str)
        zk_cluster_json = CommonRestService.exec_get_gor_big_by_ds(
            self.rest,
            cmd_str
        )
        self.all_ret_list.append(str(zk_cluster_json))
        zk_cluster_list = zk_cluster_json.get("zkCluster", [])
        return zk_cluster_list

    def get_not_in_storage_pool_ip(self, all_ip, item_ip_list, not_pass_ip):
        """
        获取不在存储池中的所有ip
        :param all_ip:
        :param item_ip_list:
        :param not_pass_ip:
        :return:
        """
        for item_ip in item_ip_list:
            if item_ip not in all_ip:
                not_pass_ip.append(item_ip)

    def is_ip_all_in_target_ips(self, fsm_not_pass_ip, zk_not_pass_ip):
        """
        判断zk和fsm是否都存在不在存储池的ip，并给出具体错误
        :param fsm_not_pass_ip:
        :param zk_not_pass_ip:
        :return:
        """
        if fsm_not_pass_ip and zk_not_pass_ip:
            raise ResultException(
                "fsm.and.zk.ip.not.pass",
                (",".join(fsm_not_pass_ip), ",".join(zk_not_pass_ip))
            )
        if fsm_not_pass_ip:
            raise ResultException(
                "fsm.or.zk.ip.not.pass",
                ("FSM", ",".join(fsm_not_pass_ip))
            )
        if zk_not_pass_ip:
            raise ResultException(
                "fsm.or.zk.ip.not.pass",
                ("ZK", ",".join(zk_not_pass_ip))
            )

    def get_not_trust_pool_name_and_ip(self, pool_name_to_ips, item_ip,
                                       not_trust_ip, not_trust_name):
        """
        检查fsm和zk所属存储池名称是否可信
        :param pool_name_to_ips:存储池与对应的ip
        :param item_ip:需要检测的管理ip
        :param not_trust_ip:
        :param not_trust_name:
        :return:
        """
        for pool_name, node_mgr_ip in pool_name_to_ips.items():
            if item_ip in node_mgr_ip:
                if not self.is_storage_pool_name_trust_zone(pool_name):
                    not_trust_ip.append(item_ip)
                    not_trust_name.append(pool_name)

    def is_storage_pool_name_trust_zone(self, pool_name):
        """
        判断存储池名是否可信
        :param pool_name:
        :return:
        """
        return "-TRU-" in pool_name

    def is_pool_name_trust(self, fsm_not_trust_ip, fsm_not_trust_pool_name,
                           zk_not_trust_ip, zk_not_trust_pool_name):
        """
        判断fsm和zk的不可信信息，并给出具体返回
        :param fsm_not_trust_ip:
        :param fsm_not_trust_pool_name:
        :param zk_not_trust_ip:
        :param zk_not_trust_pool_name:
        :return:
        """
        fsm_not_trust_pool_name = list(set(fsm_not_trust_pool_name))
        zk_not_trust_pool_name = list(set(zk_not_trust_pool_name))
        if fsm_not_trust_ip and zk_not_trust_ip:
            raise ResultException(
                "fsm.and.zk.pool.name_not.trust",
                (",".join(fsm_not_trust_ip), ",".join(fsm_not_trust_pool_name),
                 ",".join(zk_not_trust_ip), ",".join(zk_not_trust_pool_name))
            )
        if fsm_not_trust_ip:
            raise ResultException(
                "fsm.or.zk.pool.name_not.trust",
                ("FSM", ",".join(fsm_not_trust_ip),
                 ",".join(fsm_not_trust_pool_name))
            )
        if zk_not_trust_ip:
            raise ResultException(
                "fsm.or.zk.pool.name_not.trust",
                ("ZK", ",".join(zk_not_trust_ip),
                 ",".join(zk_not_trust_pool_name))
            )


def execute(rest):
    fsm_and_zk_install_area = CheckInstallArea(rest, PY_JAVA_ENV)
    flag, msg = fsm_and_zk_install_area.execute_check()
    return flag, "\n".join(fsm_and_zk_install_area.all_ret_list), msg
