# coding: UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil
from com.huawei.ism.exception import IsmException

import common
from ds_rest_util import CommonRestService

# noinspection PyUnresolvedReferences
LANG = common.getLang(py_java_env)
# noinspection PyUnresolvedReferences
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "fsm_rack_distribution"
# noinspection PyUnresolvedReferences
PY_JAVA_ENV = py_java_env


def execute(rest):
    """
    检查Fsm节点的机柜分布
    :param rest:
    :return:
    """
    check_item = CheckFsmRackDistribution(rest, PY_JAVA_ENV, LANG, LOGGER)
    check_item.execute_check()
    return check_item.get_result()


class CheckFsmRackDistribution:
    def __init__(self, rest, env, lang, logger):
        self.rest = rest
        self.lang = lang
        self.env = env
        self.observer = env.get("progressObserver")
        self.logger = logger
        self.all_ret_list = []
        self.progressMap = {}
        self.msg_list = []
        self.result = ""

    def execute_check(self):
        try:
            dev_node = self.env.get("devInfo")
            self.update_progress(1)
            self.base_uri = RestUtil.getDstorageUrlHead(dev_node)
            pools_json = self.get_all_storage_pool()
            # 没有存储池，不通过
            if not pools_json.get("storagePools"):
                self.msg_list.append(
                    common.get_err_msg(self.lang, "fsm.device.not.has.pool"))
                self.result = common.INSPECT_UNNORMAL
                return
            storage_pools = pools_json.get("storagePools", [])
            storage_pool_ids = []
            for storage_pool in storage_pools:
                storage_pool_ids.append(storage_pool.get("poolId"))
            self.update_progress(30)
            management_ips = self.get_management_ips()
            # 管理节点数不为2，不通过
            if len(management_ips) != 2:
                self.msg_list.append(
                    "fsm nodes not equals two!")
                self.result = common.INSPECT_UNNORMAL
                return
            rack_encoding = []
            temp_management_ips = management_ips[:]
            # 将存储池Id作为参数查询各存储池中的节点信息
            for storage_pool_id in storage_pool_ids:
                node_infos_json = self.get_nodes_info(storage_pool_id)
                node_infos = node_infos_json.get("nodeInfo", [])
                self.match_management_ips_with_node_infos(management_ips, node_infos, rack_encoding)
                if len(management_ips) == 0:
                    break
            self.update_progress(80)
            # 存在FSM节点的管理ip在存储池中没找到
            if len(management_ips) != 0:
                self.msg_list.append(
                    common.get_err_msg(LANG, "fsm.node.not.in.pool", ",".join(management_ips)))
                self.result = common.INSPECT_UNNORMAL
                return
            # 两个FSM节点的机柜编码一致
            if len(set(rack_encoding)) == 1:
                self.msg_list.append(
                    common.get_err_msg(LANG, "fsm.node.same.rack",
                                       (temp_management_ips[0], temp_management_ips[1], rack_encoding[0])))
                self.result = common.INSPECT_UNNORMAL
                return
            self.result = common.INSPECT_PASS
        except (IsmException, Exception) as exception:
            self.logger.logException(exception)
            self.msg_list.append(
                common.get_err_msg(LANG, "query.result.abnormal"))
            self.result = common.INSPECT_UNNORMAL

    def get_nodes_info(self, storage_pool_id):
        """
        获取存储池中的节点信息
        :return: 存储池中的节点信息
        """
        cmd_str = "{}/dsware/service/cluster/storagepool/queryStorageNodeInfo?poolId={}".format(self.base_uri,
                                                                                                storage_pool_id)
        self.all_ret_list.append(cmd_str)
        node_infos_json = CommonRestService.exec_get_gor_big_by_ds(self.rest, cmd_str)
        self.all_ret_list.append(str(node_infos_json))
        return node_infos_json

    def get_all_storage_pool(self):
        """
        获取所有存储池
        :return: 所有存储池
        """
        cmd_str = "{}/dsware/service/resource/queryStoragePool?baseInfo=true".format(
            self.base_uri
        )
        self.all_ret_list.append(cmd_str)
        pools_json = CommonRestService.exec_get_gor_big_by_ds(self.rest, cmd_str)
        self.all_ret_list.append(str(pools_json))
        return pools_json

    def get_management_ips(self):
        """
        获取管理节点的ip
        :return: 管理节点的ip
        """
        management_ips = []
        cmd_str = "{}/api/v2/cluster/servers".format(
            self.base_uri
        )
        self.all_ret_list.append(cmd_str)
        datas_json = CommonRestService.exec_get_gor_big_by_ds(self.rest, cmd_str)
        self.all_ret_list.append(str(datas_json))
        datas = datas_json.get("data", [])
        for data in datas:
            roles = data.get("role")
            if "management" in roles:
                management_ips.append(data.get("management_ip"))
        return management_ips

    def match_management_ips_with_node_infos(self, management_ips, node_infos, rack):
        """
        将FSM的管理ip和各节点信息匹配，将匹配到的机柜编码保存起来
        :return:
        """
        if len(management_ips) == 0:
            return
        # 8.1.RC1，存在问题，接口实现与OMRP不符，需要遍历两层nodeInfo
        # 8.1.RC2发布前及之后的8.1.0（TR6）修复了此问题，只有一层nodeInfo
        product_version = str(self.env.get("devInfo").getProductVersion())
        if product_version == "8.1.RC1":
            for node_info in node_infos:
                nodes = node_info.get("nodeInfo", [])
                self.get_node_info(management_ips, nodes, rack)
        else:
            self.get_node_info(management_ips, node_infos, rack)

    def get_node_info(self, management_ips, nodes, rack):
        for node in nodes:
            node_management_ip = node.get("nodeMgrIp", "")
            if node_management_ip in management_ips:
                management_ips.remove(node_management_ip)
                rack.append(node.get("rack"))
                continue

    def update_progress(self, number):
        """
        更新进度
        :return:
        """
        self.progressMap[ITEM_ID] = number
        self.observer.updateProgress(self.progressMap)

    def get_result(self):
        """
        获取检查结果
        :return: 检查结果
        """
        return self.result, "\n".join(self.all_ret_list), "\n".join(
            self.msg_list)
