# coding: UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.
from com.huawei.ism.exception import IsmException

import cli_util
import common
import constant
import entity
import hardware_util
import redfish_util

ITEM_ID = "ibmc_firmware_version"
LANG = common.getLang(py_java_env)
IBMC_FW_VER_KEY = "Main_Bord_iBMC_Ver_{}"
MSG_KEY = "Current version : {} and matching version : {}"


def execute(rest_conn):
    """
    检查IBMC配套版本
    :param rest_conn: 集群rest连接
    :return:
    """
    item = common.ItemEntity(py_java_env, ITEM_ID, PY_LOGGER, __file__)
    item.update_progress(1)
    try:
        check(item.build_param_dict(), None, item.err_info_list, item.ret_list)
        return item.get_check_result()
    except (IsmException, Exception) as exception:
        item.logger.logException(exception)
        return item.get_failed_result(common.get_err_msg(LANG, "query.result.abnormal"))


@common.check_all_ibmc_node
def check(param_dict, cluster_dev_node, err_info_list, ret_list):
    """
    BMC版本号的单节点检查逻辑，基于查询出的版本号与配套表中定义的版本号比较得出是否符合配套预期
    :param param_dict: 参数字典
    :param cluster_dev_node: 节点对象
    :param err_info_list: 检查结果中的错误消息
    :param ret_list: 原始信息
    :return: 空
    """
    node_ret = []
    cluster_node_ip = common.get_node_ip(cluster_dev_node)
    redfish = common.get_redfish_conn(cluster_dev_node)
    logger = param_dict.get("logger")
    context = param_dict.get("env")
    try:
        # 获取配套表推荐的版本
        matching_version, ret = get_matching_fw_version(redfish, logger, context)

        # 获取节点的BMC版本号
        current_version, ret = get_dev_ibmc_version(redfish, logger)
        node_ret.append(ret)
        # 版本间比较
        msg = MSG_KEY.format(current_version, matching_version)
        node_ret.append(msg)
        if entity.Compare.compare_digital_version(current_version, matching_version) < 0:
            msg = common.get_err_msg(LANG, "version.mapping.not.pass", (cluster_node_ip, current_version,
                                                                        matching_version))
            err_info_list.append(msg)
    except Exception as exception:
        logger.logException(exception)
        err_info_list.append(common.get_err_msg(LANG, "query.result.abnormal"))
    finally:
        ret_list.append(cli_util.get_format_header_ret(cluster_node_ip, "\n".join(node_ret)))


def get_dev_ibmc_version(redfish, logger):
    """
    获取服务器硬件自身的IBMC版本号
    :param redfish: redfish连接
    :param logger: 日志实体
    :return: 版本号，原始信息
    """
    resource = redfish_util.get_managers_info(redfish, logger)
    current_version = resource.get("FirmwareVersion")
    # 去掉其中的括号的描述
    current_version = current_version.split("(")[0].strip()
    return current_version, str(resource)


def get_matching_fw_version(redfish, logger, context):
    """
    获取配套表中配置的推荐版本
    :param redfish: redfish链接
    :param logger: 日志
    :param context:脚本的上下文信息
    :return: 推荐的版本，回显信息
    """
    main_bord_num, ret = redfish_util.get_ibmc_main_board_num(redfish, logger)
    uid, _ = redfish_util.get_product_unique_id(redfish, logger)
    # 判断是不是VE机型主板（1620s 2.0主板）
    if hardware_util.is_ve_supported_main_board(uid):
        main_bord_num = main_bord_num + constant.VE_SUPPORTED_NUM
    elif hardware_util.is_v6_supported_main_board(uid):
        main_bord_num += constant.V6_SUPPORTED_NUM
    logger.logInfo("iBMC fw main_bord_num: {}".format(main_bord_num))
    return hardware_util.get_mapping_fw_version(context, uid, IBMC_FW_VER_KEY.format(main_bord_num)), ret
