# coding: UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
import time
import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil
from com.huawei.ism.exception import IsmException

import common
from ds_rest_util import CommonRestService
import HTMLParser

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "major_alarm"

MAJOR_ALARM = 5
CRITICAL_ALARM = 6
WARNING_ALARM = 3


def get_alarm_count(rest, base_uri, alarm_level):
    cmd_get_major_alarm_count = "{}/api/v2/common/alarm_count?filter=level::{}%20and%20alarmStatus::1".\
                                format(base_uri, alarm_level)
    rsp_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_get_major_alarm_count)
    if not rsp_json.get("data"):
        return -1
    alarm_count = int(rsp_json.get("data").get("count"))
    return alarm_count


def get_alarm_info(rest, base_uri, alarm_level, ret_list):
    alarm_info_str_list = []
    cmd_get_alarm_info = "{}/api/v2/common/alarms?range=%5B0-20%5D&filter=level::{}%20and%20alarmStatus::1".\
                         format(base_uri, alarm_level)
    rsp_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_get_alarm_info)
    if not rsp_json.get("data"):
        return alarm_info_str_list
    alarms = rsp_json.get("data", [])
    alarm_info_str = common.get_err_msg(LANG, "alarm.info") 

    html_parser = HTMLParser.HTMLParser()
    for alarm in alarms:
        alarm_id = html_parser.unescape(alarm.get("alarm_id"))
        alarm_name = html_parser.unescape(alarm.get("description"))
        alarm_create_time = alarm.get("create_time")
        formatted_date = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(alarm_create_time))
        alarm_detail = html_parser.unescape(alarm.get("detail"))
        alarm_level_str = common.get_err_msg(LANG, "alarm.level." + str(alarm_level)) 
        str_alarm = alarm_info_str.format(alarm_level_str, alarm_id, alarm_name,
                                          formatted_date, alarm_detail, alarm_level)
        alarm_info_str_list.append(str_alarm)
    return alarm_info_str_list    


def get_all_alarm_count(rest, base_uri):
    """
    查询未恢复的告警数量
    :param env:
    :return:
    """
    major_alarm_count = get_alarm_count(rest, base_uri, MAJOR_ALARM)
    critical_alarm_count = get_alarm_count(rest, base_uri, CRITICAL_ALARM)
    warning_alarm_count = get_alarm_count(rest, base_uri, WARNING_ALARM)
    alarm_type_and_count = dict()
    alarm_type_and_count[WARNING_ALARM] = warning_alarm_count
    alarm_type_and_count[MAJOR_ALARM] = major_alarm_count
    alarm_type_and_count[CRITICAL_ALARM] = critical_alarm_count
    
    return alarm_type_and_count


def get_all_alarm_info(rest, base_uri, ret_list, all_alarm_type, alarm_type_and_count):
    for alarm_type in all_alarm_type:
        if alarm_type_and_count.get(alarm_type, 0) != 0:
            alarm_info = get_alarm_info(rest, base_uri, alarm_type, ret_list)
            if not alarm_info:
                return False
            ret_list.extend(alarm_info)
    return True


def execute(rest):
    """
    检查系统是否存在未恢复告警
    :param env:
    :return:
    """
    ret_list = []
    dev_node = py_java_env.get("devInfo")
    observer = py_java_env.get("progressObserver")
    progress_map = {}
    all_alarm_type = [WARNING_ALARM, MAJOR_ALARM, CRITICAL_ALARM]
    try:
        progress_map[ITEM_ID] = 1
        observer.updateProgress(progress_map)
        base_uri = RestUtil.getDstorageUrlHead(dev_node)
        device_esn = dev_node.getDeviceSerialNumber()
        alarm_type_and_count = get_all_alarm_count(rest, base_uri)

        for alarm_type in all_alarm_type:
            if alarm_type_and_count.get(alarm_type, -1) == -1:
                return common.INSPECT_UNNORMAL, "\n".join(ret_list), \
                        common.get_err_msg(LANG, "query.result.abnormal")                

        all_alarm_count = sum(list(alarm_type_and_count.values()))
        if all_alarm_count == 0:
            return common.INSPECT_PASS, "\n".join(ret_list), ""               

        if not get_all_alarm_info(rest, base_uri, ret_list, all_alarm_type, alarm_type_and_count):
            return common.INSPECT_UNNORMAL, "\n".join(ret_list), \
                    common.get_err_msg(LANG, "query.result.abnormal") 

        return common.INSPECT_UNNORMAL, "\n".join(ret_list), common.get_err_msg(LANG, "check.major.alarm")
    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
    return (
        common.INSPECT_UNNORMAL,
        "\n".join(ret_list),
        common.get_err_msg(LANG, "query.result.abnormal"),
    )
