# coding: UTF-8

import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil
from com.huawei.ism.exception import IsmException

import common
from ds_rest_util import CommonRestService

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "replication_cluster_address_pool"


def execute(rest):
    """
    检查复制集群IP地址池
    :param env:
    :return:
    """
    LOGGER.logInfo("Start to check replication cluster address pool")
    ret_list = []
    dev_node = py_java_env.get("devInfo")
    observer = py_java_env.get("progressObserver")
    progress_map = {}
    try:
        progress_map[ITEM_ID] = 1
        observer.updateProgress(progress_map)
        base_uri = RestUtil.getDstorageUrlHead(dev_node)

        # 获取复制集群ID
        pools_json, res_tuple = get_cluster_json(base_uri, rest)
        ret_list.extend(res_tuple)
        # 没有复制集群，直接返回
        if not pools_json.get("serviceCmdData"):
            return common.INSPECT_PASS, "\n".join(ret_list), ""
        boolean, cluster_id_set = get_cluster_id_info(pools_json)
        if not boolean:
            return common.INSPECT_UNNORMAL, "\n".join(ret_list), \
                   common.get_err_msg(LANG, "query.result.abnormal")

        # 根据复制集群ID获取复制节点的复制IP
        ip_pool_list = list()
        for ctl_cluster_id in cluster_id_set:
            # 获取复制集群ID对于的地址池信息
            pools_json, res_tuple = get_ip_pool_json(
                base_uri, ctl_cluster_id, rest)
            ret_list.extend(res_tuple)
            # 处理地址池信息，提取每一个地址池的复制IP信息
            boolean, ip_pool_info = get_ip_pool_info(pools_json)
            if not boolean:
                return common.INSPECT_UNNORMAL, "\n".join(ret_list), \
                       common.get_err_msg(LANG, "query.result.abnormal")
            ip_pool_list.extend(ip_pool_info)
        progress_map[ITEM_ID] = 80
        observer.updateProgress(progress_map)

        # 判断每个节点复制IP数量是否至少有两个
        tmp_err_dict = check_replication_ip_num(ip_pool_list)
        all_ret = "\n".join(ret_list)
        if tmp_err_dict:
            msg_list = ["\n".join(err_msg_list) for _, err_msg_list
                        in tmp_err_dict.items()]
            return common.INSPECT_UNNORMAL, all_ret, "".join(msg_list)

        return common.INSPECT_PASS, ret_list, ""
    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        return (
            common.INSPECT_UNNORMAL,
            "\n".join(ret_list),
            common.get_err_msg(LANG, "query.result.abnormal"),
        )


def get_ip_pool_info(pools_json):
    """
    处理复制集群地址池信息，获取地址池的复制IP和地址池名称
    :param pools_json: dict
    :return: list
    """
    ip_pool_list = list()
    # serviceCmdData为空时，当前复制集群无地址池，检查下一个集群
    if not pools_json.get("serviceCmdData"):
        return True, ''
    # 获取复制IP地址池复制IP信息
    for cmd_data in pools_json.get("serviceCmdData"):
        ret_value = cmd_data.get('result')
        if 0 != int(ret_value):
            return False, list()
        ip_pool_list.append(
            (cmd_data.get('name'), cmd_data.get('ipPoolIp'))
        )
    return True, ip_pool_list


def get_cluster_id_info(pools_json):
    """
    获取复制集群ID
    :param pools_json: dict
    :return: set
    """
    ret_value = pools_json.get('result')
    if 0 != int(ret_value):
        return False, ''
    cluster_id_set = {str(cmd_data.get('controlClusterId')) for cmd_data
                      in pools_json.get("serviceCmdData")}
    return True, cluster_id_set


def check_replication_ip_num(ip_pool_list):
    """
    判断每个复制节点复制IP数量是否至少有两个
    :param ip_pool_list: list
    :return: dict
    """
    tmp_err_dict = {}
    for ip_pool_name, each_ip_pool_ip in ip_pool_list:
        ip_info_list = each_ip_pool_ip.split(",")
        for ip_infos in ip_info_list:
            ip_info = ip_infos.split(":")[1]
            if len(ip_info.split("@")) < 2:
                tmp_list = tmp_err_dict.get(ip_info, [])
                tmp_list.append(
                    common.get_err_msg(
                        LANG, "query.replication.cluster.address.pool."
                              "result.abnormal", ip_pool_name))
                tmp_err_dict[ip_pool_name] = tmp_list
    return tmp_err_dict


def get_cluster_json(base_uri, rest):
    """
    批量查询复制集群信息
    :param base_uri: str
    :param rest: class
    :return: tuple
    """
    cmd_str = "{}/dsware/service/serviceCmd".format(base_uri)
    client_data = {
        "op": "drCmd",
        "serviceType": "dr",
        "subOp": "queryControlCluster",
    }
    pools_json = CommonRestService.execute_post_request(rest, cmd_str,
                                                        client_data)
    ret_tuple = (cmd_str, str(pools_json))
    return pools_json, ret_tuple


def get_ip_pool_json(base_uri, ctl_cluster_id, rest):
    """
    批量查询复制集群地址池信息
    :param base_uri: str
    :param ctl_cluster_id: str
    :param rest: class
    :return: tuple
    """
    cmd_str = "{}/dsware/service/serviceCmd".format(base_uri)
    client_data = {
        "controlClusterId": ctl_cluster_id,
        "op": "drCmd",
        "subOp": "BatchQueryIpPool",
        "serviceType": "dr",
        "type": "IP_POOL",
        "id": '0',
        "count": '100'}
    pools_json = CommonRestService.execute_post_request(
        rest, cmd_str, client_data)
    ret_tuple = (cmd_str, str(pools_json))
    return pools_json, ret_tuple
