# coding: UTF-8

import datetime
import os
from com.huawei.ism.exception import IsmException
from java.lang import Exception as JException

import com.huawei.ism.tool.inspect.utils.FileAssistant as FileAssistant
import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil
import common
from ds_rest_util import CommonRestService
from utils import Products

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "service_date"
HALF_YEAR = 180
DATE_FORMAT_STR = "%Y-%m-%d"
NOT_CONFIGURED = "0"
NORMAL = "1"
ABOUT_TO_EXPIRE = "2"
EXPIRED = "3"
MISSING_LIFECYCLE_INFORMATION = "4"


def execute(rest_conn):
    """
    检查服务时间是否过期
    :param rest_conn:集群rest连接
    :return:
    """

    ret_list = []
    dev_node = py_java_env.get("devInfo")
    base_uri = RestUtil.getDstorageUrlHead(dev_node)
    if check_storage_software(dev_node) >= 0:
        cmd_str = "{}/api/v2/cluster/lifecycle".format(base_uri)
    else:
        cmd_str = "{}/api/v2/cluster/servers".format(base_uri)
    try:
        # 获取server接口返回值
        servers_json = CommonRestService.exec_get_gor_big_by_ds(rest_conn, cmd_str)
        # 判断返回值是否为0
        result_code = servers_json.get("result").get("code")
        if result_code != 0:
            ret_list.append(str(servers_json.get("result")))
            return common.INSPECT_UNNORMAL, "\n".join(ret_list), common.get_err_msg(LANG, "service.date.abnormal")

        cluster_nodes = servers_json.get("data")
        # 如果获取的信息为空
        if not cluster_nodes:
            return common.INSPECT_PASS, ""

        export_e_guarantee(rest_conn, base_uri)

        if check_storage_software(dev_node) >= 0:
            return check_new(cluster_nodes)
        else:
            return check_old(cluster_nodes, get_system_time(base_uri, dev_node, rest_conn))
    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        return common.INSPECT_UNNORMAL, "\n".join(ret_list), common.get_err_msg(LANG, "query.result.abnormal")


def check_new(cluster_nodes):
    """
    统计所有节点的维保时间情况，并将维保信息按检查标准分组
    :param cluster_nodes: 当前检查的节点
    :return: 是否检查通过,结果列表,错误消息
    """
    inspect_result = common.INSPECT_PASS
    err_msg = ""
    normal_list = []
    not_configured_list = []
    expired_list = []
    about_expire_list = []
    tittle = "Management Ip\tService Date\tService Duration(month)\tStatus\tStatus Message"
    for cluster_node in cluster_nodes:
        info = NodeServiceInfo(cluster_node)
        if info.status == NOT_CONFIGURED:
            not_configured_list.append(info.build_ret_str_info_new("not configured"))
            continue
        elif info.status == MISSING_LIFECYCLE_INFORMATION:
            not_configured_list.append(info.build_ret_str_info_new("missing lifecycle information"))
            continue
        elif info.status == NORMAL:
            normal_list.append(info.build_ret_str_info_new("normal"))
            continue
        elif info.status == ABOUT_TO_EXPIRE:
            about_expire_list.append(info.build_ret_str_info_new("about to expire"))
        else:
            expired_list.append(info.build_ret_str_info_new("expired"))
        inspect_result = common.INSPECT_UNNORMAL
        err_msg = common.get_err_msg(LANG, "service.date.overdue")
    ret_list = expired_list + about_expire_list + normal_list + not_configured_list
    ret_list.insert(0, tittle)
    return inspect_result, "\n".join(ret_list), err_msg


def check_old(cluster_nodes, device_date):
    """
    统计所有节点的维保时间情况，并将维保信息按检查标准分组
    :param cluster_nodes: 当前检查的节点
    :param device_date 集群当前的时间
    :return: 是否检查通过,结果列表,错误消息
    """
    is_pass = common.INSPECT_PASS
    normal_list = []
    not_configured_list = []
    expired_list = []
    about_expire_list = []
    tittle = "ID\tManagement Ip\tService Date\tService Duration(year)\tRemaining Service Date(day)\tStatus"
    for cluster_node in cluster_nodes:
        info = NodeServiceInfo(cluster_node)
        if not info.need_check:
            not_configured_list.append(info.build_ret_str_info_old("--", "not configured"))
            continue

        life_span_days = calc_server_life_span_days(device_date, info)
        if life_span_days >= HALF_YEAR:
            normal_list.append(info.build_ret_str_info_old(life_span_days, "normal"))
            continue
        elif life_span_days < 0:
            expired_list.append(info.build_ret_str_info_old(life_span_days, "expired"))
        else:
            about_expire_list.append(info.build_ret_str_info_old(life_span_days, "about to expire"))
        is_pass = common.INSPECT_UNNORMAL
    err_msg = ""
    if is_pass == common.INSPECT_UNNORMAL:
        err_msg = common.get_err_msg(LANG, "service.date.overdue")
    ret_list = expired_list + about_expire_list + normal_list + not_configured_list
    ret_list.insert(0, tittle)
    return is_pass, "\n".join(ret_list), err_msg


def calc_server_life_span_days(device_date, info):
    service_utc_date = datetime.datetime.fromtimestamp(info.service_date)
    duration_date = add_year(service_utc_date, info.service_duration)
    return (duration_date - device_date).days


def add_year(service_date, service_duration):
    """
    获取过期时间
    :param service_date:服务时间
    :param service_duration: 服务年限
    :return:
    """
    try:
        return service_date.replace(year=service_date.year + service_duration)
    except ValueError:
        return service_date + (
                datetime.date(service_date.year + service_duration, 3, 1) -
                datetime.date(service_date.year, 3, 1))


def get_system_time(base_uri, dev_node, rest):
    # 获取系统时间
    esn = dev_node.getDeviceSerialNumber()
    cmd_str = ("{}/deviceManager/rest/{}/system_utc_time".format(base_uri, esn))
    device_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    device_time = int(device_json.get("data").get("CMO_SYS_UTC_TIME"))
    device_date = datetime.datetime.fromtimestamp(device_time)
    return device_date


def check_storage_software(dev_node):
    """
    :function: 检查存储软件版本，与8.1.3或8.1.RC6比较
    :param version: 节点信息
    :return: 之前：-1，相同：0，之后：1
    """
    product_version = dev_node.getProductVersion()
    # DataTurbo设备视为OceanStor Pacific 8.1.5处理
    if "dataturbo" in dev_node.getProductModel().lower():
        return 1
    if "RC" in product_version:
        return Products.compareVersion(product_version, "8.1.RC6")
    else:
        return Products.compareVersion(product_version, "8.1.3")


def export_e_guarantee(rest_conn, base_uri):
    """
    导出电子保单信息，812版本开始有此接口，脚本兼容尝试导出
    :param rest_conn: 集群浮动IP
    :param base_uri: 请求的URI
    :return:
    """
    try:
        url = "{}/api/v2/cluster/e_guarantee".format(base_uri)
        response = rest_conn.execGetFile(url, None, py_java_env.get("temp_path"))
        local_path = response.getContentMap().get("filePath")
        if not os.path.exists(local_path):
            return
        FileAssistant.deCopmressAllZipFile(py_java_env.get("result_path"), local_path)
    except JException:
        LOGGER.logWarning("Export e guarantee failed.")


class NodeServiceInfo:
    def __init__(self, data):
        self.ip = data.get('management_ip')
        self.starting_point_date = data.get("starting_point_date")
        self.session = data.get("session")
        self.status = str(data.get("status"))
        self.node_id = data.get("id")
        self.service_date = data.get("service_date")
        self.service_duration = data.get("service_duration")
        self.need_check = (isinstance(self.service_date, int)) and (isinstance(self.service_duration, int))
        if isinstance(self.service_date, int):
            self.service_time_show = datetime.datetime.fromtimestamp(self.service_date).strftime(DATE_FORMAT_STR)
        else:
            self.service_time_show = str(self.service_date).ljust(13, ' ')

    def build_ret_str_info_new(self, status_message):
        return '{}\t{}\t{}\t{}\t{}'.format(self.ip, self.starting_point_date, self.session.ljust(22, ' '),
                                           self.status, status_message.ljust(13, ' '))

    def build_ret_str_info_old(self, lifespan_days, status):
        return '{}\t{}\t{}\t{}\t{}\t{}'.format(self.node_id, self.ip, self.service_time_show,
                                               str(self.service_duration).ljust(22, ' '),
                                               str(lifespan_days).ljust(27, ' '), str(status).ljust(13, ' '))

